/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableIntBag;
import com.gs.collections.api.bag.primitive.MutableIntBag;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.block.procedure.primitive.IntIntProcedure;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedIntCollection;
import com.gs.collections.impl.factory.primitive.IntBags;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableIntBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link IntIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 *
 * @see MutableIntBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedIntBag
        extends AbstractSynchronizedIntCollection
        implements MutableIntBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedIntBag(MutableIntBag bag)
    {
        super(bag);
    }

    SynchronizedIntBag(MutableIntBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableIntBag getMutableIntBag()
    {
        return (MutableIntBag) this.getIntCollection();
    }

    @Override
    public SynchronizedIntBag with(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntBag without(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntBag withAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedIntBag withoutAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(int item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(int item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().sizeDistinct();
        }
    }

    public int occurrencesOf(int item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(IntIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableIntBag select(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().select(predicate);
        }
    }

    @Override
    public MutableIntBag reject(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(IntToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntBag().hashCode();
        }
    }

    @Override
    public LazyIntIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyIntIterableAdapter(this);
        }
    }

    @Override
    public MutableIntBag asUnmodifiable()
    {
        return new UnmodifiableIntBag(this);
    }

    @Override
    public MutableIntBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableIntBag toImmutable()
    {
        return IntBags.immutable.withAll(this);
    }
}
