/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.ShortIterable;
import com.gs.collections.api.LazyShortIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableShortBag;
import com.gs.collections.api.bag.primitive.MutableShortBag;
import com.gs.collections.api.block.function.primitive.ShortToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.ShortPredicate;
import com.gs.collections.api.block.procedure.primitive.ShortIntProcedure;
import com.gs.collections.api.iterator.ShortIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedShortCollection;
import com.gs.collections.impl.factory.primitive.ShortBags;
import com.gs.collections.impl.lazy.primitive.LazyShortIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableShortBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link ShortIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 *
 * @see MutableShortBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedShortBag
        extends AbstractSynchronizedShortCollection
        implements MutableShortBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedShortBag(MutableShortBag bag)
    {
        super(bag);
    }

    SynchronizedShortBag(MutableShortBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableShortBag getMutableShortBag()
    {
        return (MutableShortBag) this.getShortCollection();
    }

    @Override
    public SynchronizedShortBag with(short element)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedShortBag without(short element)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedShortBag withAll(ShortIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedShortBag withoutAll(ShortIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(short item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(short item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().sizeDistinct();
        }
    }

    public int occurrencesOf(short item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(ShortIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableShortBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableShortBag select(ShortPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().select(predicate);
        }
    }

    @Override
    public MutableShortBag reject(ShortPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(ShortToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableShortBag().hashCode();
        }
    }

    @Override
    public LazyShortIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyShortIterableAdapter(this);
        }
    }

    @Override
    public MutableShortBag asUnmodifiable()
    {
        return new UnmodifiableShortBag(this);
    }

    @Override
    public MutableShortBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableShortBag toImmutable()
    {
        return ShortBags.immutable.withAll(this);
    }
}
