/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.iterator.CharIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableCharSet;
import com.gs.collections.api.set.primitive.CharSet;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedCharCollection;
import com.gs.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableCharSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link CharIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 *
 * @see MutableCharSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedCharSet
        extends AbstractSynchronizedCharCollection
        implements MutableCharSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedCharSet(MutableCharSet set)
    {
        super(set);
    }

    SynchronizedCharSet(MutableCharSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableCharSet and wrap it directly in a SynchronizedCharSet.
     */
    public static SynchronizedCharSet of(MutableCharSet set)
    {
        return new SynchronizedCharSet(set);
    }

    /**
     * This method will take a MutableCharSet and wrap it directly in a SynchronizedCharSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedCharSet of(MutableCharSet set, Object lock)
    {
        return new SynchronizedCharSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableCharSet getMutableCharSet()
    {
        return (MutableCharSet) this.getCharCollection();
    }

    @Override
    public SynchronizedCharSet without(char element)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedCharSet with(char element)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedCharSet withAll(CharIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedCharSet withoutAll(CharIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableCharSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableCharSet select(CharPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().select(predicate);
        }
    }

    @Override
    public MutableCharSet reject(CharPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(CharToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().hashCode();
        }
    }

    @Override
    public LazyCharIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyCharIterableAdapter(this);
        }
    }

    @Override
    public MutableCharSet asUnmodifiable()
    {
        return new UnmodifiableCharSet(this);
    }

    @Override
    public MutableCharSet asSynchronized()
    {
        return this;
    }

    public CharSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().freeze();
        }
    }

    @Override
    public ImmutableCharSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableCharSet().toImmutable();
        }
    }
}
