/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.block.procedure.primitive.DoubleProcedure;
import com.gs.collections.api.collection.primitive.MutableDoubleCollection;
import com.gs.collections.api.iterator.DoubleIterator;

/**
 * The DoubleIteratorIterate class provides implementations of the various iteration patterns for use with the {@link DoubleIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class DoubleIteratorIterate
{
    private DoubleIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(DoubleIterator iterator, DoubleProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableDoubleCollection> R select(
            DoubleIterator iterator,
            DoublePredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            double item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableDoubleCollection> R reject(
            DoubleIterator iterator,
            DoublePredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            double item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            DoubleIterator iterator,
            DoubleToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            double item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static double detectIfNone(DoubleIterator iterator, DoublePredicate predicate, double ifNone)
    {
        while (iterator.hasNext())
        {
            double item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(DoubleIterator iterator, DoublePredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(DoubleIterator iterator, DoublePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(DoubleIterator iterator, DoublePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(DoubleIterator iterator, DoublePredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(DoubleIterator iterator, T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while (iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }


    public static double sum(DoubleIterator iterator)
    {
        double sum = 0.0;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static double max(DoubleIterator iterator)
    {
        double max = iterator.next();
        while (iterator.hasNext())
        {
            double next = iterator.next();
            if (Double.compare(max, next) < 0)
            {
                max = next;
            }
        }
        return max;
    }

    public static double min(DoubleIterator iterator)
    {
        double min = iterator.next();
        while (iterator.hasNext())
        {
            double next = iterator.next();
            if (Double.compare(next, min) < 0)
            {
                min = next;
            }
        }
        return min;
    }
}
