/*
 * Copyright 2011-2022 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.core.controller.inject.open

import java.util.concurrent.atomic.AtomicLong

import scala.concurrent.duration.Duration

import io.gatling.commons.util.Clock
import io.gatling.commons.util.Collections._
import io.gatling.core.controller.inject.{ InjectionProfile, Workload }
import io.gatling.core.scenario.Scenario
import io.gatling.core.stats.StatsEngine

import io.netty.channel.EventLoopGroup

/**
 * This class represents the configuration of a scenario
 *
 * @param steps
 *   the number of users that will behave as this scenario says
 */
private[core] final class OpenInjectionProfile(steps: List[OpenInjectionStep]) extends InjectionProfile {
  override def totalUserCount: Option[Long] = Some(steps.sumBy(_.users))

  override private[inject] def workload(
      scenario: Scenario,
      userIdGen: AtomicLong,
      startTime: Long,
      eventLoopGroup: EventLoopGroup,
      statsEngine: StatsEngine,
      clock: Clock
  ): Workload =
    new OpenWorkload(
      UserStream(steps),
      steps.foldLeft(Duration.Zero)((acc, step) => acc.plus(step.duration)),
      steps.forall(_.users == 0),
      scenario: Scenario,
      userIdGen,
      startTime,
      eventLoopGroup,
      statsEngine,
      clock
    )

  override def toString = s"OpenInjectionProfile($steps)"

  // [e]
  //
  //
  // [e]
}
