/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms.parameters;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.resourcehelpers.ApplicationUtils;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.FileParameterValue;
import csbase.logic.algorithms.parameters.FileParameter;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;

/**
 * A Viso para {@link FileParameter Parmetro do Tipo Arquivo}.
 * 
 * @author lmoreira
 */
public abstract class FileParameterView extends
  SimpleParameterView<FileParameterValue> {

  /**
   * Cria a viso em modo {@link ParameterView.Mode#CONFIGURATION}.
   * 
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro (No aceita {@code null}).
   * 
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  protected FileParameterView(Window window, FileParameter parameter) {
    this(parameter, Mode.CONFIGURATION);
  }

  /**
   * Cria a viso.
   * 
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  protected FileParameterView(FileParameter parameter, Mode mode) {
    super(parameter, mode);

    updateViewContents();
    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileParameter getParameter() {
    return (FileParameter) super.getParameter();
  }

  /**
   * Solicita um arquivo da usurio.
   * 
   * @return O arquivo ou {@code null} se o usurio cancelar ou houver um erro.
   */
  protected abstract FileParameterValue askForFile();

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new FileConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new FileReportParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setEnabled(boolean isEnabled) {
    getComponent().setEnabled(isEnabled);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateVisibilyView() {
    boolean isVisible =
      getParameter().isVisible() && !getParameter().hasLink()
        && getParameter().usesPipe() != FileParameterPipeAcceptance.ALWAYS;
    if (isVisible() != isVisible) {
      setVisible(isVisible);
      fireVisibilityWasChanged();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateViewContents() {
    ((IFileParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IFileParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#REPORT}.
   */
  private final class FileReportParameter extends JPanel implements
    IFileParameterComponent {

    /**
     * O campo de texto.
     */
    private JTextField textField;

    /**
     * Construtor.
     */
    FileReportParameter() {
      setLayout(new GridBagLayout());

      textField = new JTextField();
      textField.setToolTipText(getParameter().getDescription());
      ComponentProperties.setProperties(textField, Mode.REPORT, true);
      textField.setEditable(false);

      GridBagConstraints constraints = new GridBagConstraints();
      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.gridheight = 1;
      constraints.gridwidth = 1;
      constraints.gridx = 1;
      constraints.gridy = 1;
      constraints.insets = new Insets(2, 2, 2, 2);
      constraints.weightx = 1.0;
      constraints.weighty = 0.0;
      add(textField, constraints);

      JButton button = createShowButton(textField);
      if (null != button) {
        constraints.fill = GridBagConstraints.NONE;
        constraints.gridx++;
        constraints.weightx = 0.0;
        add(button, constraints);
      }

      updateViewContents();
    }

    /**
     * Cria o boto de mostrar arquivo de log.
     * 
     * @param field o campo com o nome do arquivo.
     * @return o boto.
     */
    private JButton createShowButton(final JTextField field) {
      final FileParameterValue value = getParameter().getValue();
      if (null != value && value.getType().equals("LOG")) {
        JButton browseButton =
          GUIUtils.createImageButton(ApplicationImages.ICON_VIEWLOG_16);
        browseButton.addActionListener(new ActionListener() {
          @Override
          public void actionPerformed(ActionEvent ev) {
            try {
              ClientProjectFile file = getClientProjectFile(value);
              ApplicationUtils.runPreferredApp(getWindow(), file);
            }
            catch (ApplicationException e) {
              final DesktopFrame desktop = DesktopFrame.getInstance();
              final DesktopComponentFrame frame = desktop.getDesktopFrame();
              StandardErrorDialogs.showErrorDialog(frame, e);
            }
          }
        });
        browseButton.setToolTipText(getParameter().getDescription());
        return browseButton;
      }
      else {
        return null;
      }
    }

    /**
     * Obtm o arquivo de projeto a partir do valor de um parmetro do tipo
     * arquivo.
     * 
     * @param paramValue o valor do parmetro.
     * @return o arquivo de projeto.
     */
    private ClientProjectFile getClientProjectFile(
      final FileParameterValue paramValue) {
      CommonClientProject project = DesktopFrame.getInstance().getProject();
      String[] path = paramValue.getPathAsArray();
      final ClientProjectFile file =
        GetFileTask.runTask(getWindow(), project, path);
      return file;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      FileParameterValue value = getParameter().getValue();
      textField.setText(null == value ? " " : value.getPath());
    }
  }

  /**
   * Viso do parmetro em modo {@link ParameterView.Mode#CONFIGURATION}.
   */
  private final class FileConfigurationParameter extends JPanel implements
    IFileParameterComponent {

    /**
     * O campo de texto.
     */
    private JTextField textField;

    /**
     * O boto Navegar.
     */
    private JButton browseButton;

    /**
     * Construtor.
     */
    FileConfigurationParameter() {
      setLayout(new GridBagLayout());

      browseButton =
        GUIUtils.createImageButton(ApplicationImages.ICON_BROWSEFILE_16);
      browseButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent ev) {
          FileParameterValue file = askForFile();
          if (file != null) {
            getParameter().setValue(file);
          }
        }
      });
      browseButton.setToolTipText(getParameter().getDescription());
      textField = new JTextField();
      textField.addFocusListener(new FocusListener() {
        @Override
        public void focusGained(FocusEvent e) {
          // Ignora o evento.
        }

        @Override
        public void focusLost(FocusEvent e) {
          updateModel();
        }
      });
      textField.setToolTipText(getParameter().getDescription());

      GridBagConstraints constraints = new GridBagConstraints();
      constraints.anchor = GridBagConstraints.NORTHWEST;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.gridheight = 1;
      constraints.gridwidth = 1;
      constraints.gridx = 1;
      constraints.gridy = 1;
      constraints.insets = new Insets(2, 2, 2, 2);
      constraints.weightx = 1.0;
      constraints.weighty = 0.0;
      add(textField, constraints);

      constraints.fill = GridBagConstraints.NONE;
      constraints.gridx++;
      constraints.weightx = 0.0;
      add(browseButton, constraints);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      FileParameterValue file = getParameter().getValue();
      if (file == null) {
        textField.setText("");
      }
      else {
        textField.setText(file.toString());
      }
    }

    /**
     * Habilita/Desabilita a viso.
     * 
     * @param isEnabled .
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      super.setEnabled(isEnabled);
      textField.setEnabled(isEnabled);
      browseButton.setEnabled(isEnabled);
    }

    /**
     * Atualiza o contedo do modelo (L as informaes da viso e preenche o
     * parmetro).
     */
    private void updateModel() {
      String text = textField.getText();
      if (text.length() == 0) {
        getParameter().setValue(null);
      }
      else {
        if (text.equals("/")) {
          text = ".";
        }
        else if (text.startsWith("/")) {
          text = text.substring(1);
        }
        String[] path = FileUtils.splitPath(text, "/");
        CommonClientProject project = DesktopFrame.getInstance().getProject();
        ClientProjectFile rootProjectFile = project.getRoot();
        ClientProjectFile clientProjectFile = rootProjectFile;
        String type = getParameter().getFileType();
        for (String component : path) {
          if (clientProjectFile == null) {
            break;
          }
          clientProjectFile =
            GetChildFromNameTask.runTask(clientProjectFile, component);
        }
        if (clientProjectFile != null) {
          if (clientProjectFile.isDirectory()) {
            type = ProjectFileType.DIRECTORY_TYPE;
          }
          else {
            type = clientProjectFile.getType();
          }
        }
        else {
          if (getParameter().getMode() == FileParameterMode.DIRECTORY) {
            type = ProjectFileType.DIRECTORY_TYPE;
          }
        }
        FileParameterValue file = new FileParameterValue(text, type);
        getParameter().setValue(file);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return false;
  }
}
