package csbase.client.applications.flowapplication.multiflow.tree;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import javax.swing.tree.DefaultTreeSelectionModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * Modelo de seleo das checkboxes da rvore. Automaticamente muda a seleo de
 * acordo com a seleo dos ns-filhos e do n-pai.
 */
public class ParameterSelectionModel extends DefaultTreeSelectionModel {

  /** O modelo da rvore. */
  private TreeModel model;

  /**
   * Construtor.
   * 
   * @param model O modelo da rvore.
   */
  public ParameterSelectionModel(TreeModel model) {
    this.model = model;
    setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addSelectionPaths(TreePath[] paths) {
    // Remove a seleo dos descendentes selecionados dos caminhos (paths)
    for (int i = 0; i < paths.length; i++) {
      TreePath path = paths[i];
      TreePath[] selectionPaths = getSelectionPaths();
      if (selectionPaths == null) {
        break;
      }
      List<TreePath> toBeRemoved = new ArrayList<TreePath>();
      for (int j = 0; j < selectionPaths.length; j++) {
        TreePath selectedPath = selectionPaths[j];
        if (path.isDescendant(selectedPath)) {
          toBeRemoved.add(selectedPath);
        }
      }
      super.removeSelectionPaths(toBeRemoved.toArray(new TreePath[toBeRemoved
        .size()]));
    }

    /*
     * Se os "irmos" j esto selecionados, remove a seleo deles e os
     * seleciona recursivamente atravs do pai. Seno seleciona s o caminho
     * mesmo.
     */
    for (int i = 0; i < paths.length; i++) {
      TreePath path = paths[i];
      TreePath temp = null;
      while (areSiblingsSelected(path)) {
        temp = path;
        if (path.getParentPath() == null) {
          break;
        }
        path = path.getParentPath();
      }
      if (temp != null) {
        if (temp.getParentPath() != null) {
          addSelectionPath(temp.getParentPath());
        }
        else {
          TreePath[] selectionPaths = getSelectionPaths();
          if (selectionPaths != null) {
            removeSelectionPaths(selectionPaths);
          }
          super.addSelectionPaths(new TreePath[] { temp });
        }
      }
      else {
        super.addSelectionPaths(new TreePath[] { path });
      }
    }
  }

  /**
   * Determina se um caminho est selecionado na rvore.
   * 
   * @param path o caminho a ser avaliado.
   * @param dig determina se o caminho  considerado selecionado se um de seus
   *        ancestrais estiver selecionado.
   * @return verdadeiro se o caminho est selecionado ou falso, caso contrrio.
   */
  public boolean isPathSelected(TreePath path, boolean dig) {
    if (!dig) {
      return super.isPathSelected(path);
    }
    TreePath selectedPath = path;
    while (selectedPath != null && !super.isPathSelected(selectedPath)) {
      selectedPath = selectedPath.getParentPath();
    }
    return selectedPath != null;
  }

  /**
   * Determina se os ns "irmos" de um caminho esto selecionados.
   * 
   * @param path o caminho.
   * @return verdadeiro se os "irmos" esto selecionados.
   */
  private boolean areSiblingsSelected(TreePath path) {
    TreePath parent = path.getParentPath();
    if (parent == null) {
      return true;
    }
    Object node = path.getLastPathComponent();
    Object parentNode = parent.getLastPathComponent();
    int childCount = model.getChildCount(parentNode);
    for (int i = 0; i < childCount; i++) {
      Object childNode = model.getChild(parentNode, i);
      if (childNode != node) {
        if (!isPathSelected(parent.pathByAddingChild(childNode))) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeSelectionPaths(TreePath[] paths) {
    for (int i = 0; i < paths.length; i++) {
      TreePath path = paths[i];
      if (path.getPathCount() == 1) {
        super.removeSelectionPaths(new TreePath[] { path });
      }
      else {
        toggleRemoveSelection(path);
      }
    }
  }

  /**
   * Trata a remoo da seleo de um caminho. Se houver algum ancestral
   * selecionado, remove sua seleo e seleciona todos os demais descendentes
   * (com exceo do caminho a ser removido e seus descendentes). Se no houver
   * ancestrais selecionados, s remove a seleo do caminho.
   * 
   * @param path o caminho removido.
   */
  private void toggleRemoveSelection(TreePath path) {
    Stack<TreePath> stack = new Stack<TreePath>();
    TreePath parent = path.getParentPath();
    while (parent != null && !isPathSelected(parent)) {
      stack.push(parent);
      parent = parent.getParentPath();
    }
    if (parent != null) {
      stack.push(parent);
    }
    else {
      super.removeSelectionPaths(new TreePath[] { path });
      return;
    }

    while (!stack.isEmpty()) {
      TreePath temp = stack.pop();
      TreePath peekPath;
      if (stack.isEmpty()) {
        peekPath = path;
      }
      else {
        peekPath = stack.peek();
      }
      Object node = temp.getLastPathComponent();
      Object peekNode = peekPath.getLastPathComponent();
      int childCount = model.getChildCount(node);
      for (int i = 0; i < childCount; i++) {
        Object childNode = model.getChild(node, i);
        if (childNode != peekNode) {
          super.addSelectionPaths(new TreePath[] { temp
            .pathByAddingChild(childNode) });
        }
      }
    }
    super.removeSelectionPaths(new TreePath[] { parent });
  }
}
