package csbase.client.applications.projectsmanager.usersearch;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.text.Document;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.project.dialogs.UsersPermissionsTable;
import csbase.logic.User;
import csbase.logic.UserOutline;

/**
 * Painel para visualizao / seleo de usurios.
 * 
 * @author Tecgraf
 */
public class UserPanel extends JPanel {

  /**
   * Texto usado como filtro para a tabela.
   */
  private JTextField filterText;
  /**
   * Boto para ativar a filtragem da tabela.
   */
  private JButton filterButton;
  /**
   * Tabela de usurios compartilhados.
   */
  protected UsersPermissionsTable sharedUsersTable;
  /**
   * Usurios com acesso RO ao projeto.
   */
  protected Set<Object> usersRO = new HashSet<Object>();
  /**
   * Usurios com acesso RW ao projeto.
   */
  protected Set<Object> usersRW = new HashSet<Object>();

  /**
   * Boolean que indica que o esquema de permisses deve ser exibido.
   */
  final protected boolean showPermissions;

  /**
   * Usurios selecionados na tabela do dilogo.
   */
  protected Set<Object> selectedUsers;

  /**
   * Checkbox que define se apenas usurios com acesso ao projeto devem aparecer
   * na lista.
   */
  final private JCheckBox usersOnlyBox = new JCheckBox();

  /**
   * Checkbox que define se apenas usurios sem acesso ao projeto devem aparecer
   * na lista.
   */
  final private JCheckBox nonUsersOnlyBox = new JCheckBox();

  /**
   * Indica se os usurios so todos membros ou no-membros dos projetos, ou se
   * isto  irrelevante para a funcionalidade em questo.
   */
  final protected boolean fixedAccess;

  /**
   * Construtor.
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   */
  public UserPanel(final List<UserOutline> allUsers,
    final boolean showPermissions) {
    this.showPermissions = showPermissions;
    this.fixedAccess = true;

    filterUsers(allUsers);
    sharedUsersTable = new UsersPermissionsTable(allUsers, usersRO, usersRW);
    initPanel();
  }

  /**
   * Construtor.
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   */
  public UserPanel(final List<UserOutline> allUsers, final Set<Object> usersRO,
    final Set<Object> usersRW, final boolean showPermissions) {
    this.showPermissions = showPermissions;
    this.fixedAccess = false;

    filterUsers(allUsers);
    sharedUsersTable = new UsersPermissionsTable(allUsers, usersRO, usersRW);
    initComponents();
    initPanel();
  }

  /**
   * Construtor.
   * 
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO Usurios com permisso de leitura.
   * @param usersRW Usurios com permisso de leitura e escrita.
   * @param showPermissions Boolean que indica que o esquema de permisses deve
   *        ser exibido.
   * @param fixedAccess Indica se os usurios so todos membros ou no-membros
   *        dos projetos.
   */
  public UserPanel(final List<UserOutline> allUsers, final Set<Object> usersRO,
    final Set<Object> usersRW, final boolean showPermissions,
    final boolean fixedAccess) {
    this.showPermissions = showPermissions;
    this.fixedAccess = fixedAccess;

    filterUsers(allUsers);
    sharedUsersTable = new UsersPermissionsTable(allUsers, usersRO, usersRW);
    initComponents();
    initPanel();
  }

  /**
   * Inicializa os componentes.
   */
  private void initComponents() {
    usersOnlyBox.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        nonUsersOnlyBox.setSelected(false);
        filterTableContentByAccess();
      }
    });
    usersOnlyBox.setText(LNG.get("UserPanel.users.only.checkbox.title"));

    nonUsersOnlyBox.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        usersOnlyBox.setSelected(false);
        filterTableContentByAccess();
      }
    });
    nonUsersOnlyBox.setText(LNG.get("UserPanel.non.users.only.checkbox.title"));
  }

  /**
   * Inicia os componentes.
   */
  private void initPanel() {
    final ListSelectionModel listSelectionModel =
      sharedUsersTable.getSelectionModel();
    listSelectionModel.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        final int[] selectedRows = sharedUsersTable.getSelectedRows();
        selectedUsers = new HashSet<Object>();
        for (int row : selectedRows) {
          final Object id = sharedUsersTable.getValueAt(row, 0);
          selectedUsers.add(id);
        }
      }
    });

    createGUI();
  }

  /**
   * Retorna os usurios selecionados.
   * 
   * @return Usurios selecionados.
   */
  public Set<Object> getSelectedUsers() {
    return selectedUsers;
  }

  /**
   * Removemos da lista de todos os usurios o dono do projeto e o
   * administrador.
   * 
   * @param allUsers lista de todos os usurios do sistema
   */
  private void filterUsers(final List<UserOutline> allUsers) {
    for (UserOutline userOutline : allUsers) {
      Object id = userOutline.getId();
      if (id.equals(User.getAdminId())) {
        allUsers.remove(userOutline);
        break;
      }
    }
  }

  /**
   * Cria a interface.
   */
  private void createGUI() {
    if (!showPermissions) {
      final TableColumnModel columnModel = sharedUsersTable.getColumnModel();
      final TableColumn column4 = columnModel.getColumn(3);
      final TableColumn column3 = columnModel.getColumn(2);
      columnModel.removeColumn(column4);
      columnModel.removeColumn(column3);
    }

    // interceptamos tentativas de fechar a janela
    setLayout(new GridBagLayout());

    // adiciona verticalmente 1 ou mais painis para entrada de dados
    addInputPanels(0);
  }

  /**
   * Adiciona um ou mais painis para entrada de dados, verticalmente, de cima
   * para baixo, a partir de uma "linha" especfica de um
   * <code>GridBagLayout</code>.
   * 
   * FIXME - Parece no haver mais motivo para retornar o ndice vertical do
   * prximo componente a ser includo no painel. Neste caso, o mtodo
   * retornaria void e a varivel y perderia o sentido.
   * 
   * @param ypos - linha inicial do layout a partir da qual devem ser inseridos
   *        os painis
   * @return y + (nmero de painis adicionados), para ser usado como referncia
   *         para insero de novos painis
   */
  private int addInputPanels(int ypos) {
    int y = ypos;
    final GBC gbc = new GBC(0, y++).both();
    add(createSharedUsersPanel(), gbc);
    return y;
  }

  /**
   * Cria o painel para configurao pontual do acesso de cada usurio no modo
   * de compartilhamento seletivo .
   * 
   * @return painel para configurao pontual do acesso de cada usurio
   */
  protected JPanel createSharedUsersPanel() {
    final JPanel panel = new JPanel(new GridBagLayout());
    final JPanel leftFilterPanel = new JPanel(new GridBagLayout());
    final JPanel rightFilterPanel = new JPanel(new GridBagLayout());

    final JScrollPane scrollPane = new JScrollPane(sharedUsersTable);
    GBC gbc = new GBC(0, 0).both().width(3);
    panel.add(scrollPane, gbc);

    gbc = new GBC(0, 0).west().insets(10, 0, 10, 0);
    final JLabel filterLabel = new JLabel(LNG.get("UserPanel.filter.label"));
    leftFilterPanel.add(filterLabel, gbc);

    filterText = new JTextField();
    gbc = new GBC(1, 0).insets(10).horizontal().filly();
    leftFilterPanel.add(filterText, gbc);

    filterButton = new JButton(LNG.get("UserPanel.filter.clear"));
    gbc = new GBC(2, 0).east().insets(10, 0, 10, 10);
    leftFilterPanel.add(filterButton, gbc);

    gbc = new GBC(0, 0).west().top(10);
    rightFilterPanel.add(nonUsersOnlyBox, gbc);

    gbc = new GBC(0, 1).west().top(2);
    rightFilterPanel.add(usersOnlyBox, gbc);

    gbc = new GBC(0, 1).horizontal().northwest().width(2).top(5);
    panel.add(leftFilterPanel, gbc);

    if (!fixedAccess) {
      gbc = new GBC(2, 1).northwest().width(1).insets(5, 15, 0, 0);
      panel.add(rightFilterPanel, gbc);
    }

    setupFilterControls();
    return panel;
  }

  /**
   * Configura os controles (textfield + boto) para filtragem da tabela com os
   * usurios.
   */
  private void setupFilterControls() {
    // Implementamos um DocumentListener para ativar o filtro quando o contedo
    // do campo for alterado de qualquer forma
    final Document document = filterText.getDocument();
    document.addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        filterTableContentByAccess();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        filterTableContentByAccess();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        filterTableContentByAccess();
      }
    });

    // Queremos que o contedo do filtro seja todo selecionado quando o campo
    // ganhar o foco
    filterText.addFocusListener(new FocusListener() {
      @Override
      public void focusGained(FocusEvent e) {
        filterText.selectAll();
      }

      @Override
      public void focusLost(FocusEvent e) {
        filterText.select(0, 0);
      }
    });

    // ao do boto "limpar"
    filterButton.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterText.setText("");
        filterTableContentByAccess();
      }
    });
  }

  /**
   * Filtra o contedo da tabela a partir do contedo do campo "filtro". O texto
   * do campo  usado como <code>".*texto.*"</code>.
   */
  private void filterTableContent() {
    final String text = filterText.getText();
    if (text.length() > 0) {
      sharedUsersTable.setRowFilter(RowFilter.regexFilter(".*" + text + ".*",
        0, 1));
    }
    else {
      sharedUsersTable.setRowFilter(null);
    }
  }

  /**
   * Filtra o contedo da tabela em funo da checkbox que exclui da lista todos
   * os usurios que no tem acesso ao projeto.
   */
  private void filterTableContentByAccess() {

    final boolean usersOnly = usersOnlyBox.isSelected();
    final boolean nonUsersOnly = nonUsersOnlyBox.isSelected();
    if (!usersOnly && !nonUsersOnly) {
      filterTableContent();
      return;
    }

    final RowFilter<Object, Object> rf1;
    if (usersOnly) {
      rf1 = RowFilter.regexFilter("true", 2, 3);
    }
    else {
      final RowFilter<Object, Object> filter1 =
        RowFilter.regexFilter("false", 2);
      final RowFilter<Object, Object> filter2 =
        RowFilter.regexFilter("false", 3);
      List<RowFilter<Object, Object>> filters =
        new ArrayList<RowFilter<Object, Object>>(2);
      filters.add(filter1);
      filters.add(filter2);
      rf1 = RowFilter.andFilter(filters);
    }

    String text = filterText.getText();
    if (text.length() > 0) {
      final RowFilter<Object, Object> rf2 =
        RowFilter.regexFilter(".*" + text + ".*", 0, 1);
      final List<RowFilter<Object, Object>> filters =
        new ArrayList<RowFilter<Object, Object>>(2);
      filters.add(rf1);
      filters.add(rf2);
      RowFilter<Object, Object> filter = RowFilter.andFilter(filters);
      sharedUsersTable.setRowFilter(filter);
    }
    else {
      sharedUsersTable.setRowFilter(rf1);
    }
  }

  /**
   * Habilita / Desabilita tabela do painel.
   * 
   * @param enabled - estado do painel
   */
  public void enableSharedUsersPanel(final boolean enabled) {
    sharedUsersTable.setEnabled(enabled);
    filterText.setEnabled(enabled);
    filterButton.setEnabled(enabled);
    usersOnlyBox.setEnabled(enabled);
    nonUsersOnlyBox.setEnabled(enabled);
  }

  /**
   * Define o modo de seleo da tabela.
   * 
   * @param multipleSelection Indice se a tabela deve permitir seleo mltipla
   *        de linhas.
   */
  public void setTableSelection(final boolean multipleSelection) {

    final ListSelectionModel selectionModel =
      sharedUsersTable.getSelectionModel();
    if (multipleSelection) {
      selectionModel
        .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    }
    else {
      selectionModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    }
  }

  /**
   * Delegate method para fillUserSets de UsersPermissionsTable.
   */
  public void fillUserSets() {
    sharedUsersTable.fillUserSets(usersRO, usersRW);
  }

  /**
   * Obtm o conjunto de usurios com acesso RO ao projeto.
   * 
   * @return conjunto de usurios com acesso RO ao projeto
   */
  public Set<Object> getUsersRO() {
    return usersRO;
  }

  /**
   * Obtm o conjunto de usurios com acesso RW ao projeto.
   * 
   * @return conjunto de usurios com acesso RW ao projeto
   */
  public Set<Object> getUsersRW() {
    return usersRW;
  }

  /**
   * Verifica se houve mudanas nas permisses de usurios.
   * 
   * @return True, caso tenham ocorrido mudanas nas permisses de usurios.
   *         False, caso contrrio.
   */
  public boolean hasUsersPermissionsChanged() {
    return sharedUsersTable.hasUsersPermissionsChanged();
  }
}
