package csbase.client.applications.statsviewer;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.statsviewer.LogsView.LogsDataType;
import csbase.logic.eventlogservice.LogsInfo;

/**
 * Painel que filtra os resultados encontrados na pesquisa de logs, conforme
 * seleo do usurio para algoritmos ou aplicaes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FilterPanel extends JPanel {

  /**
   * Referncia para aplicao.
   */
  private final StatsViewer appViewer;

  /**
   * Painel principal com cardLayout conforme tipo de estatstica selecionada.
   */
  private JPanel cardsPanel;

  /**
   * Lista com todas as aplicaes associadas a busca.
   */
  private JList applicationsList;

  /**
   * Lista com todos os algoritmos associados a busca.
   */
  private JList algorithmsList;

  /**
   * Construtor.
   * 
   * @param appViewer referncia para aplicao
   */
  public FilterPanel(StatsViewer appViewer) {
    super(new GridBagLayout());
    this.appViewer = appViewer;
    buildCardPanel();
  }

  /**
   * Cria o painel principal.
   */
  private void buildCardPanel() {
    cardsPanel = new JPanel(new CardLayout());
    cardsPanel.add(buildApplicationsPanel(), LogsDataType.ApplicationsData
      .name());
    cardsPanel.add(buildAlgorihtmsPanel(), LogsDataType.ExecutionsData.name());
    cardsPanel.add(new JPanel(), LogsDataType.LoginData.name());
    add(cardsPanel, new GBC(0, 0).center().insets(10, 10, 5, 10).both());
  }

  /**
   * @return painel para filtrar as aplicaes.
   */
  private JPanel buildApplicationsPanel() {

    applicationsList = new JList(new DefaultListModel());
    applicationsList
      .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    applicationsList.setLayoutOrientation(JList.VERTICAL);
    applicationsList.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        if (e.getValueIsAdjusting()) {
          return;
        }
        List<String> selecteds = getSelected(applicationsList);
        appViewer.filterApplications(selecteds);
      }
    });
    JScrollPane scrollApplications =
      new JScrollPane(applicationsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    JPanel panel = new JPanel(new BorderLayout());
    panel.add(new JLabel(appViewer.getString("filterPanel.applications")),
      BorderLayout.NORTH);
    panel.add(scrollApplications, BorderLayout.CENTER);
    return panel;
  }

  /**
   * @return painel para filtrar os algoritmos.
   */
  private JPanel buildAlgorihtmsPanel() {

    algorithmsList = new JList(new DefaultListModel());
    algorithmsList
      .setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    algorithmsList.setLayoutOrientation(JList.VERTICAL);
    algorithmsList.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        if (e.getValueIsAdjusting()) {
          return;
        }
        List<String> selecteds = getSelected(algorithmsList);
        appViewer.filterAlgorithms(selecteds);
      }
    });
    JScrollPane scrollAlgorithms =
      new JScrollPane(algorithmsList, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
        JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

    JPanel panel = new JPanel(new BorderLayout());
    panel.add(new JLabel(appViewer.getString("filterPanel.algorithms")),
      BorderLayout.NORTH);
    panel.add(scrollAlgorithms, BorderLayout.CENTER);
    return panel;
  }

  /**
   * Retorna uma de lista strings com os valores selecionados na lista.
   * 
   * @param list lista que se deseja consultar
   * 
   * @return lista de valores selecionados.
   */
  private List<String> getSelected(JList list) {
    int[] selectedIndices = list.getSelectedIndices();
    if (selectedIndices == null) {
      return null;
    }
    DefaultListModel model = (DefaultListModel) list.getModel();
    List<String> strList = new ArrayList<String>();
    for (int index : selectedIndices) {
      String value = (String) model.get(index);
      strList.add(value);
    }
    return strList;
  }

  /**
   * Atualiza o painel conforme o tipo de estatstica de interesse selecionado.
   * 
   * @param selectedType tipo de estatstica selecionada
   */
  public void setLogType(LogsDataType selectedType) {
    CardLayout cl = (CardLayout) (cardsPanel.getLayout());
    cl.show(cardsPanel, selectedType.name());
  }

  /**
   * Atualiza os modelos das listas de algoritmos e aplicaes com as
   * informaes conforme os dados dos logs.
   * 
   * @param info novos dados de logs obtidos
   */
  public void updateInfo(LogsInfo info) {

    // atualizando Jlist de aplicaes
    DefaultListModel appModel = (DefaultListModel) applicationsList.getModel();
    appModel.clear();
    for (String app : info.getApplications()) {
      appModel.addElement(app);
    }
    applicationsList.setSelectionInterval(0, appModel.getSize() - 1);

    // atualizando Jlist de algoritmos
    DefaultListModel algoModel = (DefaultListModel) algorithmsList.getModel();
    algoModel.clear();
    for (String algo : info.getAlgorithms()) {
      algoModel.addElement(algo);
    }
    algorithmsList.setSelectionInterval(0, algoModel.getSize() - 1);
  }

}
