/*
 * Detalhes da ltima alterao:
 * 
 * $Author: karla $ $Date: 2010-04-12 18:33:43 -0300 (Mon, 12 Apr 2010) $
 * $Revision: 104156 $
 */
package csbase.client.externalresources;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.jnlp.BasicService;
import javax.jnlp.FileContents;
import javax.jnlp.FileOpenService;
import javax.jnlp.FileSaveService;
import javax.jnlp.ServiceManager;
import javax.jnlp.UnavailableServiceException;

import csbase.exception.CSBaseException;

/**
 * Representa o ponto de acesso  recursos externos ao cliente CSBase. Esses
 * recursos so arquivos no sistema de arquivos local e exibio de URL's.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ExternalResources {
  /** Mensagem de erro para excees de servio indisponvel. */
  private static final String UNAVAILABLE_SERVICE_MESSAGE =
    "O servio {0} no est disponvel: {1}";

  /**
   * O nome da classe que representa o servio bsico. Este servio 
   * responsvel por exibir URL's.
   */
  private static final String BASIC_SERVICE_CLASS_NAME =
    "javax.jnlp.BasicService";

  /**
   * O nome da classe que representa o servio de abertura de arquivos locais.
   */
  private static final String FILE_OPEN_SERVICE_CLASS_NAME =
    "javax.jnlp.FileOpenService";

  /**
   * O nome da classe que representa o servio de salvamento de arquivos locais.
   */
  private static final String FILE_SAVE_SERVICE_CLASS_NAME =
    "javax.jnlp.FileSaveService";

  /** A instncia nica. */
  private static ExternalResources instance;

  /**
   * Construtor privado para evitar a criao de objetos.
   */
  private ExternalResources() {
    //Sem implementao
  }

  /**
   * Obtm a instncia nica.
   * 
   * @return A instncia nica.
   */
  public static ExternalResources getInstance() {
    if (instance == null) {
      instance = new ExternalResources();
    }
    return instance;
  }

  /**
   * Verifica se os recursos externos esto habilitados.
   * 
   * @return true, caso os recursos externos estejam habilitados, ou false, caso
   *         contrrio.
   */
  public boolean isEnabled() {
    try {
      ServiceManager.lookup(BASIC_SERVICE_CLASS_NAME);
      return true;
    }
    catch (UnavailableServiceException e) {
      return false;
    }
  }

  /**
   * Exibe uma URL no browser do cliente.
   * 
   * @param url A URL a ser exibida.
   * 
   * @return true, se conseguiu exibir a URL, ou false, caso contrrio.
   * 
   * @throws CSBaseException Caso ocorra algum erro durante o procedimento.
   */
  public boolean showDocument(URL url) throws CSBaseException {
    try {
      BasicService basicService =
        (BasicService) ServiceManager.lookup(BASIC_SERVICE_CLASS_NAME);
      return basicService.showDocument(url);
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          BASIC_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Obtm a URL que representa o codebase, definido no arquivo jnlp.
   * 
   * @return O codebase para a aplicao.
   * 
   * @throws CSBaseException Caso ocorra algum erro durante o procedimento.
   */
  public URL getCodeBase() throws CSBaseException {
    try {
      BasicService basicService =
        (BasicService) ServiceManager.lookup(BASIC_SERVICE_CLASS_NAME);
      return basicService.getCodeBase();
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          BASIC_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Abre um dilogo para que o usurio escolha um arquivo que ser importado
   * para o CSBase.
   * 
   * @param pathHint Uma dica de onde ser o diretrio inicial apresentado pelo
   *        dilogo.
   * @param extensions Um array com as extenses mostradas pelo dilogo.
   * 
   * @return O arquivo escolhido pelo usurio, ou null, caso o usurio cancele o
   *         dilogo.
   * 
   * @throws IOException Caso ocorra algum problema ao abrir o arquivo.
   * @throws CSBaseException Caso o servio esteja indisponvel.
   */
  public LocalFile openFileDialog(String pathHint, String[] extensions)
    throws IOException, CSBaseException {
    try {
      FileOpenService fileOpenService =
        (FileOpenService) ServiceManager.lookup(FILE_OPEN_SERVICE_CLASS_NAME);
      FileContents fileContents =
        fileOpenService.openFileDialog(pathHint, extensions);
      if (fileContents == null) {
        return null;
      }
      return new JNLPLocalFile(fileContents);
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          FILE_OPEN_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Abre um dilogo para que o usurio escolha alguns arquivos que sero
   * importados para o CSBase.
   * 
   * @param pathHint Uma dica de onde ser o diretrio inicial apresentado pelo
   *        dilogo.
   * @param extensions Um array com as extenses mostradas pelo dilogo.
   * 
   * @return Os arquivos escolhidos pelo usurio, ou null, caso o usurio
   *         cancele o dilogo.
   * 
   * @throws IOException Caso ocorra algum problema ao abrir o arquivo.
   * @throws CSBaseException Caso o servio esteja indisponvel.
   */
  public LocalFile[] openMultiFileDialog(String pathHint, String[] extensions)
    throws IOException, CSBaseException {
    try {
      FileOpenService fileOpenService =
        (FileOpenService) ServiceManager.lookup(FILE_OPEN_SERVICE_CLASS_NAME);
      FileContents[] fileContents =
        fileOpenService.openMultiFileDialog(pathHint, extensions);
      if (fileContents == null) {
        return null;
      }
      LocalFile[] localFiles = new LocalFile[fileContents.length];
      for (int i = 0; i < fileContents.length; i++) {
        localFiles[i] = new JNLPLocalFile(fileContents[i]);
      }
      return localFiles;
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          FILE_OPEN_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Abre um dilogo para que o usurio escolha um arquivo que ser salvo a
   * partir de um arquivo do CSBase.
   * 
   * @param pathHint Uma dica de onde ser o diretrio inicial apresentado pelo
   *        dilogo.
   * @param extensions Um array com as extenses mostradas pelo dilogo.
   * @param inputStream O fluxo de entrada com os dados que sero escritos no
   *        arquivo escolhido pelo usurio.
   * @param name Uma sugesto de nome para o arquivo que o usurio estar
   *        salvando.
   * 
   * @return O arquivo escolhido pelo usurio, ou null, caso o usurio cancele o
   *         dilogo.
   * 
   * @throws CSBaseException Caso o servio esteja indisponvel.
   * @throws IOException Caso ocorra algum problema ao abrir o arquivo.
   */
  public LocalFile saveFileDialog(String pathHint, String[] extensions,
    InputStream inputStream, String name) throws CSBaseException, IOException {
    try {
      FileSaveService fileSaveService =
        (FileSaveService) ServiceManager.lookup(FILE_SAVE_SERVICE_CLASS_NAME);
      FileContents fileContents =
        fileSaveService.saveFileDialog(pathHint, extensions, inputStream, name);
      if (fileContents == null) {
        return null;
      }
      return new JNLPLocalFile(fileContents);
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          FILE_SAVE_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Abre um dilogo para que o usurio escolha um arquivo que ser salvo a
   * partir de um arquivo do CSBase.
   * 
   * @param pathHint Uma dica de onde ser o diretrio inicial apresentado pelo
   *        dilogo.
   * @param extensions Um array com as extenses mostradas pelo dilogo.
   * @param localFile Um arquivo que foi salvo anteriormente utilizando o mtodo
   *        {@link #saveFileDialog(String, String[], InputStream, String)};
   * 
   * @return O arquivo escolhido pelo usurio, ou null, caso o usurio cancele o
   *         dilogo.
   * 
   * @throws CSBaseException Caso o servio esteja indisponvel.
   * @throws IOException Caso ocorra algum problema ao abrir o arquivo.
   */
  public LocalFile saveAsFileDialog(String pathHint, String[] extensions,
    JNLPLocalFile localFile) throws CSBaseException, IOException {
    try {
      FileSaveService fileSaveService =
        (FileSaveService) ServiceManager.lookup(FILE_SAVE_SERVICE_CLASS_NAME);
      FileContents fileContents =
        fileSaveService.saveAsFileDialog(pathHint, extensions, localFile
          .getFileContents());
      if (fileContents == null) {
        return null;
      }
      return new JNLPLocalFile(fileContents);
    }
    catch (UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          FILE_SAVE_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

}
