package csbase.client.util.table;

import java.util.HashMap;
import java.util.Map;

/**
 * Coluna: a coluna da {@link Table Tabela}.
 * 
 * @param <R> O tipo de valor armazenado na linha da tabela.
 */
public final class TableColumn<R> {

  /**
   * A fbrica de modelo de clula.
   */
  private CellModelFactory<R> cellModelFactory;

  /**
   * Os modelos de clulas criadas indexadas pelo ndice da linha.
   */
  private Map<Integer, CellModel> cellModelsByRowIndex;

  /**
   * As vises de clula criadas indexadas pelo ndice da linha.
   */
  private Map<Integer, CellView> cellViewsByRowIndex;

  /**
   * A fbrica de vises de clulas.
   */
  private CellViewFactory cellViewFactory;

  /**
   * O ttulo da coluna.
   */
  private String title;

  /**
   * Indica se as sero editveis ou no.
   */
  private boolean isEditable;

  /**
   * Cria uma coluna.
   * 
   * @param title O ttulo (No aceita {@code null}).
   * @param isEditable Indica se a coluna  editvel ou no.
   * @param cellModelFactory A fbrica de modelos de clulas (No aceita
   *        {@code null}).
   * @param cellViewFactory A fbrica de vises de clulas (No aceita
   *        {@code null}).
   */
  public TableColumn(String title, boolean isEditable,
    CellModelFactory<R> cellModelFactory, CellViewFactory cellViewFactory) {
    cellModelsByRowIndex = new HashMap<Integer, CellModel>();
    cellViewsByRowIndex = new HashMap<Integer, CellView>();
    this.isEditable = isEditable;
    setCellModelFactory(cellModelFactory);
    setCellViewFactory(cellViewFactory);
    setTitle(title);
  }

  /**
   * Cria uma coluna.
   * 
   * @param title O ttulo (No aceita {@code null}).
   * @param isEditable Indica se a coluna  editvel ou no.
   * @param rowClass A classe dos objetos armanzenados na linha (No aceita
   *        {@code null}).
   * @param cellModelClass A classe de modelos de clulas (No aceita
   *        {@code null}). A classe precisa ter pelo menos um construtor
   *        pblico que receba um parmetro que  do tipo {@code rowClass} ou de
   *        uma de suas superclasses.
   * @param cellViewFactory A fbrica de vises de clulas (No aceita
   *        {@code null}).
   */
  public TableColumn(String title, boolean isEditable, Class<R> rowClass,
    Class<? extends CellModel> cellModelClass, CellViewFactory cellViewFactory) {
    this(title, isEditable, new ReflectionCellModelFactory<R>(rowClass,
      cellModelClass), cellViewFactory);
  }

  /**
   * Cria uma coluna.
   * 
   * @param title O ttulo (No aceita {@code null}).
   * @param isEditable Indica se a coluna  editvel ou no.
   * @param cellModelFactory A fbrica de modelos de clulas (No aceita
   *        {@code null}).
   * @param cellViewClass A classe de vises de clulas (No aceita {@code null}).
   *        A classe precisa ter pelo menos um construtor pblico que no receba
   *        parmetros.
   */
  public TableColumn(String title, boolean isEditable,
    CellModelFactory<R> cellModelFactory,
    Class<? extends CellView> cellViewClass) {
    this(title, isEditable, cellModelFactory, new ReflectionCellViewFactory(
      cellViewClass));
  }

  /**
   * Cria uma coluna.
   * 
   * @param title O ttulo (No aceita {@code null}).
   * @param isEditable Indica se a coluna  editvel ou no.
   * @param rowClass A classe dos objetos armanzenados na linha (No aceita
   *        {@code null}).
   * @param cellModelClass A classe de modelos de clulas (No aceita
   *        {@code null}). A classe precisa ter pelo menos um construtor
   *        pblico que receba um parmetro que  do tipo {@code rowClass} ou de
   *        uma de suas superclasses.
   * @param cellViewClass A classe de vises de clulas (No aceita {@code null}).
   *        A classe precisa ter pelo menos um construtor pblico que no receba
   *        parmetros.
   */
  public TableColumn(String title, boolean isEditable, Class<R> rowClass,
    Class<? extends CellModel> cellModelClass,
    Class<? extends CellView> cellViewClass) {
    this(title, isEditable, new ReflectionCellModelFactory<R>(rowClass,
      cellModelClass), new ReflectionCellViewFactory(cellViewClass));
  }

  /**
   * Cria a clula de uma linha da tabela.
   * 
   * @param rowIndex O ndice da linha da tabela.
   * @param rowValue O objeto armazenado na linha da tabela.
   */
  public void createCell(int rowIndex, R rowValue) {
    CellModel cellModel = createCellModel(rowValue);
    final CellView cellView = createCellView();
    cellModel.addCellModelListener(new CellModelListener() {
      public void valueWasChanged(CellModel cellModel) {
        cellView.setValue(cellModel.getValue());
      }
    });
    cellModelsByRowIndex.put(rowIndex, cellModel);
    cellViewsByRowIndex.put(rowIndex, cellView);
  }

  /**
   * Obtm um modelo de clula.
   * 
   * @param rowIndex O ndice da linha (No pode ser negativo e precisa ser
   *        menor ou igual ao nmero de linhas).
   * 
   * @return O modelo de clula.
   */
  public CellModel getCellModel(int rowIndex) {
    if ((rowIndex < 0) || (cellModelsByRowIndex.size() <= rowIndex)) {
      throw new IllegalArgumentException(String.format(
        "ndice de linha fora da faixa esperada.\n"
          + "ndice: %d.\nMnimo: 0.\nMximo: %d.", rowIndex,
        cellViewsByRowIndex.size() - 1));
    }
    return cellModelsByRowIndex.get(rowIndex);
  }

  /**
   * Obtm uma viso de clula.
   * 
   * @param rowIndex O ndice da linha (No pode ser negativo e precisa ser
   *        menor ou igual ao nmero de linhas).
   * 
   * @return A viso de clula.
   */
  public CellView getCellView(int rowIndex) {
    if ((rowIndex < 0) || (cellViewsByRowIndex.size() <= rowIndex)) {
      throw new IllegalArgumentException(String.format(
        "ndice de linha fora da faixa esperada.\n"
          + "ndice: %d.\nMnimo: 0.\nMximo: %d.", rowIndex,
        cellViewsByRowIndex.size() - 1));
    }
    return cellViewsByRowIndex.get(rowIndex);
  }

  /**
   * Obtm o ttulo.
   * 
   * @return O ttulo.
   */
  public String getTitle() {
    return title;
  }

  /**
   * Indica se uma coluna  editvel ou no.
   * 
   * @return .
   */
  public boolean isEditable() {
    return isEditable;
  }

  /**
   * Remove a clula de uma linha.
   * 
   * @param rowIndex O ndica da linha
   */
  public void removeCell(int rowIndex) {
    if ((rowIndex < 0) || (cellModelsByRowIndex.size() <= rowIndex)) {
      throw new IllegalArgumentException(String.format(
        "ndice de linha fora da faixa esperada.\n"
          + "ndice: %d.\nMnimo: 0.\nMximo: %d.", rowIndex,
        cellViewsByRowIndex.size() - 1));
    }
    cellModelsByRowIndex.remove(rowIndex);
    cellViewsByRowIndex.remove(rowIndex);
    int rowCount = cellViewsByRowIndex.size();
    for (int currentRowIndex = rowIndex + 1; 
         currentRowIndex <= rowCount; 
         currentRowIndex++) {
      CellModel cellModel = cellModelsByRowIndex.get(currentRowIndex);
      cellModelsByRowIndex.put(currentRowIndex - 1, cellModel);
      CellView cellView = cellViewsByRowIndex.get(currentRowIndex);
      cellViewsByRowIndex.put(currentRowIndex - 1, cellView);
    }
  }

  @Override
  public String toString() {
    return title;
  }

  /**
   * Cria um modelo de clula.
   * 
   * @param rowValue O valor da linha.
   * 
   * @return O modelo de clula criada.
   */
  private CellModel createCellModel(R rowValue) {
    return cellModelFactory.create(rowValue);
  }

  /**
   * Cria uma viso de clula.
   * 
   * @return A viso de clula criada.
   */
  private CellView createCellView() {
    return cellViewFactory.create();
  }

  /**
   * Atribui uma fbrica de modelos de clulas a esta coluna.
   * 
   * @param cellModelFactory A fbrica de modelos de clulas (No aceita
   *        {@code null}).
   */
  private void setCellModelFactory(CellModelFactory<R> cellModelFactory) {
    if (cellModelFactory == null) {
      throw new IllegalArgumentException(
        "O parmetro cellModelFactory est nulo.");
    }
    this.cellModelFactory = cellModelFactory;
  }

  /**
   * Atribui uma fbrica de vises de clulas a esta coluna.
   * 
   * @param cellViewFactory A fbrica de vises de clulas (No aceita
   *        {@code null}).
   */
  private void setCellViewFactory(CellViewFactory cellViewFactory) {
    if (cellViewFactory == null) {
      throw new IllegalArgumentException(
        "O parmetro cellViewFactory est nulo.");
    }
    this.cellViewFactory = cellViewFactory;
  }

  /**
   * Atribui um ttulo.
   * 
   * @param title O ttulo (No aceita {@code null}).
   */
  private void setTitle(String title) {
    if (title == null) {
      throw new IllegalArgumentException("O parmetro title est nulo.");
    }
    this.title = title;
  }
}
