package csbase.client.algorithms.parameters;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;

import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationImages;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.client.util.ClientUtilities;
import csbase.logic.SGASet;
import csbase.logic.algorithms.parameters.URLListParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.URLValue;

/**
 * A Viso para {@link URLListParameter Parmetro do tipo lista de URLs}.
 *
 * @author Tecgraf
 */
public abstract class URLListParameterView extends
  SimpleParameterView<List<URLValue>> {

  /**
   * Prefixo para internacionalizao de idioma.
   */
  protected static final String LNGPREFIX = URLListParameterView.class
    .getSimpleName()
    + ".";

  /**
   * Cria a viso.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so:
   *        {@link csbase.client.algorithms.parameters.ParameterView.Mode#CONFIGURATION}
   *        ou
   *        {@link csbase.client.algorithms.parameters.ParameterView.Mode#REPORT}
   *        .
   */
  public URLListParameterView(URLListParameter parameter, Mode mode) {
    super(parameter, mode);

    updateViewContents();
    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public URLListParameter getParameter() {
    return (URLListParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new URLConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new FileReportParameter();
  }

  /**
   * Abre um navegador de arquivos para um determinado protocolo.
   *
   * @param protocol protocolo.
   * @param args argumentos extras.
   * @return URL do arquivo selecionado.
   */
  protected abstract List<URLValue> askForFile(URLProtocol protocol,
    Object... args);

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setEnabled(boolean isEnabled) {
    getComponent().setEnabled(isEnabled);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateVisibilyView() {
    setVisible(getParameter().isVisible());
    fireVisibilityWasChanged();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateViewContents() {
    ((IURLParameterComponent) getComponent()).updateViewContents();
  }

  /**
   * Interface comum s diferentes vises do parmetro.
   */
  private interface IURLParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();
  }

  /**
   * Viso do parmetro em modo
   * {@link csbase.client.algorithms.parameters.ParameterView.Mode#REPORT}.
   */
  private class FileReportParameter extends JPanel implements
    IURLParameterComponent {

    /**
     * rea de texto de exibio
     */
    private JTextArea textArea;

    /**
     * Construtor.
     */
    FileReportParameter() {
      setLayout(new GridBagLayout());

      textArea = new JTextArea();
      URLListParameter parameter = getParameter();
      textArea.setToolTipText(parameter.getDescription());
      ComponentProperties.setProperties(textArea, Mode.REPORT, false);

      JScrollPane scroll = new JScrollPane(textArea);
      scroll
        .setBorder(ComponentProperties.getInstance(Mode.REPORT).getBorder());
      updateViewContents();
      add(scroll, new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      List<URLValue> values = getParameter().getValue();
      if (values == null) {
        textArea.setRows(1);
      }
      else {
        textArea.setRows(Math.max(1, Math.min(values.size(), 6)));
        StringBuffer sb = new StringBuffer();
        for (URLValue value : values) {
          URLProtocol selectedProtocol = value.getProtocol();
          if (selectedProtocol == URLValue.DEFAULT_PROTOCOL) {
            sb.append(value.getPath());
          }
          else {
            sb.append(selectedProtocol.getDisplayName() + value.getPath());
          }
          sb.append('\n');
        }
        // remove o ltimo breakLine
        String text = sb.substring(0, sb.length() - 1);
        textArea.setText(text);
      }
    }
  }

  /**
   * Viso do parmetro em modo
   * {@link csbase.client.algorithms.parameters.ParameterView.Mode#CONFIGURATION}
   * .
   */
  private class URLConfigurationParameter extends JPanel implements
    IURLParameterComponent {

    /**
     * O Boto Adicionar.
     */
    final private JButton addButton = new JButton();

    /**
     * Indica se a viso est habilitada.
     */
    private boolean isEnabledView;

    /**
     * A lista de urls.
     */
    private JList list;

    /**
     * O modelo da lista.
     */
    private DefaultListModel model;

    /**
     * O Boto Remover.
     */
    final private JButton removeButton = new JButton();

    /**
     * O Boto up.
     */
    final private JButton upButton = new JButton();

    /**
     * O Boto down.
     */
    final private JButton dnButton = new JButton();

    /**
     * O campo de seleo de protocolos.
     */
    private JComboBox protocolCombo;

    /**
     * O campo de seleo de SGAs. (usado no caso do protocolo ser
     * {@link URLProtocol#SGA})
     */
    private JComboBox sgaCombo;

    /**
     * A label do campo de seleo de SGAs.
     */
    private JLabel sgaLabel;

    /**
     * A label do campo de seleo de protocolos.
     */
    private JLabel protocolLabel;

    /**
     * Construtor.
     */
    URLConfigurationParameter() {
      super(new BorderLayout());

      JPanel listPanel = createListPanel();
      updateSGAModel();

      add(listPanel, BorderLayout.CENTER);

      URLListParameter parameter = getParameter();
      protocolChanged(parameter.getDefaultProtocol());
    }

    /**
     * Cria o painel com a lista de URLs.
     *
     * @return o painel.
     */
    private JPanel createListPanel() {
      JPanel listPanel = new JPanel();
      model = new DefaultListModel();
      list = new JList(model);
      list.getSelectionModel().setSelectionMode(
        ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      list.setToolTipText(getParameter().getDescription());
      list.addKeyListener(new KeyAdapter() {
        @Override
        public void keyPressed(KeyEvent e) {
          int keyCode = e.getKeyCode();
          switch (keyCode) {
            case KeyEvent.VK_BACK_SPACE:
            case KeyEvent.VK_DELETE:
              removeFiles();
          }
        }
      });
      list.addListSelectionListener(new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          updateRemoveButton();
          updateUpDownButton();
        }
      });
      list.setToolTipText(getParameter().getDescription());

      URLListParameter parameter = getParameter();
      EnumSet<URLProtocol> allowedProtocols = parameter.getAllowedProtocols();
      protocolCombo = new JComboBox();
      for (URLProtocol protocol : allowedProtocols) {
        protocolCombo.addItem(protocol);
      }
      protocolCombo.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent e) {
          if (e.getStateChange() == ItemEvent.SELECTED) {
            URLProtocol item = (URLProtocol) e.getItem();
            protocolChanged(item);
          }
        }
      });
      if (allowedProtocols.size() == 1) {
        protocolCombo.setVisible(false);
      }

      sgaCombo = new JComboBox();
      sgaCombo.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent e) {
          if (e.getStateChange() == ItemEvent.SELECTED) {
            model.clear();
            updateModel();
          }
        }
      });

      String protocolLabelName = LNG.get(LNGPREFIX + "protocol.label");
      String sgaLabelName = LNG.get(LNGPREFIX + "sga.label");

      protocolLabel = new JLabel(protocolLabelName);
      sgaLabel = new JLabel(sgaLabelName);

      JPanel buttonPanel = createButtonPanel();
      JPanel urlListPanel = new JPanel(new BorderLayout());
      urlListPanel.add(new JScrollPane(list), BorderLayout.CENTER);
      urlListPanel.add(buttonPanel, BorderLayout.EAST);

      listPanel.setLayout(new GridBagLayout());
      listPanel.add(protocolLabel, new GBC(0, 0).west().none().insets(2));
      listPanel.add(protocolCombo, new GBC(1, 0).fillx().insets(2));
      listPanel.add(sgaLabel, new GBC(0, 1).west().none().insets(2));
      listPanel.add(sgaCombo, new GBC(1, 1).fillx().insets(2));
      listPanel.add(urlListPanel, new GBC(0, 2).gridwidth(2).fillx().insets(2,
        2, 4, 2));
      return listPanel;
    }

    /**
     * Cria o painel com os botes Adicionar e Remover.
     *
     * @return o painel.
     */
    private JPanel createButtonPanel() {
      final URLListParameter parameter = getParameter();

      addButton.setIcon(ApplicationImages.ICON_BROWSEFILE_16);
      addButton.setText(LNG.get(LNGPREFIX + "add.button.text"));
      addButton.setToolTipText(LNG.get(LNGPREFIX + "add.button.tooltip"));
      addButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          URLProtocol protocol = (URLProtocol) protocolCombo.getSelectedItem();
          List<URLValue> selectedUrls = askForFile(protocol, getSGASelected());
          if (selectedUrls != null) {
            for (URLValue urlValue : selectedUrls) {
              getParameter().addElement(urlValue);
            }
          }
        }
      });

      removeButton.setIcon(ApplicationImages.ICON_DELETE_16);
      removeButton.setText(LNG.get(LNGPREFIX + "del.button.text"));
      removeButton.setToolTipText(LNG.get(LNGPREFIX + "del.button.tooltip"));
      removeButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          removeFiles();
        }
      });
      removeButton.setEnabled(false);

      upButton.setIcon(ApplicationImages.ICON_UP_16);
      upButton.setText(LNG.get(LNGPREFIX + "up.button.text"));
      upButton.setToolTipText(LNG.get(LNGPREFIX + "up.button.tooltip"));
      upButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          moveUpFile();
        }
      });
      upButton.setEnabled(false);

      dnButton.setIcon(ApplicationImages.ICON_DOWN_16);
      dnButton.setText(LNG.get(LNGPREFIX + "down.button.text"));
      dnButton.setToolTipText(LNG.get(LNGPREFIX + "down.button.tooltip"));
      dnButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          moveDownFile();
        }
      });
      dnButton.setEnabled(false);

      ClientUtilities.adjustEqualSizes(addButton, removeButton, upButton,
        dnButton);

      final JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout(new GridBagLayout());
      buttonPanel.add(addButton, new GBC(0, 0).center().none().insets(2));
      buttonPanel.add(removeButton, new GBC(0, 1).center().none().insets(2));
      if (!parameter.mustSort()) {
        buttonPanel.add(new JLabel(), new GBC(0, 2).center().vertical().insets(
          2));
        buttonPanel.add(upButton, new GBC(0, 3).center().none().insets(2));
        buttonPanel.add(dnButton, new GBC(0, 4).center().none().insets(2));
      }
      return buttonPanel;
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void removeFiles() {
      int[] selIdxs = list.getSelectedIndices();
      URLValue[] files = new URLValue[selIdxs.length];
      int i = 0;
      for (int selIdx : selIdxs) {
        final ListModel listModel = list.getModel();
        final URLValue file = (URLValue) listModel.getElementAt(selIdx);
        files[i++] = file;
      }

      final URLListParameter parameter = getParameter();
      for (URLValue file : files) {
        parameter.removeElementObj(file);
      }
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void moveUpFile() {
      final int[] indexes = list.getSelectedIndices();
      final int n = indexes.length;
      if (n != 1) {
        return;
      }
      final int index = indexes[0];
      final URLListParameter parameter = getParameter();
      parameter.moveUp(index);
      list.setSelectedIndex(index - 1);
    }

    /**
     * Remove os arquivos selecionados.
     */
    private void moveDownFile() {
      final int[] indexes = list.getSelectedIndices();
      final int n = indexes.length;
      if (n != 1) {
        return;
      }
      final int index = indexes[0];
      final URLListParameter parameter = getParameter();
      parameter.moveDown(index);
      list.setSelectedIndex(index + 1);
    }

    /**
     * Atualiza o boto Remover.
     */
    private void updateRemoveButton() {
      removeButton.setEnabled(isEnabledView && !list.isSelectionEmpty());
    }

    /**
     * Atualiza os botes Up/Down.
     */
    private void updateUpDownButton() {
      final int[] selection = list.getSelectedIndices();
      final boolean flag =
        isEnabledView && !list.isSelectionEmpty() && (selection.length == 1);
      upButton.setEnabled(flag);
      dnButton.setEnabled(flag);
    }

    /**
     * Atualiza o modelo da combo de SGAs.
     */
    private void updateSGAModel() {
      if (sgaCombo != null && sgaCombo.isVisible()) {
        List<String> sgaList = getSGAList();
        Object[] sgas = sgaList.toArray();
        if (sgaList.isEmpty()) {
          sgaCombo.setEnabled(false);
          String noSGAMessage = LNG.get(LNGPREFIX + "no_sga");
          sgaCombo.setToolTipText(noSGAMessage);
        }
        else {
          sgaCombo.setEnabled(true);
          sgaCombo.setModel(new DefaultComboBoxModel(sgas));
        }
        addButton.setEnabled(sgaCombo.isEnabled());
      }
    }

    /**
     * Indica que um novo protocolo foi selecionado.
     *
     * @param protocol o novo protocolo.
     */
    private void protocolChanged(URLProtocol protocol) {
      model.clear();
      switch (protocol) {
        case LOCAL:
          sgaLabel.setVisible(false);
          sgaCombo.setVisible(false);
          addButton.setIcon(ApplicationImages.ICON_BROWSELOCALFILE_16);
          addButton.setVisible(true);
          addButton.setEnabled(true);
          list.setEnabled(true);
          break;
        case PROJECT:
          sgaLabel.setVisible(false);
          sgaCombo.setVisible(false);
          addButton.setIcon(ApplicationImages.ICON_BROWSEFILE_16);
          addButton.setVisible(true);
          addButton.setEnabled(true);
          list.setEnabled(true);
          break;
        case SGA:
          sgaLabel.setVisible(true);
          sgaCombo.setVisible(true);
          addButton.setIcon(ApplicationImages.ICON_BROWSESGAFILE_16);
          addButton.setVisible(true);
          addButton.setEnabled(sgaCombo.isEnabled());
          list.setEnabled(sgaCombo.isEnabled());
          break;
        default:
          sgaLabel.setVisible(false);
          sgaCombo.setVisible(false);
          addButton.setVisible(false);
          addButton.setEnabled(false);
          list.setEnabled(true);
      }
      updateSGAModel();
      updateModel();
    }

    /**
     * Atualiza o contedo do modelo (L as informaes da viso e preenche o
     * parmetro).
     */
    private void updateModel() {
      if (model.getSize() == 0) {
        getParameter().setValue(null);
      }
      else {
        List<URLValue> urls = new ArrayList<URLValue>();
        for (int i = 0; i < model.getSize(); i++) {
          URLValue url = (URLValue) model.getElementAt(i);
          urls.add(url);
        }
        getParameter().setValue(urls);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      model.removeAllElements();
      List<URLValue> values = getParameter().getValue();
      if (values != null) {
        URLValue url = values.iterator().next();
        if (url != null) {
          if (protocolCombo != null) {
            URLProtocol protocol = url.getProtocol();
            protocolCombo.setSelectedItem(protocol);
          }
          if (sgaCombo != null) {
            updateSGAModel();
            String host = url.getHost();
            if (host != null) {
              sgaCombo.setSelectedItem(host);
            }
          }
        }
        for (URLValue value : values) {
          model.addElement(value);
        }
      }
      updateModel();
    }

    /**
     * Habilita/Desabilita a viso.
     *
     * @param isEnabled .
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      isEnabledView = isEnabled;
      super.setEnabled(isEnabled);
      list.setEnabled(isEnabled);
      addButton.setEnabled(isEnabled);
      addButton.setEnabled(isEnabled);
      updateRemoveButton();
      updateUpDownButton();
      if (protocolCombo != null) {
        protocolCombo.setEnabled(isEnabled);
      }
      if (sgaCombo != null) {
        sgaCombo.setEnabled(isEnabled);
      }
    }

    /**
     * Retorna o nome do SGA selecionado na combo.
     *
     * @return nome do SGA.
     */
    private String getSGASelected() {
      return (String) sgaCombo.getSelectedItem();
    }

    /**
     * Lista com todos os nomes de SGA cadastrados no servidor.
     *
     * @return lista com todos os nomes de SGA cadastrados no servidor.
     */
    private List<String> getSGAList() {
      Collection<String> sgaNames = SGAProxy.getAllSGANames();
      if (sgaNames == null) {
        return new ArrayList<String>();
      }

      List<String> sgaList = new ArrayList<String>();
      for (String name : sgaNames) {
        SGASet sgaSet = SGAProxy.getSGASet(name);
        if (sgaSet.mayExecuteCommand()) {
          sgaList.add(name);
        }
      }

      return sgaList;
    }
  }
}
