package csbase.client.applications.flowapplication.multiflow;

import java.awt.Dialog.ModalityType;
import java.util.Collections;
import java.util.List;

import tecgraf.javautils.gui.StatusBar;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.applications.flowapplication.multiflow.tree.ParameterSelection;
import csbase.client.util.csvpanel.CSVFileUtils;
import csbase.exception.OperationFailureException;
import csbase.logic.ClientFile;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowNode;
import csbase.logic.algorithms.flows.NodeParameter;

/**
 * Assistente de configurao de execuo de fluxo mltiplo.
 * 
 * @author Tecgraf
 */
public class MultiFlowWizard {

  /**
   * Tamanho do cabealho do arquivo de configurao de fluxos mltiplos.
   */
  public static final int CONFIG_FILE_HEADER_SIZE = 3;

  /**
   * Aplicao dona do assistente.
   */
  private FlowApplication application;
  /**
   * ltimo arquivo selecionado.
   */
  private ClientFile selectedFile;

  /**
   * Construtor.
   * 
   * @param application aplicao dona do assistente.
   */
  public MultiFlowWizard(FlowApplication application) {
    this.application = application;
  }

  /**
   * Mostra a janela para seleo de parmetros a serem exportados para a
   * configurao do fluxo mltiplo.
   * 
   * @param flow o fluxo.
   * @return a lista de parmetros selecionados.
   */
  public List<ParameterSelection> chooseExportedParameters(Flow flow) {
    ParameterSelectionDialog dialog =
      new ParameterSelectionDialog(application, flow);
    dialog.setVisible(true);
    if (dialog.isConfirmed()) {
      return dialog.getParameterSelection();
    }
    else {
      return Collections.emptyList();
    }
  }

  /**
   * Mostra a janela para exportao da configurao para execuo de mltiplos
   * fluxos e retorna o arquivo de exonfigurao selecionado.
   * 
   * @param flow o fluxo a ser executado.
   * @return o arquivo de configurao selecioando.
   */
  public ClientFile exportMultipleFlowConfiguration(Flow flow) {
    if (flow.getName() == null) {
      flow.setName(application.getFlowName());
    }
    List<ParameterSelection> parameters = chooseExportedParameters(flow);
    if (parameters != null && !parameters.isEmpty()) {
      final ExportMultipleFlowConfigurationDialog dialog =
        new ExportMultipleFlowConfigurationDialog(application);
      dialog.setModalityType(ModalityType.APPLICATION_MODAL);
      dialog.setVisible(true);
      Integer numExecs = dialog.getExecutionNumber();
      ClientFile file = dialog.getConfigurationFile();
      if (dialog.isConfirmed()) {
        String[][] content =
          generateParameterConfigurationTable(parameters, numExecs);
        if (content != null) {
          try {
            boolean writeFile = true;
            if (dialog.showPreview()) {
              CSVPreviewDialog preview =
                new CSVPreviewDialog(application, file, content,
                  CONFIG_FILE_HEADER_SIZE);
              preview.setVisible(true);
              if (!preview.isConfirmed()) {
                writeFile = false;
              }
            }
            if (writeFile) {
              ApplicationFrame applicationFrame =
                application.getApplicationFrame();
              CSVFileUtils.writeCSVFile(applicationFrame, content, file);
              StatusBar statusBar = applicationFrame.getStatusBar();
              statusBar.setInfo(application
                .getString("exportConfigFileSuccess"), 10);
            }
          }
          catch (OperationFailureException e) {
            application.showException(application
              .getString("exportConfigFileError"), e.getCause());
          }
        }
        else {
          application.showWarning(application
            .getString("noParametersToExportWarning"));
        }
      }
      return file;
    }
    return null;
  }

  /**
   * Gera a tabela de valores de parmetros com os valores dos parmetros a
   * serem exportados para configurao de execuo de mltiplos fluxos a partir
   * da parametrizao atual.
   * 
   * @param parameters parmetros a serem exportados.
   * @param numExecs o nmero de execues
   * @return string[][] a tabela de valores de parmetros.
   */
  private String[][] generateParameterConfigurationTable(
    List<ParameterSelection> parameters, int numExecs) {
    int numLines = CONFIG_FILE_HEADER_SIZE + numExecs;
    int numColumns = parameters.size();
    String[][] table = new String[numLines][numColumns];

    for (int j = 0; j < numColumns; j++) {
      ParameterSelection selection = parameters.get(j);
      NodeParameter parameter = selection.parameter;
      FlowNode node = selection.node;

      /*
       * Se alterar o nmero de campos setados aqui, precisamos atualizar o
       * CSV_HEADER_SIZE.
       */
      table[0][j] = node.getAlgorithmName();
      table[1][j] = parameter.getLabel();
      table[2][j] = node.getId() + "." + parameter.getName();
      for (int i = CONFIG_FILE_HEADER_SIZE; i < numExecs
        + CONFIG_FILE_HEADER_SIZE; i++) {
        table[i][j] = parameter.getValue();
      }
    }
    return table;
  }

  /**
   * Mostra a janela para importao da configurao para execuo de mltiplos
   * fluxos.
   * 
   * @param suggestedFile arquivo de configurao sugerido.
   * 
   * @return o mapeamento dos parmetros resultante da configurao.
   * @throws OperationFailureException em caso de erro durante a importao.
   */
  public String[][] importMultipleFlowConfiguration(ClientFile suggestedFile)
    throws OperationFailureException {
    ImportMultipleFlowConfigurationDialog importDialog =
      new ImportMultipleFlowConfigurationDialog(application, suggestedFile);
    importDialog.setModalityType(ModalityType.APPLICATION_MODAL);
    importDialog.setVisible(true);
    selectedFile = importDialog.getConfigurationFile();
    if (importDialog.isConfirmed() && selectedFile != null) {
      boolean continueImport = true;
      if (importDialog.showPreview()) {
        CSVPreviewDialog preview =
          new CSVPreviewDialog(application, selectedFile,
            CONFIG_FILE_HEADER_SIZE);
        preview.setVisible(true);
        if (!preview.isConfirmed()) {
          continueImport = false;
        }
      }
      if (continueImport) {
        return CSVFileUtils.readCSVFile(application.getApplicationFrame(),
          selectedFile);
      }
    }
    return null;
  }

  /**
   * Executa mltiplos fluxos com variao em seus parmetros.
   * 
   * @param flow o fluxo a ser executado.
   * @return mapeamento dos parmetros para execuo mltipla.
   */
  public String[][] configureMultipleExecution(Flow flow) {
    try {
      String[][] mappings = importMultipleFlowConfiguration(selectedFile);
      if (mappings != null && mappings.length > 0) {
        mappings = removeHeader(mappings, CONFIG_FILE_HEADER_SIZE - 1);
        return mappings;
      }
    }
    catch (OperationFailureException e) {
      application.showException(application.getString("importConfigFileError"),
        e.getCause());
    }
    return null;
  }

  /**
   * Remove as linhas de cabealho que no fazem parte do mapeamento.
   * 
   * @param mappings o mapeamento com cabealho.
   * @param num nmero de linha do cabealho.
   * @return o mapeamento sem o cabealho.
   */
  private String[][] removeHeader(String[][] mappings, int num) {
    String[][] newArray = new String[mappings.length - num][];
    System.arraycopy(mappings, num, newArray, 0, mappings.length - num);
    return newArray;
  }

}
