package csbase.client.applications.jobmonitor;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.imageio.ImageIO;
import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.jobmonitor.actions.DetailsJobInfoAction;
import csbase.client.applications.jobmonitor.actions.RefreshJobInfoAction;
import csbase.client.applications.jobmonitor.drivers.JobInfoDriver;
import csbase.client.applications.jobmonitor.drivers.PBSDriver;
import csbase.client.applications.jobmonitor.filters.FilteredTablePanel;
import csbase.client.applications.jobmonitor.filters.JobInfoFilter;
import csbase.client.applications.jobmonitor.rowmodel.JobInfoRow;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;
import csbase.client.facilities.configurabletable.table.RowToKey;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.types.PVTables;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.exception.CSBaseException;
import csbase.logic.SGASet;
import csbase.logic.Utilities;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Aplicao que exibe todas as informaes coletadas dos jobs em execuo no
 * SGA.
 * 
 * @author Tecgraf
 */
public class JobMonitor extends Application {

  /**
   * Painel que exibe a(s) tabela(s) de informaes de jobs.
   */
  private JPanel contentPanel;

  /**
   * Mapa que armazena os drives que convertem informaes de jobs em uma tabela
   * configurvel.
   */
  private Map<String, JobInfoDriver> drivers;

  /**
   * Mapa que armazena as tabelas da aplicao.
   */
  private Map<String, ConfigurableTable<JobInfoRow>> tables;

  /**
   * Constante usada para mapear o driver de PBS. <br/>
   * TODO - [CSBASE-2901] refatorar a carga dos drivers.
   */
  private String PBS_DRIVER = "PBS_DRIVER";

  /**
   * Ao que atualiza a lista de servidores.
   */
  private AbstractAction refresh;

  /**
   * Ao que exibe os detalhes de um jobinfo.
   */
  private AbstractAction details;

  /**
   * Ao que fecha a aplicao.
   */
  private AbstractAction exit;

  /**
   * Thread encarregada de atualizar os dados da aplicao.
   */
  private Thread updateDataThread;

  /**
   * Construtor padro.
   * 
   * @param id - identificador da aplicao.
   * @throws ApplicationException - caso de erro na task.
   */
  public JobMonitor(String id) throws ApplicationException {
    super(id);

    contentPanel = new JPanel(new GridBagLayout());
    drivers = new HashMap<String, JobInfoDriver>();
    tables = new HashMap<String, ConfigurableTable<JobInfoRow>>();

    refresh = new RefreshJobInfoAction(this);
    details = new DetailsJobInfoAction(this);
    exit = new ApplicationExitAction(this);

    setConsoleInfo(getString("console.wait.jobs.info"));

    loadDrivers();

    buildInterface();

    updateDataThread = new UpdateDataThread(this);
    updateDataThread.start();
  }

  /**
   * Atualiza o painel que exibe as tabelas com as informaes dos jobs.
   */
  public void updateContent() {

    try {
      List<SGASet> allSGASet = getSGASetList();

      // Antes de atualizar o conteudo da aplicao,  necessrio separar cada
      // SGASet com seu respectivo driver.
      // TODO - [CSBASE-2901] refatorar a carga dos drivers.
      Map<String, List<SGASet>> sgaByDriver =
        new HashMap<String, List<SGASet>>();

      sgaByDriver.put(PBS_DRIVER, new ArrayList<SGASet>());

      for (SGASet sgaSet : allSGASet) {
        // TODO - At agora s temos clusters PBS usando essa aplicao.
        if (sgaSet.isCluster()) {
          sgaByDriver.get(PBS_DRIVER).add(sgaSet);
        }
      }

      // Dado que os SGASets esto separados por driver, basta atualizar cada
      // tabela associada ao driver.
      for (Entry<String, List<SGASet>> entry : sgaByDriver.entrySet()) {
        String driverId = entry.getKey();
        List<SGASet> sgaSetList = entry.getValue();

        updateView(driverId, sgaSetList);
      }

      String timeText = Utilities.getFormattedDate((new Date()).getTime());
      setConsoleInfo(getString("console.last.update") + timeText);

    }
    catch (CSBaseException e) {
      setConsoleError(getString("console.connecting.error"));
    }

  }

  /**
   * Retorna a primeira linha selecionada da primeira tabela que contem linhas
   * selecionadas.
   * 
   * @return primeira linha selecionada da primeira tabela que contem linhas
   *         selecionadas.
   */
  public JobInfoRow getSelectedObject() {
    for (ConfigurableTable<JobInfoRow> table : tables.values()) {
      if (table.getSelectedRowCount() > 0) {
        return table.getSelectedObjects().get(0);
      }
    }

    return null;
  }

  /**
   * Obtm informaes do sistema definidas no arquivo de propriedades da
   * aplicao.
   * 
   * @return informaes do sistema definidas no arquivo de propriedades da
   *         aplicao.
   */
  public JobMonitorSystemInfo getSystemInfo() {
    Icon systemIcon = getSystemIcon();
    String systemName = getStringSpecificProperty("system.name");

    return new JobMonitorSystemInfo(systemIcon, systemName);
  }

  // ---------------- Mtodos sobrescritos --------------------------

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
    PreferenceCategory pc = getPreferences();
    PVTables pv = (PVTables) pc.getPreference(JobMonitorPref.PBS_TABLES);

    pv.storeTables(new ArrayList<ConfigurableTable<?>>(tables.values()));

    savePreferences();

    if (!updateDataThread.isInterrupted()) {
      updateDataThread.interrupt();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  // ---------------- Mtodos privados -----------------------------

  /**
   * Carrega os drives que convertem informaes de jobs em uma tabela
   * configurvel.
   */
  private void loadDrivers() {
    // TODO - [CSBASE-2901] refatorar a carga dos drivers.
    drivers.put(PBS_DRIVER, new PBSDriver(this));
  }

  /**
   * Constroi a interface da aplicao.
   */
  private void buildInterface() {
    final ApplicationFrame appFrame = getApplicationFrame();
    appFrame.setLayout(new GridBagLayout());
    appFrame.setJMenuBar(createMenuBar());
    appFrame.add(createToolBar(), new GBC(0, 0).horizontal());

    appFrame.add(contentPanel, new GBC(0, 1).both().top(10));

    appFrame.setPreferredSize(new Dimension(700, 400));
    appFrame.pack();
  }

  /**
   * Atualiza os dados das tabelas.
   * 
   * @param driverId - identificador do driver.
   * @param sgas - lista de SGASets.
   */
  private void updateView(String driverId, List<SGASet> sgas) {
    JobInfoDriver driver = drivers.get(driverId);

    if (tables.containsKey(driverId)) {
      ConfigurableTable<JobInfoRow> table = tables.get(driverId);
      List<JobInfoRow> newRows = driver.getRows(sgas);
      table.updateRows(newRows);
    }
    else {
      ConfigurableTable<JobInfoRow> table = driver.getTable(sgas);

      if (table != null) {
        addTableListeners(table);
        table.setFilter(new JobInfoFilter());
        table.setRowToKey(new RowToKey<JobInfoRow>() {
          @Override
          public String getKey(JobInfoRow row) {
            String value = row.get(JobInfoDriver.JOB_ID_PROPERTY);
            return value;
          }
        });

        PreferenceCategory pc = getPreferences();
        PVTables pv = (PVTables) pc.getPreference(JobMonitorPref.PBS_TABLES);
        pv.loadTable(table);

        contentPanel.add(new FilteredTablePanel(this, table),
          new GBC(0, tables.size()).both().insets(10));

        tables.put(driverId, table);

        contentPanel.repaint();
      }
    }
  }

  /**
   * Cria o menu da aplicao.
   * 
   * @return menu da aplicao.
   */
  private JMenuBar createMenuBar() {
    JMenu servers = new JMenu(getString("menu.servers"));
    servers.add(details);
    servers.add(refresh);
    servers.add(exit);

    JMenuBar menuBar = new JMenuBar();
    menuBar.add(servers);

    return menuBar;
  }

  /**
   * Cria a barra de ferramentas da aplicao.
   * 
   * @return barra de ferramentas da aplicao.
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);

    toolBar.add(details);
    toolBar.add(refresh);

    return toolBar;
  }

  /**
   * Adiciona ouvintes de seleo na tabela.
   * 
   * @param table - tabela.
   */
  private void addTableListeners(final ConfigurableTable<JobInfoRow> table) {

    // adicionando ouvinte de seleo de linhas da tabela
    table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          details.setEnabled(table.getSelectedRowCount() == 1);
        }
      });

    // adicionando ouvinte de duplo-click do mouse
    table.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2 && details.isEnabled()) {
          details.actionPerformed(null);
        }
      }
    });

  }

  /**
   * Lista com todos os SGASets cadastrados no servidor.
   * 
   * @return lista de SGASets
   * 
   * @throws CSBaseException - quando no foi possvel obter a lista de SGASets.
   */
  private List<SGASet> getSGASetList() throws CSBaseException {

    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;

    RemoteTask<List<SGASet>> task = new RemoteTask<List<SGASet>>() {
      @Override
      protected void performTask() {
        try {
          List<SGASet> result = new ArrayList<SGASet>();
          for (String sgaName : sgaService.getAllSGANames()) {
            SGASet sgaSet = sgaService.getSGASet(sgaName);
            if (sgaSet.getAlive()) {
              result.add(sgaSet);
            }
          }
          setResult(result);
        }
        catch (RemoteException e) {
          setResult(null);
        }
      }
    };

    task.execute(getApplicationFrame(), getName(),
      getString("task.message.sga.list"));

    List<SGASet> result = task.getResult();

    if (result == null) {
      throw new CSBaseException(
        "No foi possvel obter as informaes dos SGAs.");
    }

    return result;
  }

  /**
   * Obtem o icone que representa o sistema CSBASE em execuo.
   * 
   * @return icone que representa o sistema CSBASE em execuo.
   */
  private Icon getSystemIcon() {

    final String systemIconPath = getStringSpecificProperty("system.icon");

    String[] resourcePath = systemIconPath.split("/");

    InputStream input = getResource(resourcePath);

    Icon icon = null;

    if (input != null) {
      try {
        BufferedImage image = ImageIO.read(input);
        icon = new ImageIcon(image);
      }
      catch (IOException e) {
        // faz nada
      }
      finally {
        FileUtils.close(input);
      }
    }

    return icon;
  }

  /**
   * Define a mensagem do tipo INFO a ser exibida no console da aplicao.
   * 
   * @param msg - mensagem.
   */
  private void setConsoleInfo(String msg) {
    getApplicationFrame().getStatusBar().setInfo(msg);
  }

  /**
   * Define a mensagem do tipo ERROR a ser exibida no console da aplicao.
   * 
   * @param msg - mensagem.
   */
  private void setConsoleError(String msg) {
    getApplicationFrame().getStatusBar().setError(msg);
  }

  /**
   * Thread encarregada de atualizar os dados.
   * 
   * @author Tecgraf
   */
  private class UpdateDataThread extends Thread {

    /** Flag utilizada pela thread que atualiza os dados do SGA. */
    private boolean hasToUpdate;

    /** Referncia para a aplicao. */
    private JobMonitor app;

    /**
     * Construtor default.
     * 
     * @param app - referncia para a aplicao.
     */
    public UpdateDataThread(JobMonitor app) {
      super("UpdateDataThread");
      this.app = app;
      this.hasToUpdate = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      int updateIntervalAsInt = SGAProxy.getUpdateInterval();
      if (updateIntervalAsInt == -1) {
        return;
      }
      final long updateInterval = (long) updateIntervalAsInt * (long) 1000;

      while (hasToUpdate) {
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            updateContent();

            app.setChanged();
            app.notifyObservers();
          }
        });

        waitUpdateInterval(updateInterval);
      }
    }

    /**
     * Mtodo auxiliar que faz com que a thread espere um determinado intervalo.
     * 
     * @param updateInterval - intervalo que a thread ir esperar.
     */
    private void waitUpdateInterval(long updateInterval) {
      try {
        Thread.sleep(updateInterval);
      }
      catch (InterruptedException e) {
        // faz nada
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void interrupt() {
      synchronized (this) {
        this.hasToUpdate = false;
      }
      super.interrupt();
    }
  }

}
