package csbase.client.applications.statsviewer;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.TitledBorder;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.calendar.CalendarDateField;
import csbase.client.applications.statsviewer.LogsView.DateViewType;
import csbase.client.applications.statsviewer.LogsView.LogsDataType;

/**
 * Painel com todos os dados para realizao da busca nos logs de eventos.
 *
 * @author Tecgraf/PUC-Rio
 */
public class SearchPanel extends JPanel {

  /**
   * Radio-button para selecionar estatsticas de Acesso ao Sistema (login de
   * usurios).
   */
  private JRadioButton loginRadio;

  /**
   * Radio-button para selecionar estatsticas da Utilizao das Aplicaes.
   */
  private JRadioButton applicationsRadio;

  /**
   * Radio-button para selecionar estatsticas da Execuo de Comandos (simples
   * e fluxo).
   */
  private JRadioButton executionsRadio;

  /**
   * Combo-box para exibir os perodos de tempo predefinidos disponveis.
   */
  private JComboBox selectTimeCombo;

  /**
   * Perodos de tempo pr-determinados existentes na combo.
   *
   * @author Tecgraf/PUC-Rio
   */
  public enum TimeType {

    /** Perodo personalizado (qualquer data inicial e final) */
    Custom,

    /** Ms anterior */
    PreviousMonth,

    /** 6 Meses anteriores */
    PreviousSixMonth,

    /** Ano anterior */
    PreviousYear,

    /** 3 Anos anteriores */
    PreviousThreeYear,

    ;

  }

  /**
   * Calendrio para escolha do incio do perodo.
   */
  private CalendarDateField startDateCalendar;

  /**
   * Calendrio para escolha do trmino do perodo.
   */
  private CalendarDateField endDateCalendar;

  /**
   * Label para separar perodo de tempo inicial e final.
   */
  private JLabel divLabel;

  /**
   * Boto para pesquisar logs.
   */
  private JButton searchButton;

  /**
   * Boto para limpar dados e fazer nova busca.
   */
  private JButton newsearchButton;

  /**
   * Boto para exportar CSV com dados da busca corrente.
   */
  private JButton exportCSVButton;

  /**
   * Nvel de detalhe para agrupar a data nos grficos: por ano.
   */
  private JRadioButton byYearButton;

  /**
   * Nvel de detalhe para agrupar a data nos grficos: por ms.
   */
  private JRadioButton byMonthButton;

  /**
   * Nvel de detalhe para agrupar a data nos grficos: por dia.
   */
  private JRadioButton byDayButton;

  /**
   * Referncia para aplicao.
   */
  private final StatsViewer appViewer;

  /**
   * Construtor.
   *
   * @param appViewer referncia para aplicao
   */
  public SearchPanel(StatsViewer appViewer) {
    super(new GridBagLayout());
    this.appViewer = appViewer;

    GBC gbc = new GBC(0, 0).northwest().insets(5).horizontal();
    add(buildStatsPanel(), gbc);

    gbc = new GBC(0, 1).northwest().insets(5).horizontal();
    add(buildTimePanel(), gbc);

    gbc = new GBC(0, 2).northwest().insets(5).horizontal();
    add(buildButtonsPanel(), gbc);

    gbc = new GBC(0, 3).northwest().insets(5).horizontal();
    add(buildFilterDatePanel(), gbc);

    selectTimeCombo.setSelectedIndex(TimeType.PreviousYear.ordinal());
  }

  /**
   * Cria o painel com os check-boxes para escolha da estatstica de interesse
   * na busca.
   *
   * @return painel para seleo da estatstica de interesse.
   */
  private JPanel buildStatsPanel() {

    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(new TitledBorder(appViewer.getString("statsPanel.title")));

    loginRadio = new JRadioButton(appViewer.getString("statsLogin"), true);
    loginRadio.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        appViewer.setLogType(LogsDataType.LoginData);
      }
    });

    applicationsRadio =
      new JRadioButton(appViewer.getString("statsApplications"));
    applicationsRadio.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        appViewer.setLogType(LogsDataType.ApplicationsData);
      }
    });

    executionsRadio = new JRadioButton(appViewer.getString("statsExecutions"));
    executionsRadio.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        appViewer.setLogType(LogsDataType.ExecutionsData);
      }
    });

    ButtonGroup group = new ButtonGroup();
    group.add(loginRadio);
    group.add(applicationsRadio);
    group.add(executionsRadio);

    GBC gbc = new GBC(0, 0).insets(5, 5, 0, 5).northwest().pushx();
    panel.add(loginRadio, gbc);

    gbc = new GBC(0, 1).insets(0, 5, 0, 5).northwest();
    panel.add(applicationsRadio, gbc);

    gbc = new GBC(0, 2).insets(0, 5, 0, 5).northwest();
    panel.add(executionsRadio, gbc);

    // opo default
    appViewer.setLogType(LogsDataType.LoginData);

    return panel;
  }

  /**
   * Cria o painel para escolha do perdo de tempo pesquisado, contm a combo
   * para seleo de perodo pre-determinado e calendrios para definio de
   * data de incio e de trmino.
   *
   * @return painel para escolha do perodo de tempo.
   */
  private JPanel buildTimePanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(new TitledBorder(appViewer
      .getString("statsPanel.time.title")));

    List<String> items = new ArrayList<String>();
    for (TimeType time : TimeType.values()) {
      items.add(appViewer.getString(time.name()));
    }
    selectTimeCombo = new JComboBox(items.toArray());
    GUIUtils.matchPreferredSizes(selectTimeCombo);

    selectTimeCombo.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        newSelectedTime();
      }
    });
    GBC gbc = new GBC(0, 0).northwest().width(3).insets(10, 10, 0, 5);
    panel.add(selectTimeCombo, gbc);

    startDateCalendar = new CalendarDateField();
    gbc = new GBC(0, 1).northwest().insets(10, 10, 5, 0).horizontal();
    panel.add(startDateCalendar, gbc);

    divLabel = new JLabel("-");
    gbc = new GBC(1, 1).center().insets(10, 5, 5, 5);
    panel.add(divLabel, gbc);

    endDateCalendar = new CalendarDateField();
    gbc = new GBC(2, 1).northwest().insets(10, 5, 5, 5).horizontal();
    panel.add(endDateCalendar, gbc);

    return panel;
  }

  /**
   * Trata a seleo de um novo perodo de tempo na combo. Os campos data de
   * incio e de trmino ficam disponveis para edio somente se o item
   * selecionado na combo for "personalizado". <br>
   * Alm disso, conforme o perodo escolhido atualiza os valores nos
   * calendrios com data de incio e trmino.
   */
  private void newSelectedTime() {
    TimeType selected = TimeType.values()[selectTimeCombo.getSelectedIndex()];
    boolean isCustom = selected.equals(TimeType.Custom);
    startDateCalendar.setEnabled(isCustom);
    divLabel.setEnabled(isCustom);
    endDateCalendar.setEnabled(isCustom);

    Calendar calendar = Calendar.getInstance();
    switch (selected) {
      case PreviousMonth:
        calendar.add(Calendar.MONTH, -1);
        startDateCalendar.setDate(calendar.getTime());
        calendar.add(Calendar.MONTH, 1);
        endDateCalendar.setDate(calendar.getTime());
        break;
      case PreviousSixMonth:
        calendar.add(Calendar.MONTH, -6);
        startDateCalendar.setDate(calendar.getTime());
        calendar.add(Calendar.MONTH, 6);
        endDateCalendar.setDate(calendar.getTime());
        break;
      case PreviousYear:
        calendar.add(Calendar.YEAR, -1);
        startDateCalendar.setDate(calendar.getTime());
        calendar.add(Calendar.YEAR, 1);
        endDateCalendar.setDate(calendar.getTime());
        break;
      case PreviousThreeYear:
        calendar.add(Calendar.YEAR, -3);
        startDateCalendar.setDate(calendar.getTime());
        calendar.add(Calendar.YEAR, 3);
        endDateCalendar.setDate(calendar.getTime());
        break;
      case Custom:
        // vazio -- nada a fazer
    }

    updatedTimeType(selected);
  }

  /**
   * @return painel com boto para pesquisar os logs.
   */
  private JPanel buildButtonsPanel() {
    JPanel panel = new JPanel(new GridBagLayout());

    searchButton =
      new JButton(new AbstractAction(appViewer.getString("search")) {
        @Override
        public void actionPerformed(ActionEvent e) {

          if (!validateSearch()) {
            return;
          }
          searchButton.setEnabled(false);
          selectTimeCombo.setEnabled(false);
          startDateCalendar.setEnabled(false);
          endDateCalendar.setEnabled(false);
          loginRadio.setEnabled(false);
          applicationsRadio.setEnabled(false);
          executionsRadio.setEnabled(false);
          appViewer
            .find(startDateCalendar.getDate(), endDateCalendar.getDate());
          newsearchButton.setEnabled(true);
          exportCSVButton.setEnabled(true);
        }
      });

    newsearchButton =
      new JButton(new AbstractAction(appViewer.getString("newsearch")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          newsearchButton.setEnabled(false);
          exportCSVButton.setEnabled(false);
          appViewer.clearInfo();
          searchButton.setEnabled(true);
          selectTimeCombo.setEnabled(true);
          startDateCalendar.setEnabled(true);
          endDateCalendar.setEnabled(true);
          loginRadio.setEnabled(true);
          applicationsRadio.setEnabled(true);
          executionsRadio.setEnabled(true);
        }
      });

    exportCSVButton =
      new JButton(new AbstractAction(appViewer.getString("export.csv")) {
        @Override
        public void actionPerformed(ActionEvent arg0) {
          appViewer.exportCSV();
        }
      });
    newsearchButton.setEnabled(false);
    exportCSVButton.setEnabled(false);

    panel.add(searchButton, new GBC(0, 0).insets(10, 0, 10, 0));
    panel.add(newsearchButton, new GBC(1, 0).insets(10, 10, 10, 0));
    panel.add(exportCSVButton, new GBC(0, 1).insets(10, 10, 10, 0).width(2));
    return panel;
  }

  /**
   * Valida os campos obrigatrios do painel de busca.
   *
   * @return true, se data de incio e fim foram preenchidas, false, caso
   *         contrrio.
   */
  private boolean validateSearch() {
    if (startDateCalendar.isEnabled() && startDateCalendar.getDate() == null) {
      StandardDialogs.showInfoDialog(appViewer.getApplicationFrame(), appViewer
        .getString("warning.title"), appViewer.getString("noStartDate"));
      return false;
    }
    if (endDateCalendar.isEnabled() && endDateCalendar.getDate() == null) {
      StandardDialogs.showInfoDialog(appViewer.getApplicationFrame(), appViewer
        .getString("warning.title"), appViewer.getString("noEndDate"));
      return false;
    }
    return true;
  }

  /**
   * @return painel para filtrar detalhe de exibio por data.
   */
  private JPanel buildFilterDatePanel() {

    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(new TitledBorder(appViewer.getString("filterPanel.title")));

    ActionListener listener = new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        appViewer.filterDateViewType(getDateViewType());
      }
    };
    byYearButton =
      new JRadioButton(appViewer.getString("filterPanel.byYearButton"));
    byYearButton.addActionListener(listener);

    byMonthButton =
      new JRadioButton(appViewer.getString("filterPanel.byMonthButton"));
    byMonthButton.addActionListener(listener);

    byDayButton =
      new JRadioButton(appViewer.getString("filterPanel.byDayButton"));
    byDayButton.addActionListener(listener);

    ButtonGroup detailGroup = new ButtonGroup();
    detailGroup.add(byYearButton);
    detailGroup.add(byMonthButton);
    detailGroup.add(byDayButton);

    GBC gbc = new GBC(0, 0).insets(0, 5, 5, 0).northwest();
    panel.add(byYearButton, gbc);

    gbc = new GBC(1, 0).insets(0, 5, 5, 0).northwest();
    panel.add(byMonthButton, gbc);

    gbc = new GBC(2, 0).insets(0, 5, 5, 5).northwest().pushy();
    panel.add(byDayButton, gbc);

    return panel;
  }

  /**
   * Trata a atualizao do item selecionado na combo com os perodos de tempo.
   * Conforme o perodo de tempo selecionado os checks sofrem alterao da opo
   * default selecionada e das opes disponveis (habilitadas/desabilitadas).
   *
   * @param selectedTime novo perodo selecionado.
   */
  private void updatedTimeType(TimeType selectedTime) {

    switch (selectedTime) {

      case PreviousMonth:
        byDayButton.setSelected(true);
        byDayButton.setEnabled(true);
        byMonthButton.setEnabled(false);
        byYearButton.setEnabled(false);
        break;

      case PreviousSixMonth:
        byMonthButton.setSelected(true);
        byDayButton.setEnabled(true);
        byMonthButton.setEnabled(true);
        byYearButton.setEnabled(false);
        break;

      case PreviousYear:
        byMonthButton.setSelected(true);
        byDayButton.setEnabled(true);
        byMonthButton.setEnabled(true);
        byYearButton.setEnabled(true);
        break;

      case Custom:
      case PreviousThreeYear:
        byYearButton.setSelected(true);
        byDayButton.setEnabled(true);
        byMonthButton.setEnabled(true);
        byYearButton.setEnabled(true);
        break;
    }

    // radios de configurao de agregador de datas foram modificados
    // atualizar informaes
    appViewer.filterDateViewType(getDateViewType());

  }

  /**
   * @return consulta o tipo de configurao atualmente selecionado.
   */
  public DateViewType getDateViewType() {
    if (byYearButton.isSelected()) {
      return DateViewType.YEAR;
    }
    else if (byMonthButton.isSelected()) {
      return DateViewType.MONTH;
    }
    else {
      return DateViewType.DAY;
    }
  }

}
