package csbase.client.project;

import java.awt.Window;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.ProjectFileNameAndTypeFilter;
import csbase.client.kernel.ClientException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileFilter;
import csbase.logic.filters.ProjectFileCompositeFilter;
import csbase.logic.filters.ProjectFileCompositeOrFilter;
import csbase.logic.filters.ProjectFileDirectoryFilter;
import csbase.logic.filters.ProjectFileNotDirectoryFilter;
import csbase.logic.filters.ProjectFileTypeFilter;

/**
 * Essa classe implementa um navegador para abertura de um ou mais tipos de
 * arquivos de um projeto. Pode-se configurar se o navegador vai permitir, por
 * exemplo, a seleo mltipla de arquivos, ou se vai permitir somente a seleo
 * de arquivos e no de diretrios.
 */
public class ProjectFileChooserOpen extends ProjectFileChooser {

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto, de
   * acordo com o modo de seleo especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode) throws ClientException {
    this(owner, project, isMultiSelectionEnabled, mode, null);
  }

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto,
   * que sejam de um determinado tipo, de acordo com o modo de seleo
   * especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileType tipo de arquivo default do navegador
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode, String fileType)
    throws ClientException {
    super(owner, LNG.get("PRJ_FILE_CHOOSER_OPEN_TITLE"), project,
      isMultiSelectionEnabled, mode, fileType, createVisualFilter(mode,
        fileType), createSelectionFilter(mode, fileType), false, null, false);
    String title = getTitle();
    if (title != null) {
      dialog.setTitle(title);
    }
    dialog.setVisible(true);
  }

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto,
   * que sejam de um determinado tipo, de acordo com o modo de seleo
   * especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileType tipo de arquivo default do navegador
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode, String fileType,
    boolean useFilter) throws ClientException {
    super(owner, LNG.get("PRJ_FILE_CHOOSER_OPEN_TITLE"), project,
      isMultiSelectionEnabled, mode, fileType, createVisualFilter(mode,
        fileType), createSelectionFilter(mode, fileType), false, null,
      useFilter);
    String title = getTitle();
    if (title != null) {
      dialog.setTitle(title);
    }
    dialog.setVisible(true);
  }

  /**
   * Contri um navegador para abertura de um ou mais tipos de arquivos de um
   * projeto. O modo de seleo no navegador permite somente a seleo de
   * arquivos (@see ProjectFileChooser.FILE_ONLY).
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @param defaultType tipo default que vai aparecer selecionado na lista
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, String[] fileTypes, String defaultType)
    throws ClientException {
    super(owner, LNG.get("PRJ_FILE_CHOOSER_OPEN_TITLE"), project,
      isMultiSelectionEnabled, ProjectFileChooser.FILE_ONLY, defaultType,
      createVisualFilter(fileTypes), createSelectionFilter(fileTypes), false,
      null, false);
    String title = getTitle();
    if (title != null) {
      dialog.setTitle(title);
    }
    dialog.setVisible(true);
  }

  /**
   * Cria um filtro de seleo de mltiplos tipos de arquivos.
   * 
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @return o filtro criado
   */
  private static ProjectFileFilter createSelectionFilter(String[] fileTypes) {
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();

    projectFileCompositeFilter.addChild(new ProjectFileTypeFilter(fileTypes));

    return projectFileCompositeFilter;
  }

  /**
   * Cria um filtro visual de mltiplos tipos de arquivos.
   * 
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @return o filtro criado
   */
  private static ProjectFileFilter createVisualFilter(String[] fileTypes) {
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());

    projectFileCompositeFilter.addChild(new ProjectFileTypeFilter(fileTypes));
    return projectFileCompositeFilter;
  }

  /**
   * Cria um filtro de seleo de um determinado tipo de arquivo, de acordo com
   * o modo de seleo especificado para o navegador.
   * 
   * @param mode modo de seleo no navegador
   * @param fileType tipo de arquivo permitido
   * @return o filtro criado
   */
  private static ProjectFileFilter createSelectionFilter(int mode,
    String fileType) {
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      if (fileType == null) {
        projectFileCompositeFilter.addChild(ProjectFileNotDirectoryFilter
          .getInstance());
      }
      else {
        projectFileCompositeFilter
          .addChild(new ProjectFileTypeFilter(fileType));
      }
    }
    if ((mode == ProjectFileChooser.DIRECTORY_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
        .getInstance());
    }
    return projectFileCompositeFilter;
  }

  /**
   * Cria um filtro visual de um determinado tipo de arquivo, de acordo com o
   * modo de seleo especificado para o navegador.
   * 
   * @param mode modo de seleo no navegador
   * @param fileType tipo de arquivo permitido
   * @return o filtro criado
   */
  private static ProjectFileFilter createVisualFilter(int mode, String fileType) {
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      if (fileType == null) {
        projectFileCompositeFilter.addChild(ProjectFileNotDirectoryFilter
          .getInstance());
      }
      else {
        projectFileCompositeFilter
          .addChild(new ProjectFileTypeFilter(fileType));
      }
    }
    return projectFileCompositeFilter;
  }

  @Override
  protected ProjectFileFilter getVisualFilter() {
    return ProjectFileChooserOpen.createVisualFilter(this.mode, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getVisualFilter(String fileType) {
    return ProjectFileChooserOpen.createVisualFilter(this.mode, fileType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getVisualFilter(final String nameFilter,
    final String fileType) {
    if (nameFilter == null || nameFilter.isEmpty()) {
      return ProjectFileChooserOpen.createVisualFilter(mode, fileType);
    }
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      projectFileCompositeFilter.addChild(new ProjectFileNameAndTypeFilter(
        nameFilter, fileType));
    }
    return projectFileCompositeFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter() {
    return ProjectFileChooserOpen.createSelectionFilter(this.mode, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(String fileType) {
    return ProjectFileChooserOpen.createSelectionFilter(this.mode, fileType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(final String nameFilter,
    final String fileType) {
    if (nameFilter == null || nameFilter.isEmpty()) {
      return ProjectFileChooserOpen.createSelectionFilter(mode, fileType);
    }
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      projectFileCompositeFilter.addChild(new ProjectFileNameAndTypeFilter(
        nameFilter, fileType));
    }
    return projectFileCompositeFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean handleAction() {
    ClientProjectFile[] selectedFiles = this.getSelectedFiles();
    if (selectedFiles != null) {
      this.selectedPaths = new ProjectTreePath[selectedFiles.length];
      for (int i = 0; i < this.selectedPaths.length; i++) {
        this.selectedPaths[i] = new ProjectTreePath(selectedFiles[i]);
      }
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void makeSelection(ClientProjectFile[] projectFileArray) {
    if (projectFileArray.length == 0) {
      this.selectedPaths = null;
      this.fileNameText.setText("");
    }
    else {
      this.selectedPaths = new ProjectTreePath[projectFileArray.length];
      String newText = "";
      String separator = "";
      for (int i = 0; i < projectFileArray.length; i++) {
        this.selectedPaths[i] = new ProjectTreePath(projectFileArray[i]);
        newText += (separator + projectFileArray[i].getName());
        separator = ",";
      }
      this.fileNameText.setText(newText);
    }
  }
}
