/*
 * $Id$
 */
package csbase.client.remote;

import java.awt.Window;
import java.util.Date;

import javax.swing.JDialog;
import javax.swing.JOptionPane;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.remote.srvproxies.EventLogProxy;
import csbase.logic.MonitoredServerListener;
import csbase.logic.ServerURI;

/**
 * Possui uma implementao padro de listener do cliente para monitorao de
 * conexo com o servidor. Exibe um JOptioPane quando a comunicao  perdida ou
 * restabelecida.
 */
public class ClientRemoteMonitorListener implements MonitoredServerListener {

  /**
   * Janela que partir da qual o JOptioPane ser exibido.
   */
  private final Window parent;

  /** JOptionPane unico para exibir mensagens de conexo. */
  private static JOptionPane messagePane = null;

  /** DIalogo que conter o JOptionPane */
  private static JDialog dialog;

  /**
   * Fila de log de aes do cliente
   */
  private static String[] desktopQueue =
    new String[] { "ConnectionMonitoring" };

  /**
   * Construtor.
   * 
   * @param parent Janela que partir da qual o JOptioPane ir aparecer.
   */
  public ClientRemoteMonitorListener(Window parent) {
    this.parent = parent;
    if (messagePane == null) {
      messagePane = new JOptionPane();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyConnectionLost(final ServerURI serverURI) {

    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        String serverName = serverURI.getHost();
        String msg =
          String.format(LNG.get("ClientRemoteMonitorListener.connection.lost"),
            serverName, FormatUtils.format(new Date()));
        String title =
          String.format(
            LNG.get("ClientRemoteMonitorListener.connection.lost.title"),
            serverName);

        messagePane.setMessage(msg);
        messagePane.setMessageType(JOptionPane.ERROR_MESSAGE);

        buildDialog(title);
      }
    });
  }

  /**
   * {@inheritDoc} Aps o restabelecimento da conexo, verifica se a verso 
   * compatvel com o servidor. Caso no seja, informa ao usurio.
   */
  @Override
  public void notifyConnectionReestablished(final ServerURI serverURI) {

    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {

        ClientRemoteMonitor monitor = ClientRemoteMonitor.getInstance();
        String serverName = serverURI.getHost();
        String title;

        if (monitor.isSameVersion()) {
          EventLogProxy.addClientInformation(desktopQueue,
            LNG.get("ClientRemoteMonitorListener.connection.reestablished.log"));

          String msg =
            String.format(
              LNG.get("ClientRemoteMonitorListener.connection.reestablished"),
              serverName, FormatUtils.format(new Date()));
          title =
            String.format(
              LNG
                .get("ClientRemoteMonitorListener.connection.reestablished.title"),
              serverName);

          messagePane.setMessage(msg);
          messagePane.setMessageType(JOptionPane.INFORMATION_MESSAGE);

        }
        else {
          EventLogProxy.addClientInformation(desktopQueue,
            LNG.get("ClientRemoteMonitorListener.different.version.log"));

          String msg =
            String.format(
              LNG.get("ClientRemoteMonitorListener.different.version"),
              serverName, FormatUtils.format(new Date()));
          title =
            String.format(
              LNG.get("ClientRemoteMonitorListener.different.version.title"),
              serverName);

          messagePane.setMessage(msg);
          messagePane.setMessageType(JOptionPane.ERROR_MESSAGE);
        }

        buildDialog(title);
      }

    });
  }

  private void buildDialog(String title) {
    if (dialog == null) {
      dialog = messagePane.createDialog(parent, title);
    }
    else {
      dialog.setTitle(title);
      dialog.repaint();
    }
    dialog.pack();
    dialog.setVisible(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyLoggedIn(ServerURI serverURI) {
    /** nada a fazer */
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyLoggedOut(ServerURI serverURI) {
    /** nada a fazer */
  }

}
