package csbase.client;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;

import csbase.client.ClientSmartFileEvent.OptimizationFailCause;
import csbase.client.desktop.DesktopFrame;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientOptimizationMode;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Modela a viso do cliente para um arquivo ou diretrio de um projeto no modo
 * otimizado.
 * <p>
 * Operaes de IO: Sempre tenta utilizar o mounting point definido em
 * localProjectPath. Caso no consiga, tem o comportamento de ClientProjectFile
 * <br>
 * Operao de controle: Em geral, repassa a ao para o ClientProjectFile
 * original.
 * <p>
 * Deve ser instanciada sempre pelo mtodo <code>create()</code> de
 * {@link ClientSmartFileFactory}
 *
 * @author Tecgraf
 */
public class ClientSmartFile implements ClientFile {

  /**
   * Modo de otimizao
   */
  private ClientOptimizationMode optimizationMode;

  /**
   * Viso de arquivo de projeto para acesso ao arquivo de projeto
   */
  private ClientProjectFile clientProjectFile = null;

  /**
   * Viso de arquio local para acesso ao arquivo de projeto no ambiente "local"
   */
  private ClientLocalFile clientLocalFile = null;

  /**
   * Arquivo que est aberto. Pode ser um ClienteProjetFile ou um
   * ClientLocalFile
   */
  private ClientFile openedFile = null;

  /**
   * Indica se o ltimo canRead() foi executado no arquivo local
   */
  private boolean canReadLocalFile = false;

  /**
   * 
   * Construtor. Deve ser chamado sempre por {@link ClientSmartFileFactory}.<br>
   * O arquivo a princpio mantm o modo de otimizao passado no construtor,
   * mas caso o arquivo local no seja encontrado ou no esteja acessvel para
   * leitura, a otimizao neste arquivo fica desabilitada.
   * 
   * @param cpf ClientProjectFile que ser utilizado para acesso ao projeto
   * @param optimizationMode Modo de otimizao
   * @param localProjectPath Caminho da rea local de projetos
   */
  public ClientSmartFile(ClientProjectFile cpf,
    ClientOptimizationMode optimizationMode, String localProjectPath) {

    this.optimizationMode = optimizationMode;

    /* ClientProjectFile */
    clientProjectFile = cpf;

    if (isOptimizedMode()) {
      /* Criando ClientLocalFile */
      String filePath = buildLocalFilePath(cpf.getPath(), localProjectPath);

      File file = new File(filePath);
      if (!file.exists()) {
        ClientSmartFileFactory.getInstance().notifyListeners(localProjectPath,
          filePath, OptimizationFailCause.FILE_PATH_NOT_VALID);
        this.optimizationMode = ClientOptimizationMode.NONE;
      }
      else if (!file.canRead()) {
        ClientSmartFileFactory.getInstance().notifyListeners(localProjectPath,
          filePath, OptimizationFailCause.CANT_READ_FILE);
        this.optimizationMode = ClientOptimizationMode.NONE;
      }
      else {
        clientLocalFile = new ClientLocalFile(file);
      }
    }
  }

  /**
   * 
   * @return O projeto que est aberto
   */
  private CommonClientProject getProject() {
    final DesktopFrame dsk = DesktopFrame.getInstance();
    final CommonClientProject prj = dsk.getProject();
    return prj;
  }

  /**
   * 
   * @return O ClientPtojectFile encapsulado
   */
  public ClientProjectFile getClientProjectFile() {
    return clientProjectFile;
  }

  /**
   * 
   * @return O ClientLocalFile (Arquivo na pasta local de projetos)
   */
  public ClientLocalFile getClientLocalFile() {
    return clientLocalFile;
  }

  /**
   * Cria o path "local" do arquivo
   * 
   * @param filePath Path do arquivo relativo ao projeto
   * @param localProjectPath Path do projeto
   * @return Path absoluto do arquivo como String
   */
  private String buildLocalFilePath(String[] filePath,
    String localProjectPath) {
    CommonClientProject project = getProject();
    String userAndProjectPath = (String) project.getId();
    userAndProjectPath.replace('/', File.separatorChar);

    StringBuilder strPath = new StringBuilder();

    strPath.append(localProjectPath);
    strPath.append(File.separator);
    strPath.append(userAndProjectPath);
    if (filePath.length > 0) {
      strPath.append(File.separator);
    }
    strPath.append(FileUtils.joinPath(filePath));

    return strPath.toString();
  }

  /**
   * 
   * @return true, se est em modo de otiizao
   */
  private boolean isOptimizedMode() {
    return optimizationMode == ClientOptimizationMode.GLOBAL;
  }

  /**
   * {@inheritDoc} - Repassa o controle sempre ao clientProjectFile (confirmar
   * isso!)
   */
  @Override
  public ClientFile[] getChildren() throws Exception {
    ClientProjectFile[] children = clientProjectFile.getChildren();
    ClientSmartFile[] smartChildren = new ClientSmartFile[children.length];
    for (int i = 0; i < children.length; i++) {
      ClientProjectFile child = (ClientProjectFile) children[i];
      ClientSmartFile csf = ClientSmartFileFactory.getInstance().create(child);
      smartChildren[i] = csf;
    }
    return smartChildren;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    if (!isOptimizedMode()) {
      return clientProjectFile.getInputStream();
    }
    try {
      return clientLocalFile.getInputStream();
    }
    catch (Exception e) {
      return clientProjectFile.getInputStream();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    if (!isOptimizedMode()) {
      return clientProjectFile.getOutputStream();
    }
    try {
      return clientLocalFile.getOutputStream();
    }
    catch (Exception e) {
      return clientProjectFile.getOutputStream();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() throws IOException {
    return clientProjectFile.exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    return clientProjectFile.getModificationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return clientProjectFile.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientFile getParent() {
    ClientProjectFile parent = clientProjectFile.getParent();
    return ClientSmartFileFactory.getInstance().create(parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return clientProjectFile.getPath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return clientProjectFile.getStringPath();
  }

  /**
   * 
   * @return O path local usado na otimizao (path de ClientLocalFile)
   */
  public String getLocalPath() {
    if (clientLocalFile != null) {
      return clientLocalFile.getStringPath();
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return clientProjectFile.getType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory() {
    return clientProjectFile.isDirectory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    if (!isOptimizedMode()) {
      canReadLocalFile = false;
      return clientProjectFile.canRead();
    }
    try {
      canReadLocalFile = true;
      return clientLocalFile.canRead();
    }
    catch (Exception e) {
      canReadLocalFile = false;
      return clientProjectFile.canRead();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    if (!isOptimizedMode()) {
      return clientProjectFile.canWrite();
    }
    try {
      return clientLocalFile.canWrite();
    }
    catch (Exception e) {
      return clientProjectFile.canWrite();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    if (!isOptimizedMode()) {
      return clientProjectFile.canExecute();
    }
    try {
      return clientLocalFile.canExecute();
    }
    catch (Exception e) {
      return clientProjectFile.canExecute();
    }
  }

  /**
   * Faz um teste, tentando abrir o arquivo. Se for um diretrio, verifica se
   * consegue executar <code>canRead()</code> do arquivo local
   * 
   * 
   * @return true, se conseguir abrir o arquivo utilizando otimizao. False,
   *         caso contrrio.
   * @throws RemoteException Em caso de erro no acesso a arquivo de projeto
   */
  public boolean testOptimization() throws RemoteException {
    boolean ret = false;
    if (isDirectory()) {
      canRead();
      return canReadLocalFile;
    }
    try {
      open(true);
      /* Verifica se openedFile foi atribuido a clientLocalFile */
      if (openedFile.getClientFileType() == ClientFileType.LOCAL) {
        ret = true;
      }
      close(true);
    }
    catch (Exception e) {
      System.out.println("Ocorreu erro no teste: " + e.getMessage());
      if (e instanceof RemoteException) {
        throw (RemoteException) e;
      }
    }
    return ret;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) throws Exception {
    if (!isOptimizedMode()) {
      clientProjectFile.open(readOnly);
      openedFile = clientProjectFile;
      return;
    }
    try {
      clientLocalFile.open(readOnly);
      openedFile = clientLocalFile;
    }
    catch (Exception e) {
      clientProjectFile.open(readOnly);
      openedFile = clientProjectFile;
    }

  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    return openedFile.read(dst, position);
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len, long position)
    throws Exception {
    return openedFile.read(dst, off, len, position);
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    if (openedFile == null) {
      return;
    }
    openedFile.close(force);
    openedFile = null;
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public long size() {
    if (!isOptimizedMode()) {
      return clientProjectFile.size();
    }
    try {
      return clientLocalFile.size();
    }
    catch (Exception e) {
      return clientProjectFile.size();
    }
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public ClientFileType getClientFileType() {
    return clientProjectFile.getClientFileType();
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    openedFile.write(src, off, len, position);
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position)
    throws IOException, FileLockedException {
    openedFile.write(src, position);
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public long position() throws IOException {
    return openedFile.position();
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    openedFile.position(newPosition);
  }

  /**
   * 
   * @return O modo de otimizao que est sendo utilizado neste arquivo
   */
  public ClientOptimizationMode getOptimizationMode() {
    return optimizationMode;
  }

}
