/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms.parameters;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Toolkit;
import java.awt.Window;
import java.rmi.RemoteException;
import java.text.MessageFormat;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;

import csbase.client.algorithms.validation.ValidationTranslator;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopFrame;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.SimpleParameterListener;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;

/**
 * Viso para {@link SimpleParameter Parmetro Simples}.
 *
 * @param <V> O tipo do parmetro.
 * @author lmoreira
 */
public abstract class SimpleParameterView <V> extends
  ParameterView<SimpleParameter<V>> {

  /**
   * Opes de alinhamento dos componentes.
   */
  private enum ComponentAlignment {
    /**
     * Rtulo em cima do componente
     */
    VERTICAL,
    /**
     * Rtulo na mesma linha do componente, antes do componente
     */
    HORIZONTAL_LABEL_FIRST,
    /**
     * Rtulo na mesma linha do componente, depois do componente
     */
    HORIZONTAL_LABEL_LAST
  }

  /**
   * Alinhamento padro.
   */
  private static final ComponentAlignment DEFAULT_ALIGNMENT =
    ComponentAlignment.HORIZONTAL_LABEL_FIRST;

  /**
   * O rtulo.
   */
  private JLabel label;

  /**
   * O componente principal da viso.
   */
  private JComponent component;

  /**
   * Cria a viso.
   *
   * @param simpleParameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   * valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   * {@link ParameterView.Mode#REPORT}
   * @param componentArgs argumentos que sero passados aos mtodos de
   * construo do componente representando a viso, tanto em modo
   * {@link ParameterView.Mode#CONFIGURATION} quanto em modo
   * {@link ParameterView.Mode#REPORT}.
   */
  protected SimpleParameterView(SimpleParameter<V> simpleParameter, Mode mode,
    Object... componentArgs) {
    super(simpleParameter, mode);
    this.component = createComponent(componentArgs);

    getParameter().addSimpleParameterListener(new SimpleParameterListener<V>() {
      @Override
      public void capabilityWasChanged(SimpleParameter<V> parameter) {
        updateCapabilityView();
      }

      @Override
      public void defaultValueWasChanged(SimpleParameter<V> parameter) {
        // Ignora o evento.
      }

      @Override
      public void valueWasChanged(SimpleParameter<V> parameter) {
        updateViewContents();
      }

      @Override
      public void visibilityWasChanged(SimpleParameter<V> parameter) {
        if (getMode() == Mode.CONFIGURATION) {
          updateVisibilyView();
        }
      }

      @Override
      public void labelWasChanged(SimpleParameter<V> parameter) {
        updateLabel();

      }
    });
    this.label = new JLabel();
    updateLabel();
  }

  /**
   * Atualiza a label do parmetro.
   */
  private void updateLabel() {
    String paramLabel = getParameter().getLabel();
    label.setText(paramLabel);

    // Adiciona indicativo (*) se parmetro  obrigatrio
    if (!getParameter().isOptional() && !paramLabel.trim().isEmpty()) {
      this.label.setHorizontalTextPosition(SwingConstants.LEFT);
      this.label.setIcon(ApplicationImages.ICON_MANDATORY_7);
      this.label.setIconTextGap(1);
    }

    String name = getParameter().getName();
    String toolTipText = MessageFormat.format("{0} ({1})", paramLabel, name);
    this.label.setToolTipText(toolTipText);
  }

  /**
   * Obtm os componentes da viso de acordo com o alinhamento escolhido.
   *
   * @return Os componentes.
   */
  public final JComponent[] getComponents() {
    ComponentAlignment alignment = getComponentAlignment();
    switch (alignment) {
      case VERTICAL:
        JPanel verticalPanel = new JPanel(new BorderLayout(0, 5));
        verticalPanel.add(getLabel(), BorderLayout.NORTH);
        verticalPanel.add(getComponent(), BorderLayout.CENTER);
        return new JComponent[] { verticalPanel };
      case HORIZONTAL_LABEL_LAST:
        JPanel panel = new JPanel(new BorderLayout());
        panel.add(getComponent(), BorderLayout.WEST);
        panel.add(getLabel(), BorderLayout.CENTER);
        return new JComponent[] { panel };
      case HORIZONTAL_LABEL_FIRST:
      default:
        return new JComponent[] { this.label, getComponent() };
    }
  }

  /**
   * Obtm o alinhamento do componente do parmetro no Configuration Manager.
   *
   * @return o alinhamento.
   */
  protected final ComponentAlignment getComponentAlignment() {
    try {
      ConfigurationManager manager = ConfigurationManager.getInstance();
      Configuration configuration =
        manager.getConfiguration(SimpleParameterView.class);
      String propName = getClass().getName();
      return configuration
        .getOptionalEnumerationProperty(propName, ComponentAlignment.class,
          DEFAULT_ALIGNMENT);
    }
    catch (ConfigurationManagerException e) {
      e.printStackTrace();
      return DEFAULT_ALIGNMENT;
    }
  }

  /**
   * Obtm o rtulo da viso.
   *
   * @return .
   */
  public final JLabel getLabel() {
    return this.label;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    if (!result.isWellSucceded()) {
      String message = ValidationTranslator.translateMessage(result);
      showErrorMessage(message);
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ViewValidationResult validate(
    ValidationMode mode) throws RemoteException {
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    SimpleParameter<V> parameter = getParameter();

    Object projectId = (project == null) ? null : project.getId();
    Object userId = User.getLoggedUser().getId();
    ValidationContext context = new ValidationContext(mode, projectId, userId);

    Validation validate = parameter.validate(context);
    ViewValidationResult result = new ViewValidationResult(validate, this);
    return result;
  }

  /**
   * Limpa a indicao de erro.
   */
  public final void clearError() {
    this.label.setForeground(Color.BLACK);
  }

  /**
   * @return a janela que ter o componente que representa esta viso.
   */
  public Window getWindow() {
    JComponent paramComponent = getComponent();
    return null == paramComponent ? null : SwingUtilities
      .windowForComponent(paramComponent);
  }

  /**
   * Obtm o componente principal.
   *
   * @return .
   */
  public JComponent getComponent() {
    return component;
  }

  /**
   * Obtm o componente principal.
   *
   * @param componentArgs Os parmetros para criao do componente.
   * @return O componente principal.
   */
  protected abstract JComponent createConfigurationComponent(
    Object... componentArgs);

  /**
   * Obtm o componente principal.
   *
   * @param componentArgs Os parmetros para criao do componente.
   * @return O componente principal.
   */
  protected abstract JComponent createReportComponent(Object... componentArgs);

  /**
   * Marca o erro de validao.
   */
  public final void highlightError() {
    this.label.setForeground(Color.RED);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final boolean isVisible() {
    return this.label.isVisible();
  }

  /**
   * Habilita/Desabilita a viso.
   *
   * @param isEnabled .
   */
  protected void setEnabled(boolean isEnabled) {
    if (null != component) {
      component.setEnabled(isEnabled);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final void setVisible(boolean isVisible) {
    this.label.setVisible(isVisible);
    if (null != getComponent()) {
      getComponent().setVisible(isVisible);
    }
  }

  /**
   * Exibe um erro.
   *
   * @param errorMessage A mensagem de erro (No aceita {@code null}).
   */
  protected final void showErrorMessage(String errorMessage) {
    highlightError();
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(getWindow(), errorMessage,
      "Erro no configurador de parmetros", JOptionPane.ERROR_MESSAGE);
    getComponent().requestFocus();
    clearError();
  }

  /**
   * Atualiza a viso em relao ao estado Habilitado/Desabilitado.
   */
  protected final void updateCapabilityView() {
    this.label.setEnabled(getParameter().isEnabled());
    setEnabled(getParameter().isEnabled());
  }

  /**
   * Atualiza o contedo exibido pela viso.
   */
  protected abstract void updateViewContents();

  /**
   * Cria o componente principal.
   *
   * @param componentArgs parmetros para criao dos componentes.
   * @return O componente principal.
   */
  private JComponent createComponent(Object... componentArgs) {
    switch (getMode()) {
      case REPORT: {
        return createReportComponent(componentArgs);
      }
      default: {
        return createConfigurationComponent(componentArgs);
      }
    }
  }
}
