package csbase.client.algorithms.parameters;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;

import csbase.client.algorithms.validation.TableCellValidationResult;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.desktop.DesktopFrame;
import csbase.client.util.table.AbstractCellModel;
import csbase.client.util.table.AbstractRowModel;
import csbase.client.util.table.CellModel;
import csbase.client.util.table.CellModelFactory;
import csbase.client.util.table.CellViewFactory;
import csbase.client.util.table.RowValueFactory;
import csbase.client.util.table.Table;
import csbase.exception.BugException;
import csbase.exception.ConfigurationException;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.algorithms.parameters.RowValue;
import csbase.logic.algorithms.parameters.RowValueListener;
import csbase.logic.algorithms.parameters.TableColumn;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.TableParameterListener;
import csbase.logic.algorithms.parameters.validators.TableParameterValidator;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationMode;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.configurationmanager.MissingPropertyException;
import tecgraf.javautils.gui.GBC;

/**
 * Viso para o {@link TableParameter Parmetro do Tipo Tabela}.
 *
 * @author lmoreira
 */
public final class TableParameterView extends
  SimpleParameterView<List<RowValue>> {

  /**
   * Cria uma viso em modo {@link ParameterView.Mode#CONFIGURATION}.
   *
   * @param window NO EST SENDO UTILIZADO. Existe somente para manter
   *        compatibilidade com o WebSintesi.
   * @param parameter O parmetro (No aceita {@code null}).
   *
   * @deprecated para manter compatibilidade com o WebSintesi
   */
  @Deprecated
  public TableParameterView(Window window, TableParameter parameter) {
    this(parameter, Mode.CONFIGURATION);
  }

  /**
   * Cria a viso.
   *
   * @param parameter O parmetro (No aceita {@code null}).
   * @param mode Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public TableParameterView(TableParameter parameter, Mode mode) {
    super(parameter, mode);

    updateCapabilityView();
    updateVisibilyView();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableParameter getParameter() {
    return (TableParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final ViewValidationResult validate(ValidationMode mode)
    throws RemoteException {
    TableParameter parameter = getParameter();
    TableParameterValidator parameterValidator =
      (TableParameterValidator) parameter.getParameterValidator();
    CommonClientProject project = DesktopFrame.getInstance().getProject();

    Object projectId = project.getId();
    Object userId = User.getLoggedUser().getId();
    ValidationContext context = new ValidationContext(mode, projectId, userId);

    Validation result = parameter.validate(context);
    if (!result.isWellSucceded()) {
      return new TableCellValidationResult(result, this, parameterValidator
        .getColumnIndex(), parameterValidator.getRowIndex());
    }
    else {
      return new ViewValidationResult(result, this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean highlightValidationResult(ViewValidationResult result) {
    if (!result.isWellSucceded() && isVisible()) {
      if (result instanceof TableCellValidationResult) {
        TableCellValidationResult cellValidation =
          (TableCellValidationResult) result;
        ITableParameterComponent tableComponent =
          (ITableParameterComponent) getComponent();
        if (tableComponent != null && tableComponent.getRowCount() != 0) {
          tableComponent.selectCell(cellValidation.getRowIndex(), cellValidation
            .getColumnIndex());
        }
      }
    }
    return super.highlightValidationResult(result);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final JComponent createConfigurationComponent(
    Object... componentArgs) {
    return new TableConfigurationParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final JComponent createReportComponent(Object... componentArgs) {
    return new TableReportParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final void updateViewContents() {
    // No faz nada.
  }

  /**
   * Cria a fbrica de {@link TableColumnView}.
   *
   * @return A fbrica.
   */
  private TableColumnViewFactory createTableColumnViewFactory() {
    TableColumnViewFactory factory;
    String key = "tableColumnViewFactory";
    Configuration configuration;
    try {
      configuration = ConfigurationManager.getInstance().getConfiguration(
        TableParameterView.class);
    }
    catch (ConfigurationManagerException e) {
      String message = String.format(
        "Erro ao consultar a configurao da classe %s no ConfigurationManager.",
        TableParameterView.class.getName());
      throw new ConfigurationException(message);
    }
    Class<?> clazz;
    try {
      clazz = configuration.getMandatoryClassProperty(key);
    }
    catch (MissingPropertyException e) {
      String message = String.format(
        "Erro ao consultar a configurao da classe %s no ConfigurationManager.",
        TableParameterView.class.getName());
      throw new ConfigurationException(e, message);
    }
    catch (ClassNotFoundException e) {
      String message = String.format(
        "A classe referenciada pela chave %s da configurao da classe %s"
          + "no ConfigurationManager no foi encontrada no classpath do sistema.",
        key, TableParameterView.class.getName());
      throw new ConfigurationException(e, message);
    }
    if (!TableColumnViewFactory.class.isAssignableFrom(clazz)) {
      String message = String.format(
        "Erro ao criar a fbrica das vises das colunas. "
          + "A classe %s no  uma fbrica de vises de coluna (%s). "
          + "Corrija a configurao da classe %s no ConfigurationManager "
          + "incluindo um tipo adequado.", clazz.getName(),
        TableColumnViewFactory.class.getName(), TableParameterView.class
          .getName());
      throw new ConfigurationException(message);
    }
    try {
      factory = (TableColumnViewFactory) clazz.newInstance();
    }
    catch (InstantiationException e) {
      String message =
        "Ocorreu uma exceuo ao criar a fbrica das vises das colunas.";
      throw new BugException(message, e);
    }
    catch (IllegalAccessException e) {
      String message = String.format("O construtor da classe %s no  pblico.",
        clazz.getName());
      ;
      throw new BugException(message, e);
    }
    return factory;
  }

  /**
   * Interface que o componente que representa a viso da tabela deve
   * implementar.
   */
  private interface ITableParameterComponent {
    /**
     * Atualiza o contedo exibido pela viso.
     */
    void updateViewContents();

    /**
     * @return A quantidade de linhas da tabela.
     */
    int getRowCount();

    /**
     * Seleciona uma clula especfica, removendo as seleo das outras clulas.
     *
     * @param rowIndex O ndice da linha (O ndice da linha tem que ser maior ou
     *        igual a 0 e menor do que a quantidade de linhas).
     * @param columnIndex O ndice da coluna (O ndice da coluna tem que ser
     *        maior ou igual a 0 e menor do que a quantidade de colunas).
     */
    void selectCell(int rowIndex, int columnIndex);
  }

  /**
   * Viso da tabela em modo relatrio.
   */
  private final class TableReportParameter extends JPanel implements
    ITableParameterComponent {

    /**
     * A tabela.
     */
    private JTable table;

    /**
     * Construtor.
     */
    TableReportParameter() {

      setLayout(new GridLayout());

      this.table = new JTable(new AbstractTableModel() {
        @Override
        public int getColumnCount() {
          return getParameter().getColumnCount();
        }

        @Override
        public String getColumnName(int column) {
          return getParameter().getColumns().get(column).getLabel();
        }

        @Override
        public int getRowCount() {
          return getParameter().getRowCount();
        }

        @Override
        public Class<?> getColumnClass(int c) {
          return getValueAt(0, c).getClass();
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
          return getParameter().getItemValue(rowIndex, columnIndex);
        }
      });
      ComponentProperties.setProperties(table, Mode.REPORT, false);

      JScrollPane scroll = new JScrollPane(table);
      scroll.setBorder(ComponentProperties.getInstance(Mode.REPORT)
        .getBorder());
      scroll.setPreferredSize(new Dimension(400, 100));
      add(scroll, new GBC(0, 0).both());
      updateViewContents();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      ((AbstractTableModel) table.getModel()).fireTableDataChanged();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void selectCell(int rowIndex, int columnIndex) {
      table.changeSelection(rowIndex, columnIndex, false, false);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return table.getRowCount();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean enabled) {
      // Este mtodo no faz nada, pois este componente no pode ser habilitado.
    }
  }

  /**
   * Viso da tabela em modo de configurao.
   */
  private final class TableConfigurationParameter extends JPanel implements
    ITableParameterComponent {

    /**
     * Tabela.
     */
    private Table<RowValue> table;

    /**
     * Construtor.
     */
    TableConfigurationParameter() {
      setLayout(new GridBagLayout());

      List<csbase.client.util.table.TableColumn<RowValue>> tableColumns =
        createTableColumns();
      AbstractRowModel<RowValue> rowModel = createRowModel();
      this.table = createTable(rowModel, tableColumns);
      add(table.getView(), new GBC(0, 0).both());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updateViewContents() {
      // No faz nada.
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return table.getRowCount();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void selectCell(int rowIndex, int columnIndex) {
      table.selectCell(rowIndex, columnIndex);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setEnabled(boolean isEnabled) {
      super.setEnabled(isEnabled);
      table.setEnabled(isEnabled);
    }

    /**
     * Cria colunas da tabela.
     *
     * @return A lista com as colunas criadas.
     */
    private List<csbase.client.util.table.TableColumn<RowValue>> createTableColumns() {
      List<csbase.client.util.table.TableColumn<RowValue>> tableColumns =
        new ArrayList<csbase.client.util.table.TableColumn<RowValue>>(
          getParameter().getColumnCount());
      List<TableColumn<?>> columns = getParameter().getColumns();
      for (int i = 0; i < getParameter().getColumnCount(); i++) {
        TableColumn<?> column = columns.get(i);
        CellViewFactory cellViewFactory = createCellViewFactory(column);
        final int columnIndex = i;
        CellModelFactory<RowValue> cellModelFactory =
          new CellModelFactory<RowValue>() {
            @Override
            public CellModel create(final RowValue rowValue) {
              final AbstractCellModel cellModel = new AbstractCellModel() {

                @Override
                protected void keepValue(Object value) {
                  rowValue.setCellValue(value, columnIndex);
                }

                @Override
                public Object getValue() {
                  return rowValue.getCellValue(columnIndex);
                }
              };
              RowValueListener listener = new RowValueListener() {

                @Override
                public void valueWasChanged(Object newValue, int aColumnIndex) {
                  if (aColumnIndex == columnIndex) {
                    cellModel.setValue(newValue);
                  }
                }

              };
              rowValue.addRowValueListener(listener);
              return cellModel;
            }
          };
        csbase.client.util.table.TableColumn<RowValue> tableColumn =
          new csbase.client.util.table.TableColumn<RowValue>(column.getLabel(),
            column.isEditable(), cellModelFactory, cellViewFactory);
        tableColumns.add(tableColumn);
      }

      return tableColumns;
    }

    /**
     * Cria o modelo das linhas da tabela.
     *
     * @return O modelo.
     */
    private AbstractRowModel<RowValue> createRowModel() {
      final AbstractRowModel<RowValue> rowModel =
        new AbstractRowModel<RowValue>() {

          @Override
          public void addRow(RowValue rowValue) {
            getParameter().addRow(rowValue);
            fireRowWasAdded(rowValue, getParameter().getRowCount() - 1);
          }

          @Override
          public RowValue getRow(int rowIndex) {
            return getParameter().getRowValues().get(rowIndex);
          }

          @Override
          public int getRowCount() {
            return getParameter().getRowCount();
          }

          @Override
          public List<RowValue> getRows() {
            return getParameter().getRowValues();
          }

          @Override
          public boolean removeRow(int rowIndex) {
            if (rowIndex < 0 || rowIndex >= getRowCount()) {
              return false;
            }
            getParameter().removeRow(rowIndex);
            return true;
          }
        };

      getParameter().addTableParameterListener(new TableParameterListener() {
        @Override
        public void rowWasCreated(TableParameter parameter,
          RowValue newRowValue, int newRowIndex) {
          rowModel.fireRowWasAdded(newRowValue, newRowIndex);
        }

        @Override
        public void rowWasRemoved(TableParameter parameter,
          RowValue oldRowValue, int oldRowIndex) {
          rowModel.fireRowWasRemoved(oldRowValue, oldRowIndex);
        }
      });

      return rowModel;
    }

    /**
     * Cria a tabela.
     *
     * @param rowModel O modelo das linhas.
     * @param tableColumns As colunas da tabela.
     * @return A tabela.
     */
    private Table<RowValue> createTable(AbstractRowModel<RowValue> rowModel,
      List<csbase.client.util.table.TableColumn<RowValue>> tableColumns) {

      Table<RowValue> newTable;
      final TableParameter parameter = getParameter();
      if (parameter.hasFixedRowCount()) {
        newTable = new Table<RowValue>(rowModel, tableColumns);
      }
      else {
        final RowValueFactory<RowValue> rowValueFactory =
          new RowValueFactory<RowValue>() {
            @Override
            public RowValue create(int rowIndex) {
              Object[] values = new Object[parameter.getColumnCount()];
              for (int i = 0; i < parameter.getColumnCount(); i++) {
                TableColumn<?> column = parameter.getColumns().get(i);
                values[i] = column.getDefaultValue(rowIndex);
              }
              return new RowValue(values);
            }
          };
        final Integer visRowCount = parameter.getVisibleRowCount();
        final Integer minRowCount = parameter.getMinRowCount();
        final Integer maxRowCount = parameter.getMaxRowCount();
        newTable = new Table<RowValue>(rowValueFactory, rowModel, tableColumns,
          visRowCount, minRowCount, maxRowCount);
      }
      newTable.setToolTipText(parameter.getDescription());

      return newTable;
    }

    /**
     * Cria a fbrica de vises de clulas para a coluna informada.
     *
     * @param column A coluna.
     *
     * @return .
     */
    private CellViewFactory createCellViewFactory(TableColumn<?> column) {
      TableColumnView<?, ?> columnView = createTableColumnViewFactory().create(
        TableParameterView.this, column);
      if (columnView == null) {
        String message = String.format(
          "O tipo da coluna %s (%s) no  suportado.", column.getClass()
            .getName(), column.getId());
        throw new ConfigurationException(message);
      }
      return columnView.getFactory();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return true;
  }
}
