/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.table;

import java.awt.Component;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.script.Bindings;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.algorithms.commands.cache.CommandsFilter;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.commandsmonitor.dal.ICommandsTableDAO;
import csbase.client.applications.commandsmonitor.models.ColumnDTO;
import csbase.client.applications.commandsmonitor.models.CommandsFilterDTO;
import csbase.client.applications.commandsmonitor.models.CommandsTableDTO;
import csbase.client.applications.commandsmonitor.models.ExpressionDTO;
import csbase.client.applications.commandsmonitor.table.column.AbstractCommandsTableColumn;
import csbase.client.applications.commandsmonitor.table.script.ScriptCommand;
import csbase.client.facilities.commandtable.AbstractCommandMonitoringTable;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Fbrica de tabelas de monitorao de comandos.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class CommandsTableFactory {

  /**
   * Registro da aplicao, utilizado para se obter textos internacionalizados.
   */
  private ApplicationRegistry applicationRegistry;
  /**
   * Mapeia o identificador de uma coluna as informaes necessrias para a
   * criao da mesma.
   */
  private Map<String, ColumnDTO> columnsInfoById;
  /**
   * Mapeia as informaes necessrias para a criao de uma tabela ao
   * identificador da mesma.
   */
  private Map<String, CommandsTableDTO> tablesInfoById;
  /**
   * Mapeia as informaes necessrias para a criao de um filtro de comando ao
   * identificador do mesmo.
   */
  private Map<String, CommandsFilter> filtersById;

  /**
   * Construtor.
   * 
   * @param applicationRegistry registro da aplicao, utilizado para se obter
   *        textos internacionalizados.
   * @param dataSource Fonte de dados para coletar informaes sobre as
   *        possveis tabelas.
   */
  public CommandsTableFactory(ApplicationRegistry applicationRegistry,
    ICommandsTableDAO dataSource) {

    this.applicationRegistry = applicationRegistry;

    this.columnsInfoById = new HashMap<String, ColumnDTO>();
    this.tablesInfoById = new HashMap<String, CommandsTableDTO>();
    this.filtersById = new HashMap<String, CommandsFilter>();

    Set<ColumnDTO> columnsInfo = dataSource.getColumnsInfo();
    for (ColumnDTO aColumnInfo : columnsInfo) {
      columnsInfoById.put(aColumnInfo.getId(), aColumnInfo);
    }

    Set<CommandsTableDTO> tablesInfo = dataSource.getTablesInfo();
    for (CommandsTableDTO aTableInfo : tablesInfo) {
      tablesInfoById.put(aTableInfo.getId(), aTableInfo);
    }

    Set<CommandsFilterDTO> filtersInfo = dataSource.getFiltersInfo();
    for (CommandsFilterDTO aFilterInfo : filtersInfo) {
      CommandsFilter filter = createFilter(aFilterInfo);
      filtersById.put(aFilterInfo.getId(), filter);
    }
  }

  /**
   * Obtm um conjunto dos identificadores das tabelas que podem ser criadas.
   * 
   * @return um conjunto de identificadores das tabelas que podem ser criadas.
   */
  public Set<String> getTablesId() {
    return tablesInfoById.keySet();
  }

  /**
   * Cria uma instncia de {@link AbstractCommandMonitoringTable} dado seu
   * identificador.
   * 
   * @param id identificador nico de uma tabela de monitorao de comandos.
   * 
   * @return a instncia de {@link AbstractCommandMonitoringTable} referente ao
   *         identificador dado.
   * 
   * @throws ApplicationException Caso a classe que represente o modelo da
   *         coluna a ser criada seja desconhecida, no esteja no classpath ou
   *         no possua um construtor que receba apenas uma {@link String} e uma
   *         instncia de {@link ApplicationRegistry}.
   */
  public AbstractCommandMonitoringTable createTable(String id)
    throws ApplicationException {

    CommandsTableDTO tableInfo = tablesInfoById.get(id);
    if (null == tableInfo) {
      throw new IllegalArgumentException("Unknown table identifier. Table id: "
        + id);
    }
    CommandsFilter filter = filtersById.get(tableInfo.getFilterId());
    if (null == filter) {
      throw new IllegalArgumentException(
        "Unknown filter identifier. Table id: " + id + ", filter id: "
          + tableInfo.getFilterId());
    }

    int oderbyColumnIndex = -1;
    final List<AbstractCommandsTableColumn> columns =
      new ArrayList<AbstractCommandsTableColumn>();
    for (int index = 0; index < tableInfo.getColumnsId().size(); index++) {
      String columnId = tableInfo.getColumnsId().get(index);
      columns.add(createColumn(columnId));

      if (columnId.equals(tableInfo.getOrderby())) {
        oderbyColumnIndex = index;
      }
    }

    final ObjectTableProvider provider = new CommandsTableProvider(columns);

    AbstractCommandMonitoringTable table =
      new AbstractCommandMonitoringTable(
        ListSelectionModel.MULTIPLE_INTERVAL_SELECTION, filter) {

        @Override
        protected ObjectTableProvider getObjectTableProvider() {
          return provider;
        }

        @Override
        protected TableCellRenderer createTableCellRenderer(int columnIndex) {
          return columns.get(columnIndex).createTableCellRenderer();
        }

        @Override
        protected TableCellEditor createTableCellEditor(int columnIndex) {
          return columns.get(columnIndex).createTableCellEditor();
        }
      };

    /*
     * Altera o tamanho preferencial de cada coluna de acordo com o tamanho de
     * seu ttulo.
     */
    TableColumnModel columnsModel = table.getColumnModel();
    TableCellRenderer headerRenderer =
      table.getTableHeader().getDefaultRenderer();
    for (int inx = 0; inx < table.getColumnCount(); inx++) {
      String columnName = table.getColumnName(inx);
      Component header =
        headerRenderer.getTableCellRendererComponent(table, columnName, false,
          false, 0, inx);
      int width = header.getPreferredSize().width;
      TableColumn aColumnModel = columnsModel.getColumn(inx);
      aColumnModel.setPreferredWidth(width);
    }

    if (oderbyColumnIndex >= 0) {
      // Ordena a tabela
      SortOrder order =
        tableInfo.isAscending() ? SortOrder.ASCENDING : SortOrder.DESCENDING;
      table.sort(oderbyColumnIndex, order);
    }
    
    Long cellsEditingMinTime = tableInfo.getCellsEditingMinTime();
    if (cellsEditingMinTime != null) {      
      /*
       * Atribui o tempo mnimo que o usurio tem para editar uma clula da tabela 
       * sem que essa edio seja cancelada por uma atualizao da tabela.  
       */
      table.setCellsEditingMinTime(tableInfo.getCellsEditingMinTime());
    }
    
    return table;
  }

  /**
   * Cria uma instncia de {@link AbstractCommandsTableColumn} dado seu
   * identificador.
   * 
   * @param id identificador nico de uma coluna de tabela de comando.
   * 
   * @return a instncia de {@link AbstractCommandsTableColumn} referente ao
   *         identificador dado.
   * @throws ApplicationException Caso a classe que represente o modelo da
   *         coluna a ser criada seja desconhecida, no esteja no classpath ou
   *         no possua um construtor que receba apenas uma {@link String} e uma
   *         instncia de {@link ApplicationRegistry}.
   */
  public AbstractCommandsTableColumn createColumn(String id)
    throws ApplicationException {

    if (null == id) {
      throw new NullPointerException("id");
    }

    try {

      if (!columnsInfoById.containsKey(id)) {
        throw new ClassNotFoundException("Unknown classe for column " + id);
      }

      ColumnDTO info = columnsInfoById.get(id);
      Class<?> clazz = Class.forName(info.getColumnClassName());
      Constructor<?> constructor =
        clazz.getConstructor(String.class, ApplicationRegistry.class);

      AbstractCommandsTableColumn column =
        (AbstractCommandsTableColumn) constructor.newInstance(id,
          applicationRegistry);

      return column;
    }
    catch (Exception e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * Cira uma instncia de {@link CommandsFilter} dado um
   * {@link CommandsFilterDTO}
   * 
   * @param filterInfo informao de filtro.
   * 
   * @return uma instncia de um {@link CommandsFilter}.
   */
  private CommandsFilter createFilter(CommandsFilterDTO filterInfo) {

    boolean in = filterInfo.areStatusesIncluded();
    CommandStatus[] statuses = filterInfo.getStatuses();
    final ExpressionDTO expressionInfo = filterInfo.getExpressionInfo();

    if (expressionInfo == null) {
      return new CommandsFilter(in, statuses);
    }

    final CommandsFilter filter = new CommandsFilter(in, statuses) {
      ScriptEngineManager manager = new ScriptEngineManager();
      ScriptEngine engine = manager.getEngineByName(expressionInfo
        .getLanguage());

      @Override
      protected boolean acceptCommand(final CommandInfo cmd) {
        // Cria o escopo que ir carregar a instncia do comando para dentro do script.
        ScriptContext context = new SimpleScriptContext();
        Bindings engineScope = context.getBindings(ScriptContext.ENGINE_SCOPE);
        // Adiciona o comando ao novo escopo.        
        engineScope.put("command", new ScriptCommand(cmd));
        // Executa o script passando o contexto.
        try {
          return (Boolean) engine.eval(expressionInfo.getValue(), context);
        }
        catch (ScriptException e) {
          e.printStackTrace(); //!pasti logar em algum lugar?
          return false;
        }
      }
    };

    return filter;

  }
}
