/*
 * $Id: StarCanvas.java 157822 2014-10-30 13:25:28Z pietroguedes $
 */
package csbase.client.facilities.monitoring;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Polygon;

import javax.swing.JPanel;

import csbase.client.util.MonitoringTheme;

/**
 * Classe que representa o elemento star-graph
 * 
 * @author Tecgraf/PUC-Rio.
 */
public class StarCanvas extends JPanel {

  /**
   * Configuraes do grfico
   */
  private final int pixelsMargin = 30;

  /**
   * Configuraes do grfico
   */
  private boolean drawName = false;

  /**
   * Configuraes do grfico
   */
  private boolean drawPerc = false;

  /**
   * Configuraes do grfico
   */
  private boolean drawRef = false;

  /**
   * Configuraes do grfico
   */
  private boolean drawScale = true;

  /**
   * Configuraes do grfico
   */
  private boolean drawRadius = false;

  /**
   * Configuraes do grfico
   */
  private boolean drawRegions = true;

  /**
   * Configuraes do grfico
   */
  private int factor = 1;

  /**
   * Gerenciador das informaes visualizadas
   */
  private DynamicInfoHandler infoHandler = null;

  /**
   * Seta a referncia para o gerenciador de informaes.
   * 
   * @param infoHandler referncia para o gerenciador das informaes
   *        visualizadas
   */
  public void setInfoHandler(DynamicInfoHandler infoHandler) {
    this.infoHandler = infoHandler;
  }

  /**
   * Redesenha o componente
   * 
   * @param g .
   */
  @Override
  public void paintComponent(Graphics g) {
    super.paintComponent(g);
    drawGraph(g.create());
  }

  /**
   * Configura a visualizao dos nomes dos elementos.
   * 
   * @param state indica se os nomes devem ser visualizados (true) ou no
   *        (false)
   */
  public void setNameState(boolean state) {
    this.drawName = state;
    repaint();
  }

  /**
   * Configura a visualizao dos percentuais.
   * 
   * @param state indica se os percentuais devem ser visualizados (true) ou no
   *        (false)
   */
  public void setPercState(boolean state) {
    this.drawPerc = state;
    repaint();
  }

  /**
   * Configura a visualizao do eixo de referncia.
   * 
   * @param state indica se o eixo deve ser visualizado (true) ou no (false)
   */
  public void setRefState(boolean state) {
    this.drawRef = state;
    repaint();
  }

  /**
   * Configura a visualizao das escalas.
   * 
   * @param state indica se as escalas devem ser visualizadas (true) ou no
   *        (false)
   */
  public void setScaleState(boolean state) {
    this.drawScale = state;
    repaint();
  }

  /**
   * Configura a visualizao dos raios do polgono.
   * 
   * @param state indica se os raios devem ser visualizados (true) ou no
   *        (false)
   */
  public void setRadiusState(boolean state) {
    this.drawRadius = state;
    repaint();
  }

  /**
   * Configura a visualizao das regies do grfico.
   * 
   * @param state indica se as regies devem ser visualizadas (true) ou no
   *        (false)
   */
  public void setRegionsState(boolean state) {
    this.drawRegions = state;
    repaint();
  }

  /**
   * Configura o fator de zoom.
   * 
   * @param factor valor do fator
   */
  public void setFactor(int factor) {
    this.factor = factor;
    repaint();
  }

  /**
   * Desenha os eixos do grfico
   * 
   * @param graphics .
   * @param cx .
   * @param cy .
   * @param radius .
   */
  private void drawAxis(Graphics graphics, int cx, int cy, int radius) {
    int height = getHeight();
    int y = height - (pixelsMargin);
    graphics.setColor(MonitoringTheme.getAxisColor());
    int minx = cx - (radius / 2);
    int maxx = cx + (radius / 2);
    graphics.drawLine(minx, y, maxx, y);
    graphics.drawLine(minx, y - 1, maxx, y - 1);
    int value = 100;
    boolean flag = false;
    for (int r = radius; r > 0; r = r - (radius / 10)) {
      int x1 = cx - (r / 2);
      int x2 = cx + (r / 2);
      if (value != 0) {
        graphics.drawLine(x1, y - 2, x1, y + 2);
        graphics.drawLine(x2, y - 2, x2, y + 2);
      }
      if (flag) {
        String str = value + "";
        graphics.drawString(str, x1 - 5, y + 13);
        graphics.drawString(str, x2 - 5, y + 13);
      }
      flag = !flag;
      value = value - 10;
    }
    final int asize = (int) (pixelsMargin / 2.5);
    graphics.fillPolygon(new Polygon(
      new int[] { minx, minx, minx - asize, minx }, new int[] { y,
          y - (asize / 2), y, y + (asize / 2) }, 4));
    graphics.fillPolygon(new Polygon(
      new int[] { maxx, maxx, maxx + asize, maxx }, new int[] { y,
          y - (asize / 2), y, y + (asize / 2) }, 4));
  }

  /**
   * Desenha as escalas.
   * 
   * @param graphics .
   * @param cx .
   * @param cy .
   * @param radius .
   */
  private void drawScales(Graphics graphics, int cx, int cy, int radius) {
    graphics.setColor(MonitoringTheme.getGridColor());
    if (drawScale) {
      for (int r = radius; r >= 0; r = r - (radius / 10)) {
        graphics.drawArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
      }
    }
    else {
      int r = radius;
      graphics.drawArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
    }
  }

  /**
   * Exibe as informaes.
   * 
   * @param graphics .
   * @param cx .
   * @param cy .
   * @param radius .
   */
  private void drawData(Graphics graphics, int cx, int cy, int radius) {
    int numNodes = infoHandler.getValidSize();
    int angle = 360 / numNodes;
    int[] xp = new int[numNodes];
    int[] yp = new int[numNodes];
    int nextNode = -1;
    for (int i = 0; i < numNodes; i++) {
      nextNode++;
      double a = i * angle;
      while (!infoHandler.isValid(nextNode)) {
        nextNode++;
      }
      double load = infoHandler.getCurrentValue(nextNode);
      if (load > 100.0) {
        load = 100.0;
      }
      int r = (int) ((load / 100 * radius) / 2);
      r = (r > radius) ? radius : r;
      xp[i] = cx + (int) (Math.cos(Math.toRadians(a)) * r);
      yp[i] = cy + (int) (Math.sin(Math.toRadians(a)) * r);
    }
    graphics.setColor(MonitoringTheme.getProcessorColor());
    graphics.fillPolygon(xp, yp, numNodes);
    graphics.setColor(MonitoringTheme.getAxisColor());
    graphics.drawPolygon(xp, yp, numNodes);
    /* desenho dos textos por cima do polgono */
    graphics.setColor(Color.BLACK);
    nextNode = -1;
    for (int i = 0; i < numNodes; i++) {
      nextNode++;
      String txt = "";
      while (!infoHandler.isValid(nextNode)) {
        nextNode++;
      }
      int load = (int) infoHandler.getCurrentValue(nextNode);
      if (load > 100) {
        load = 100;
      }
      if (drawName) {
        txt = txt + infoHandler.getName(nextNode) + " ";
      }
      if (drawPerc) {
        txt = txt + "(" + load + "%" + ")";
      }
      graphics.drawString(txt, xp[i], yp[i]);
    }
  }

  /**
   * Desenha o polgono.
   * 
   * @param graphics .
   * @param cx .
   * @param cy .
   * @param radius .
   */
  private void drawStar(Graphics graphics, int cx, int cy, int radius) {
    if (drawRadius) {
      graphics.setColor(MonitoringTheme.getGridColor());
      for (int a = 0; a < 360; a = a + (360 / infoHandler.getValidSize())) {
        int x = cx + (int) (Math.cos(Math.toRadians(a)) * radius * 0.55);
        int y = cy + (int) (Math.sin(Math.toRadians(a)) * radius * 0.55);
        graphics.drawLine(cx, cy, x, y);
      }
    }
  }

  /**
   * Colore o grfico.
   * 
   * @param graphics .
   * @param cx .
   * @param cy .
   * @param radius .
   */
  private void drawForeground(Graphics graphics, int cx, int cy, int radius) {
    if (drawRegions) {
      int r = radius;
      graphics.setColor(MonitoringTheme.getRangeAlertColor());
      graphics.fillArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
      r = (int) (radius * 0.75);
      graphics.setColor(MonitoringTheme.getRangeWarningColor());
      graphics.fillArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
      r = (int) (radius * 0.5);
      graphics.setColor(MonitoringTheme.getRangeNormalColor());
      graphics.fillArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
    }
    else {
      int r = radius;
      graphics.setColor(MonitoringTheme.getRangeDefaultColor());
      graphics.fillArc(cx - (r / 2), cy - (r / 2), r, r, 0, 360);
    }
  }

  /**
   * Tira clipping
   * 
   * @param graphics graphics
   */
  private void unClip(Graphics graphics) {
    int width = getWidth();
    int height = getHeight();
    graphics.setClip(0, 0, width, height);
  }

  /**
   * Tira clipping
   * 
   * @param graphics graphics
   */
  private void clip(Graphics graphics) {
    int width = getWidth();
    int height = getHeight();
    graphics.setClip(0, 0, width, height - pixelsMargin);
  }

  /**
   * Desenha o grfico com as informaes correntes.
   * 
   * @param graphics graphics.
   */
  private void drawGraph(Graphics graphics) {
    if (infoHandler == null) {
      return;
    }
    if (infoHandler.getValidSize() < 1) {
      return;
    }
    int width = getWidth();
    int height = getHeight();
    graphics.setColor(Color.WHITE);
    graphics.fillRect(0, 0, width, height);
    int size = ((width < height) ? width : height) - pixelsMargin;
    int cx = width / 2;
    int cy = height / 2;
    int radius = (int) (size * 0.9) * factor;
    if (drawRef) {
      clip(graphics);
    }
    drawForeground(graphics, cx, cy, radius);
    drawStar(graphics, cx, cy, radius);
    drawData(graphics, cx, cy, radius);
    drawScales(graphics, cx, cy, radius);
    if (drawRef) {
      unClip(graphics);
      drawAxis(graphics, cx, cy, radius);
    }
  }

  /**
   * Constri o grfico estrela (radar)
   */
  public StarCanvas() {
    super();
    Dimension size = new Dimension(500, 400);
    setMinimumSize(size);
    setPreferredSize(size);
  }
}
