package csbase.client.preferences;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.editors.EmptyEditor;
import csbase.client.preferences.util.PreferenceBundle;
import csbase.client.preferences.util.PreferenceListener;

/**
 * Classe abstrata que define o tipo do valor de uma preferncia do usurio.
 * 
 * Um valor de preferncia encapsula um valor de fato ({@link String},
 * {@link Boolean}, {@link Map} etc), o valor default e a poltica de
 * visibilidade deste valor.
 * 
 *  importante frisar que o conceito de valor de preferncia est separado do
 * conceito de edio do valor de preferncia.  possvel que um valor seja
 * editvel atravs de vrias interface grficas diferentes. Para isso, cada
 * valor de preferncia possui uma referncia para o seu editor, sendo este,
 * podendo ser alterado.
 * 
 * Mais sobre a edio de valores de preferncias {@link PreferenceEditor}.<br/>
 * Mais sobre a poltica de visibilidade {@link PreferencePolicy}.
 * 
 * @see PreferenceCategory
 * 
 * @param <T> tipo do valor armazenado.
 * 
 * @author Tecgraf
 */
public abstract class PreferenceValue<T> implements Cloneable {

  /** Nome da preferncia que possui este valor. */
  protected PreferenceDefinition name;

  /** Valor da preferncia. */
  protected T value;

  /** String definida como o valor default da preferncia. */
  protected String defaultValue;

  /** Poltica de visibilidade da preferncia */
  protected PreferencePolicy policy;

  /** Objeto responsvel pela internacionalizao. */
  protected PreferenceBundle preferenceBundle;

  /** Classe do editor desta preferncia para incializao tardia. */
  protected Class<? extends PreferenceEditor<T>> preferenceEditorClass;

  /** Referncia para o editor desta preferncia. */
  protected PreferenceEditor<T> editor;

  /** Lista com os ouvintes desta preferncia. */
  protected List<PreferenceListener<T>> listeners;

  /**
   * Este construtor  usado para instanciar os valores por reflexo na hora em
   * que as descries de preferncias so carregadas. Neste momento, usamos
   * este construtor que recebe o valor como um {@link String} e o converte para
   * o tipo correto usando o mtodo {@link PreferenceValue#toValue(String)}.
   * 
   * Logo, todas as subclasses de {@link PreferenceValue} devem definir o mtodo
   * {@link PreferenceValue#toValue(String)} e definir um construtor com esta
   * mesma assinatura.
   * 
   * @param name nome da preferncia que possui este valor.
   * @param value valor da preferncia.
   * @param defaultValue valor default da preferncia.
   * @param policy poltica de visibilidade do valor.
   * @param preferenceBundle objeto responsvel pela internacionalizao.
   */
  public PreferenceValue(PreferenceDefinition name, String value,
    String defaultValue, PreferencePolicy policy,
    PreferenceBundle preferenceBundle) {
    if (name == null) {
      throw new IllegalArgumentException("name no pode ser nulo.");
    }
    if (value == null) {
      throw new IllegalArgumentException("value no pode ser nulo.");
    }
    if (defaultValue == null) {
      throw new IllegalArgumentException("defaultValue no pode ser nulo.");
    }
    if (policy == null) {
      throw new IllegalArgumentException("policy no pode ser nulo.");
    }
    if (preferenceBundle == null) {
      throw new IllegalArgumentException("preferenceBundle no pode ser nulo.");
    }

    this.name = name;
    this.value = toValue(value);
    this.defaultValue = defaultValue;
    this.policy = policy;
    this.preferenceBundle = preferenceBundle;

    this.listeners = new ArrayList<PreferenceListener<T>>();
  }

  /**
   * Nome da preferncia que possui este valor.
   * 
   * @return nome da preferncia que possui este valor.
   */
  public PreferenceDefinition getName() {
    return name;
  }

  /**
   * Rtulo da preferncia.
   * 
   * @return rtulo da preferncia.
   */
  public String getLabel() {
    return preferenceBundle.get(name.getClass().getSimpleName() + "." + name
      + ".label");
  }

  /**
   * Descrio da preferncia.
   * 
   * @return descrio da preferncia.
   */
  public String getDescription() {
    return preferenceBundle.get(name.getClass().getSimpleName() + "." + name
      + ".description");
  }

  /**
   * Retorna a poltica de visibilidade da preferncia.
   * 
   * @return poltica de visibilidade da preferncia.
   */
  public PreferencePolicy getPolicy() {
    return policy;
  }

  /**
   * Retorna o objeto responsvel pela internacionalizao.
   * 
   * @return objeto responsvel pela internacionalizao.
   */
  public PreferenceBundle getPreferenceBundle() {
    return preferenceBundle;
  }

  /**
   * Retorna o objeto que encapsula o componente visual que permite a edio do
   * valor.
   * 
   * @return objeto que encapsula o componente visual que permite a edio do
   *         valor.
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  public PreferenceEditor<T> getEditor() {
    if (editor != null) {
      return editor;
    }

    if (preferenceEditorClass == null) {
      editor = new EmptyEditor<T>(this);
    }
    else {
      try {
        Constructor<?> construtor =
          preferenceEditorClass.getConstructor(PreferenceValue.class);
        editor = (PreferenceEditor) construtor.newInstance(this);
      }
      catch (Exception e) {
        throw new PreferenceException(
          "Erro ao criar o objeto que encapsula o editor de uma preferncia.",
          e);
      }
    }

    return editor;
  }

  /**
   * Retorna a classe que implementa o editor desta preferncia.
   * 
   * @return classe que implementa o editor desta preferncia.
   */
  public Class<? extends PreferenceEditor<T>> getPreferenceEditorClass() {
    return preferenceEditorClass;
  }

  /**
   * Retorna o valor default da preferncia.
   * 
   * @return valor default da preferncia.
   */
  public T getDefaultValue() {
    return toValue(defaultValue);
  }

  /**
   * Retorna o valor da preferncia.
   * 
   * @return valor da preferncia.
   */
  public T getValue() {
    return value;
  }

  /**
   * Seta a classe do editor deste valor de preferncia.
   * 
   * @param preferenceEditorClass classe do editor desta preferncia.
   */
  public void setPreferenceEditorClass(
    Class<? extends PreferenceEditor<T>> preferenceEditorClass) {
    this.preferenceEditorClass = preferenceEditorClass;
  }

  /**
   * Seta o valor da preferncia.
   * 
   * @param value valor da preferncia.
   */
  public void setValue(T value) {
    if (value == null) {
      throw new IllegalArgumentException("value no pode ser nulo.");
    }
    T old = this.value;
    this.value = value;

    for (PreferenceListener<T> l : listeners) {
      l.valueChanged(old, value);
    }
  }

  /**
   * Adiciona ouvinte para esta preferncia.
   * 
   * @param l ouvinte.
   */
  public void addPreferenceListener(PreferenceListener<T> l) {
    this.listeners.add(l);
  }

  /**
   * Remove ouvinte desta preferncia.
   * 
   * @param l ouvinte.
   */
  public void removePreferenceListener(PreferenceListener<T> l) {
    this.listeners.remove(l);
  }

  /**
   * Retorna o nome da classe.
   * 
   * @return o nome da classe.
   */
  public abstract String getClassName();

  /**
   * Converte o valor do tipo {@link String} para o tipo correto. Por exemplo,
   * converte a string "123" para o inteiro 123, a string "true" para o valor
   * boleano true.
   * 
   * @param value valor em string.
   * @return valor no tipo correto.
   */
  public abstract T toValue(String value);

  /**
   * Converte o valor em um {@link String}.
   */
  @Override
  public abstract String toString();

  /**
   * Cria o clone desta preferncia.
   * 
   * @return clone do valor.
   */
  @Override
  public abstract PreferenceValue<T> clone();

  /**
   * Seta o valor da preferncia.
   * 
   * @param value valor da preferncia.
   */
  void setValueAsStr(String value) {
    T v = toValue(value);
    setValue(v);
  }

  /**
   * Seta o valor default da preferncia.
   * 
   * @param defaultValue valor default da preferncia.
   */
  void setDefaultValueAsStr(String defaultValue) {
    this.defaultValue = defaultValue;
  }
}
