package csbase.client.applications.algorithmsmanager.actions;

import javax.swing.ImageIcon;

import csbase.client.applications.algorithmsmanager.dialogs.CategoryCreateDataPanel;
import csbase.client.applications.algorithmsmanager.dialogs.CategorySelectionPanel;
import csbase.client.applications.algorithmsmanager.dialogs.CategoryTreeView;
import csbase.client.applications.algorithmsmanager.dialogs.CommonSelectionPanel;
import csbase.client.applications.algorithmsmanager.models.CategoryNodeInterface;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.Category;

/**
 * Essa classe representa a ao de criar uma nova categoria de algoritmos, na
 * funcionalidade de gerenciamento de categorias.
 * 
 * O mtodo <code> handleEditOperation </code> deve ser redefinido para que seja
 * criado um painel para a criao da nova categoria.
 * 
 */
public class CategoryCreateAction extends CommonEditAction {
  /**
   * Painel de categorias de algoritmos
   */
  private CategoryCreateDataPanel createPanel;

  /**
   * Categoria pai onde sero criadas as novas categorias
   */
  private CategoryNodeInterface parentCategory;

  /**
   * Constri a ao de criao de categorias de algoritmos.
   * 
   * @param selectionPanel painel de seleo de dados, onde essa ao foi
   *        originada
   * @param icon imagem da ao
   */
  public CategoryCreateAction(CommonSelectionPanel selectionPanel,
    ImageIcon icon) {
    super(selectionPanel, icon);
  }

  @Override
  protected void handleEditOperation() {
    createPanel = new CategoryCreateDataPanel(this);
    parentCategory = (CategoryNodeInterface) getSelectionPanel()
      .getSelectedData();
    getSelectionPanel().getManagementAction().setCreatePanel(createPanel);
    createPanel.initializeData();
  }

  /**
   * Obtm a categoria pai onde vai ser criada a nova categoria.
   * 
   * @return a categoria pai da nova categoria
   */
  public Category getParentCategory() {
    if (parentCategory != null) {
      return parentCategory.getNode();
    }
    return null;
  }

  /**
   * Obtm o identificador da categoria pai.
   * 
   * @return retorna o identificador da categoria pai, ou null, caso as novas
   *         categorias devam ser criadas na raiz (primeiro nvel)
   */
  protected String getParentCategoryId() {
    if (parentCategory != null) {
      return parentCategory.getId();
    }
    return null;
  }

  /**
   * Obtm o nome da categoria pai.
   * 
   * @return retorna o nome da categoria pai, ou null, caso as novas categorias
   *         devam ser criadas na raiz (primeiro nvel)
   */
  public String getParentCategoryName() {
    if (parentCategory == null) {
      return CategoryTreeView.ROOT_NAME;
    }
    return parentCategory.getFullName();
  }

  /**
   * Cria efetivamente a categoria no servidor.
   * 
   * @param categoryName nome da categoria a ser criada
   * 
   * @return a categoria efetivamente criada
   * 
   */
  public Category createCategory(String categoryName) {
    Category category = AlgorithmManagementProxy.createCategory(
      getParentCategoryId(), categoryName, getApplication()
        .getApplicationFrame());
    return category;
  }

  /**
   * Realiza o cancelamento da ao de criar uma categoria.
   * 
   */
  public void cancelCreation() {
    getSelectionPanel().getManagementAction().setCreatePanel(null);
  }

  @Override
  public CategorySelectionPanel getSelectionPanel() {
    return (CategorySelectionPanel) super.getSelectionPanel();
  }

}
