/*
 * $Id$
 */

package csbase.client.applications.algorithmsmanager.versiontree;

import java.awt.datatransfer.DataFlavor;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreePath;

import csbase.client.applications.algorithmsmanager.versiontree.datatransfer.ITransferableSource;
import csbase.client.applications.algorithmsmanager.versiontree.datatransfer.ITransferableTarget;
import csbase.client.applications.algorithmsmanager.versiontree.datatransfer.VersionTreeTransferHandler;
import csbase.client.project.ClientProjectFileType;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.FileInfo;
import csbase.logic.IPathFactory;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * N abstrato que representa um arquivo.
 * 
 * @author Tecgraf / PUC-Rio
 */
abstract class AbstractFileInfoNode extends AbstractVersionTreeNode implements
  ITransferableSource, ITransferableTarget, Serializable {

  /** Imagem padro utilizada para representar um arquivo. */
  private transient static final ImageIcon FILE_ICON;

  /** Imagem utilizada para representar um diretrio. */
  private transient static final ImageIcon DIRECTORY_ICON;

  /** Verso que detm o arquivo representado por este n. */
  private final AlgorithmVersionInfo version;

  /** Arquivo representado por este n. */
  private final FileInfo file;

  /** Fbrica de ns filho, caso o arquivo represente um diretrio. */
  private final IChildrenFactory childrenFactory;

  /** Indica se o arquivo  um executvel. */
  private final boolean executable;

  static {
    FILE_ICON = ClientProjectFileType.getFileType(ClientProjectFileType.UNKNOWN)
      .getBaseIcon();
    DIRECTORY_ICON = ClientProjectFileType.getFileType(
      ClientProjectFileType.DIRECTORY_TYPE).getBaseIcon();
  }

  /**
   * Construtor.
   *
   * @param tree rvore que detm o arquivo representado por este n.
   * @param version Verso que detm o arquivo representado por este n.
   * @param file Arquivo representado por este n.
   * @param childrenFactory Fbrica de ns filho, caso o arquivo represente um
   *        diretrio.
   * @param executable <tt>true</tt> se o arquivo for um executvel.
   */
  protected AbstractFileInfoNode(VersionTree tree, AlgorithmVersionInfo version,
    FileInfo file, IChildrenFactory childrenFactory, boolean executable) {
    super(tree, file.getName());

    this.version = version;
    this.file = file;
    this.childrenFactory = childrenFactory;
    this.executable = executable;

    for (FileInfo childFile : file.getChildren()) {
      addChild(childFile);
    }
  }

  /**
   * Obtm a verso que detm este arquivo.
   *
   * @return a verso que detm este arquivo.
   */
  public final AlgorithmVersionInfo getVersion() {
    return version;
  }

  /**
   * Obtm o arquivo representado por este n.
   *
   * @return o arquivo representado por este n.
   */
  public final FileInfo getFile() {
    return file;
  }

  /**
   * Indica se este n pode ser selecionado em conjunto com o n passado.
   *
   * @param other n a ser selecionado junto com esse.
   *
   * @return <tt>true</tt> se o n passado tiver o mesmo pai deste n.
   */
  @Override
  public final boolean allowMultipleSelection(AbstractVersionTreeNode other) {
    return (null != this.getParent() && this.getParent().equals(other
      .getParent()));
  }

  /**
   * Mtodo para comparao de ns. Compara os nomes sendo exibidos, permitindo
   * uma ordenao alfabtica crescente.
   *
   * @see java.lang.Comparable#compareTo(java.lang.Object)
   */
  @Override
  public final int compareTo(MutableTreeNode node) {
    if (node instanceof AbstractFileInfoNode) {
      AbstractFileInfoNode fileNode = (AbstractFileInfoNode) node;
      if (file.isDirectory() && !fileNode.file.isDirectory()) {
        return -1;
      }
      else if (!file.isDirectory() && fileNode.file.isDirectory()) {
        return 1;
      }
    }

    return super.compareTo(node);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final ImageIcon getImageIcon() {
    return file.isDirectory() ? DIRECTORY_ICON : FILE_ICON;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean canImport(Object[] data, DataFlavor flavor) {
    return getFile().isDirectory() && null != data && 0 < data.length
      && getDataFlavor().equals(flavor);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean importData(Object[] data) {
    DataTransferObject dto = (DataTransferObject) data[0];
    final AlgorithmVersionInfo sourceVersion = dto.getVersion();
    List<FileInfo> files = new ArrayList<FileInfo>(data.length);
    for (Object obj : data) {
      dto = (DataTransferObject) obj;
      FileInfo fileInfo = dto.getFile();
      files.add(fileInfo);
    }

    IPathFactory sourcePathFactory = createSourcePathFactory(dto);
    IPathFactory targetPathFactory = createTargetPathFactory();
    AlgorithmManagementProxy.copyFiles(sourceVersion, files, sourcePathFactory,
      getVersion(), getFile(), targetPathFactory, executable, getWindow());

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Serializable getData() {
    return createDataTransferObject();
  }

  /**
   * Cria e adiciona um n filho representando o arquivo passado.
   *
   * @param childFile Arquivo que ser representado por um n filho.
   */
  protected final void addChild(FileInfo childFile) {
    add(childrenFactory.createNode(childFile));
  }

  /**
   * Obtm uma fbrica que gere o caminho fsico at o diretrio representado
   * por este n. Esta fbrica ser utilizada quando arquivos forem ser copiados
   * para este diretrio.
   *
   * @return uma fbrica que gere o caminho fsico at o diretrio representado
   *         por este n. Esta fbrica ser utilizada quando quiser se copiar
   *         arquivos para este diretrio.
   */
  protected abstract IPathFactory createTargetPathFactory();

  /**
   * Obtm uma fbrica que gere um caminho fsico at os arquivos representados
   * pelo dto. Esta fbrica ser utilizada quando os arquivos representados pelo
   * dto forem copiados.
   *
   * @param dto Objeto utilizado para encapsular dados a serem transmitidos
   *        atravs de um {@link VersionTreeTransferHandler TransferHandler}.
   * @return uma fbrica que gere o caminho fsico.
   */
  protected abstract IPathFactory createSourcePathFactory(
    DataTransferObject dto);

  /**
   * Cria um objeto utilizado para encapsular dados a serem transmitidos atravs
   * de um {@link VersionTreeTransferHandler TransferHandler}.
   *
   * @return um objeto utilizado para encapsular dados a serem transmitidos
   *         atravs de um {@link VersionTreeTransferHandler TransferHandler}.
   */
  protected DataTransferObject createDataTransferObject() {
    return new DataTransferObject(this);
  }

  /**
   * Obtm um array com os arquivos cujos ns que os representam foram
   * selecionados na rvore e so do mesmo tipo do n que executou este mtodo.
   *
   *
   * @return um array de arquivos selecionados ou uma lista vazia caso no haja
   *         arquivos selecionados ou os arquivos no so do tipo correto.
   */
  protected FileInfo[] getSelectedFiles() {
    TreePath[] paths = getTree().getSelectionPaths();
    if (null == paths) {
      return new FileInfo[0];
    }
    else {
      FileInfo[] files = new FileInfo[paths.length];
      for (int inx = 0; inx < paths.length; inx++) {
        TreePath path = paths[inx];
        Object component = path.getLastPathComponent();
        if (getClass().isAssignableFrom(component.getClass())) {
          AbstractFileInfoNode node = (AbstractFileInfoNode) component;
          files[inx] = node.getFile();
        }
        else {
          /**
           * Se o arquivo no for do mesmo tipo de quem o pediu, uma lista vazia
           *  retornada.
           */
          return new FileInfo[0];
        }
      }

      return files;
    }
  }

  /**
   * Contrato de fbrica de ns filhos.
   */
  protected interface IChildrenFactory {
    /**
     * Cria um n filho.
     *
     * @param file Arquivo a ser representado por este n filho.
     *
     * @return um a ser adicionado como filho.
     */
    AbstractFileInfoNode createNode(FileInfo file);
  }

  /**
   * Representa um objeto utilizado para encapsular dados a serem transmitidos
   * atravs de um {@link VersionTreeTransferHandler TransferHandler}.
   */
  public static class DataTransferObject implements Serializable {

    /** Verso que detm o arquivo a ser copiado. */
    private final AlgorithmVersionInfo version;

    /** Arquivo a ser copiado. */
    private final FileInfo file;

    /**
     * Construtor.
     *
     * @param node N que representa o arquivo a ser copiado.
     */
    public DataTransferObject(AbstractFileInfoNode node) {
      version = node.getVersion();
      file = node.getFile();
    }

    /**
     * Obtm a verso que detm o arquivo a ser copiado.
     *
     * @return a verso que detm o arquivo a ser copiado.
     */
    public AlgorithmVersionInfo getVersion() {
      return version;
    }

    /**
     * Obtm o arquivo a ser copiado.
     *
     * @return o arquivo a ser copiado.
     */
    public FileInfo getFile() {
      return file;
    }
  }
}
