package csbase.client.applications.projectsmanager;

import java.util.Observable;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectPermissions;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.SharedProjectEvent;
import csbase.logic.User;

/**
 * Esta classe observa o servio de administrao de projetos, invocando mtodos
 * da aplicao ProjectsManager quando necessrio, para manter o status de
 * compartilhamento dos projetos na aplicao coerente com o sistema.
 * 
 * @author jnlopes
 * 
 */
class ProjectSharingStatusObserver extends AbstractProjectsManagerObserver {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  ProjectSharingStatusObserver(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(Observable o, Object arg) {
    if (o == null) {
      return;
    }

    if (!(arg instanceof SharedProjectEvent)) {
      return;
    }

    final SharedProjectEvent event = (SharedProjectEvent) arg;
    final CommonProjectInfo cpi = event.getProjectInfo();

    final ProjectsManagerData pmd = projectsManager.getProject(cpi);

    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        final SharingType newSharingType = event.getSharingType();
        if (pmd == null) {
          if (newSharingType != SharingType.PRIVATE) {
            addProject(cpi);
          }
          return;
        }

        final User user = ProjectsManager.getUser();
        final Object currentUserID = user.getId();
        if (!user.isAdmin()
          && userLostAccess(currentUserID, newSharingType, cpi)) {
          /*
           * Projeto estava disponvel para este usurio (que no  o seu dono),
           * mas o usurio foi removido da lista de acesso ou o projeto
           * tornou-se privado
           */
          projectsManager.removeProject(cpi);
        }
        else if (!setSharingStatusInfo(pmd, event)) {
          return;
        }
        projectsManager.refreshProjectsTable();
      }
    });
  }

  /**
   * Verifica se o usurio tinha acesso ao projeto e passou a no ter mais.
   * 
   * @param currentUserID ID do usurio corrente
   * @param newSharingType novo tipo de compartilhamento
   * @param cpi informaes do projeto
   * @return <code>true</code> se o usurio tinha acesso ao projeto e no tem
   *         mais
   */
  private boolean userLostAccess(final Object currentUserID,
    final SharingType newSharingType, final CommonProjectInfo cpi) {
    if (currentUserID.equals(cpi.userId)) {
      return false;
    }
    if (newSharingType == SharingType.PRIVATE
      || (newSharingType == SharingType.PARTIAL && !ProjectPermissions
        .userHasSelectiveAccess(cpi, currentUserID))) {
      return true;
    }
    return false;
  }

  /**
   * Realiza as alteraes de compartilhamento realizadas ao projeto na
   * estrutura ProjectsManagerData.
   * 
   * @param pmd dados do projeto
   * @param event Evento de compartilhamento de projeto.
   * @return boolean indicando se a tabela deve sofrer refresh ou no (i.e. se
   *         algum parmetro foi alterado).
   */
  private boolean setSharingStatusInfo(final ProjectsManagerData pmd,
    final SharedProjectEvent event) {

    final SharingType st = event.getSharingType();
    final boolean ret = pmd.setSharingType(st);

    /*
     * Mesmo que no haja alterao no modo de acesso, podem ter ocorrido
     * alteraes nos conjuntos de usurios compartilhados.
     */
    if (!ret && st != SharingType.PARTIAL) {
      return false;
    }

    if (event.getSharingType() == SharingType.PARTIAL) {
      pmd.setUsers(event.getUsersRO(), event.getUsersRW());
    }
    return true;
  }
}
