/**
 * $Id$
 */

package csbase.client.applications.projectsmanager.panels.filters;

import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.event.KeyListener;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;

/**
 * Painel que contem os filtros por espao alocado e/ou ocupado por projetos.
 * 
 * @author jnlopes
 * 
 */
public class ProjectSpaceFilterPanel extends AbstractProjectFilter {

  /**
   * Campo onde o espao mnimo de alocao deve ser preenchido.
   */
  private JTextField initialAllocatedSpaceField;

  /**
   * Campo onde o espao mximo de alocao deve ser preenchido.
   */
  private JTextField finalAllocatedSpaceField;

  /**
   * Campo onde o espao mnimo de ocupao deve ser preenchido.
   */
  private JTextField initialOccupiedSpaceField;

  /**
   * Campo onde o espao mximo de ocupao deve ser preenchido.
   */
  private JTextField finalOccupiedSpaceField;

  /**
   * Painel de alocao mnima de espao.
   */
  private JPanel allocatedFromSpacePanel;

  /**
   * Painel de alocao mxima de espao.
   */
  private JPanel allocatedUntilSpacePanel;

  /**
   * Label de espao de alocao.
   */
  private JLabel allocatedSpaceLabel;

  /**
   * Painel de ocupao mnima de espao.
   */
  private JPanel occupiedFromSpacePanel;

  /**
   * Painel de ocupao mxima de espao.
   */
  private JPanel occupiedUntilSpacePanel;

  /**
   * Label de espao de ocupao.
   */
  private JLabel occupiedSpaceLabel;

  /**
   * Label de espao de alocao inicial.
   */
  private JLabel initialAllocatedLabel;

  /**
   * Label de espao de alocao final.
   */
  private JLabel finalAllocatedLabel;

  /**
   * Label de espao de ocupao inicial.
   */
  private JLabel initialOccupiedLabel;

  /**
   * Label de espao de ocupao final.
   */
  private JLabel finalOccupiedLabel;

  /**
   * Classe privada para armazenar os parmetros de alocao e ocupao
   * definidos na interface.
   * 
   * @author Tecgraf
   */
  private static class SpaceData {
    /**
     * Alocao mnima.
     */
    long allocationMin;
    /**
     * Alocao mxima.
     */
    long allocationMax;
    /**
     * Ocupao mnima.
     */
    double occupationMin;
    /**
     * Ocupao mxima.
     */
    double occupationMax;

    /**
     * Verifica se os valores esto consistentes.
     * 
     * @return <code>true</code> se os valores esto consistentes
     */
    public boolean passesBasicTests() {
      return allocationMin >= 0L && allocationMin < allocationMax
        && occupationMin < occupationMax;
    }

    /**
     * Verifica se determinados valores de alocao e ocupao esto dentro das
     * faixas definidas por este objeto.
     * 
     * @param allocation valor de alocao a ser comparado
     * @param occupation valor de ocupao a ser comparado
     * @return <code>true</code> se ambos os valores esto dentro das
     *         respectivas faixas
     */
    public boolean contains(final long allocation, final double occupation) {
      return (allocation >= allocationMin && allocation <= allocationMax
        && occupation >= occupationMin && occupation <= occupationMax);
    }
  }

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ProjectSpaceFilterPanel(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * Inicia os componentes.
   */
  @Override
  protected void initComponents() {
    initialAllocatedSpaceField = new JTextField();
    finalAllocatedSpaceField = new JTextField();
    initialOccupiedSpaceField = new JTextField();
    finalOccupiedSpaceField = new JTextField();
    allocatedSpaceLabel =
      new JLabel(getString("ProjectSpaceFilterPanel.allocated.space.title"));
    occupiedSpaceLabel =
      new JLabel(getString("ProjectSpaceFilterPanel.occupied.space.title"));
    initialAllocatedLabel =
      new JLabel(
        getString("ProjectSpaceFilterPanel.initial.allocated.space.label"));
    finalAllocatedLabel =
      new JLabel(
        getString("ProjectSpaceFilterPanel.final.allocated.space.label"));
    initialOccupiedLabel =
      new JLabel(
        getString("ProjectSpaceFilterPanel.initial.occupied.space.label"));
    finalOccupiedLabel =
      new JLabel(
        getString("ProjectSpaceFilterPanel.final.occupied.space.label"));

    initialAllocatedLabel.setForeground(Color.GRAY);
    finalAllocatedLabel.setForeground(Color.GRAY);
    initialOccupiedLabel.setForeground(Color.GRAY);
    finalOccupiedLabel.setForeground(Color.GRAY);
  }

  /**
   * Monta o painel.
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());

    final JLabel gbytesLabel =
      new JLabel(getString("ProjectSpaceFilterPanel.gbytes.label"));
    gbytesLabel.setForeground(Color.GRAY);
    final JLabel mbytesLabel =
      new JLabel(getString("ProjectSpaceFilterPanel.mbytes.label"));
    mbytesLabel.setForeground(Color.GRAY);

    allocatedFromSpacePanel = new JPanel(new GridBagLayout());

    allocatedFromSpacePanel.add(initialAllocatedLabel, new GBC(0, 0));
    allocatedFromSpacePanel.add(initialAllocatedSpaceField, new GBC(1, 0)
      .horizontal().insets(0, 10, 0, 0));
    allocatedFromSpacePanel.add(gbytesLabel, new GBC(2, 0).insets(0, 0, 0, 10));

    allocatedUntilSpacePanel = new JPanel(new GridBagLayout());

    allocatedUntilSpacePanel.add(finalAllocatedLabel, new GBC(1, 0));
    allocatedUntilSpacePanel.add(finalAllocatedSpaceField, new GBC(2, 0)
      .horizontal().insets(0, 10, 0, 0));
    allocatedUntilSpacePanel.add(gbytesLabel, new GBC(3, 0));

    add(allocatedFromSpacePanel, new GBC(0, 0).horizontal());
    add(allocatedUntilSpacePanel, new GBC(0, 0).horizontal());

    occupiedFromSpacePanel = new JPanel(new GridBagLayout());

    occupiedFromSpacePanel.add(initialOccupiedLabel, new GBC(0, 0));
    occupiedFromSpacePanel.add(initialOccupiedSpaceField, new GBC(1, 0)
      .horizontal().insets(0, 10, 0, 0));
    occupiedFromSpacePanel.add(mbytesLabel, new GBC(2, 0).insets(0, 0, 0, 10));

    occupiedUntilSpacePanel = new JPanel(new GridBagLayout());

    occupiedUntilSpacePanel.add(finalOccupiedLabel, new GBC(0, 0));
    occupiedUntilSpacePanel.add(finalOccupiedSpaceField, new GBC(1, 0)
      .horizontal().insets(0, 10, 0, 0));
    occupiedUntilSpacePanel.add(mbytesLabel, new GBC(2, 0));

    add(occupiedFromSpacePanel, new GBC(1, 0).horizontal());
    add(occupiedUntilSpacePanel, new GBC(2, 0).horizontal());
  }

  /**
   * Converte o texto de um campo para o respectivo valor numrico, ou para um
   * default fornecido. A converso  feita baseada no tipo do valor default.
   * 
   * @param field campo
   * @param defaultValue valor default
   * @return o valor numrico associado ao campo ou o valor default caso no
   *         seja possvel fazer a converso
   */
  private Number fieldToValue(final JTextField field, final Number defaultValue) {
    final String text = field.getText();
    if (text.trim().isEmpty()) {
      return defaultValue;
    }
    Number result = defaultValue;
    try {
      if (defaultValue instanceof Long) {
        result = Long.valueOf(text);
      }
      else if (defaultValue instanceof Double) {
        result = Double.valueOf(text);
      }
      else {
        throw new RuntimeException(
          getString("ProjectSpaceFilterPanel.invalid.default.value.message")
            + defaultValue);
      }
    }
    catch (NumberFormatException e) {
      final ProjectsManager projectsManager = getProjectsManager();
      projectsManager.showError("<<"
        + getString("ProjectSpaceFilterPanel.invalid.value.message") + text
        + ">>");
    }
    return result;
  }

  /**
   * Cria e preenche uma estrutura com os parmetros de alocao e ocupao
   * definidos na interface.
   * 
   * @return estrutura com os parmetros de alocao e ocupao definidos na
   *         interface
   * 
   * @see SpaceData
   */
  private SpaceData fillSpaceData() {
    final SpaceData spaceData = new SpaceData();
    spaceData.allocationMin =
      fieldToValue(initialAllocatedSpaceField, 0L).longValue();
    spaceData.allocationMax =
      fieldToValue(finalAllocatedSpaceField, Long.MAX_VALUE).longValue();
    spaceData.occupationMin =
      fieldToValue(initialOccupiedSpaceField, -1.0).doubleValue();
    spaceData.occupationMax =
      fieldToValue(finalOccupiedSpaceField, Double.MAX_VALUE).doubleValue();

    return spaceData;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void runSpecificFilter(final List<ProjectsManagerData> allProjects,
    final List<ProjectsManagerData> filteredProjects) {

    final SpaceData spaceData = fillSpaceData();
    if (!spaceData.passesBasicTests()) {
      return;
    }

    for (ProjectsManagerData pmd : allProjects) {
      if (projectMatchesFilter(spaceData, pmd)) {
        filteredProjects.add(pmd);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean projectMatchesFilter(final ProjectsManagerData prj) {
    return projectMatchesFilter(fillSpaceData(), prj);
  }

  /**
   * Verifica se um projeto  aceito por este filtro.
   * 
   * @param spaceData parmetros de alocao e ocupao definidos na interface
   * @param prj projeto
   * @return <code>true</code> se o projeto  aceito por este filtro
   */
  private boolean projectMatchesFilter(final SpaceData spaceData,
    final ProjectsManagerData prj) {
    if (!spaceData.passesBasicTests()) {
      return false;
    }

    final long allocatedSpace = prj.getAllocatedSpace();
    final double occupiedSpace = prj.getOccupiedSpace();
    return (spaceData.contains(allocatedSpace, occupiedSpace));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addListener(KeyListener kl) {
    initialAllocatedSpaceField.addKeyListener(kl);
    finalAllocatedSpaceField.addKeyListener(kl);

    initialOccupiedSpaceField.addKeyListener(kl);
    finalOccupiedSpaceField.addKeyListener(kl);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void emptyFilterData() {
    initialAllocatedSpaceField.setText("");
    finalAllocatedSpaceField.setText("");

    initialOccupiedSpaceField.setText("");
    finalOccupiedSpaceField.setText("");
  }

  /**
   * Retorna o label de alocao de espao.
   * 
   * @return O label de alocao de espao.
   */
  public JLabel getAllocatedSpaceLabel() {
    return allocatedSpaceLabel;
  }

  /**
   * Retorna o label de ocupao de espao.
   * 
   * @return O label de ocupao de espao.
   */
  public JLabel getOccupiedSpaceLabel() {
    return occupiedSpaceLabel;
  }

  /**
   * Retorna o painel de ocupao de espao inicial.
   * 
   * @return O painel de ocupao de espao inicial.
   */
  public JPanel getOccupiedFromSpacePanel() {
    return occupiedFromSpacePanel;
  }

  /**
   * Retorna o painel de ocupao de espao final.
   * 
   * @return O painel de ocupao de espao final.
   */
  public JPanel getOccupiedUntilSpacePanel() {
    return occupiedUntilSpacePanel;
  }

  /**
   * Retorna o painel de alocao de espao inicial.
   * 
   * @return O painel de alocao de espao inicial.
   */
  public JPanel getAllocatedFromSpacePanel() {
    return allocatedFromSpacePanel;
  }

  /**
   * Retorna o painel de alocao de espao final.
   * 
   * @return O painel de alocao de espao final.
   */
  public JPanel getAllocatedUntilSpacePanel() {
    return allocatedUntilSpacePanel;
  }

}
