package csbase.client.project;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;

import tecgraf.javautils.core.lng.LNG;
import csbase.logic.ProjectFileType;
import csbase.logic.ProjectFileTypeFilter;

/**
 * ComboBox para seleo de tipos de arquivos de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ProjectFileTypeComboBox extends JComboBox<Object> {

  /** Representa todos os arquivos. */
  private static final String ALL_FILES = LNG.get("UTIL_ALL_FILES");

  /** Representa todos os diretrios. */
  private static final String ALL_DIRECTORIES = LNG.get("UTIL_ALL_DIRECTORIES");

  /** Representa todos os arquivos e diretrios. */
  private static final String ALL_FILES_AND_DIRECTORIES =
    LNG.get("UTIL_ALL_FILES_AND_DIRECTORIES");
  
  public enum Mode {
    FILE_ONLY,
    DIRECTORY_ONLY,
    FILE_AND_DIRECTORY
  }

  /** Valor utilizado para sinalizar todos os itens */
  private String allItem;

  /**
   * Cria uma ComboBox.
   * 
   * @param mode O modo.
   * @param useAllItem Indica se deve mostrar o item "Todos".
   * 
   * @param filter O filtro (no pode ser null).
   */
  public ProjectFileTypeComboBox(Mode mode, boolean useAllItem) {
    DefaultComboBoxModel<Object> model =
      new DefaultComboBoxModel<Object>();
    ProjectFileTypeFilter filter;
    switch (mode) {
    case DIRECTORY_ONLY:
      filter = ProjectFileTypeFilter.directories();
      break;
    case FILE_ONLY:
      filter = ProjectFileTypeFilter.files();
      break;
    case FILE_AND_DIRECTORY:
      filter = ProjectFileTypeFilter.all();
      break;
    default:
      String msg = String.format("Modo invlido: %s.", mode);
      throw new IllegalArgumentException(msg);
    }
    for (ProjectFileType type : ProjectFileType.getFileTypes(filter)) {
      model.addElement(type);
    }
    if (useAllItem) {
      switch (mode) {
      case DIRECTORY_ONLY:
        allItem = ALL_DIRECTORIES;
        break;
      case FILE_ONLY:
        allItem = ALL_FILES;
        break;
      case FILE_AND_DIRECTORY:
        allItem = ALL_FILES_AND_DIRECTORIES;
        break;
      default:
        String msg = String.format("Modo invlido: %s.", mode);
        throw new IllegalArgumentException(msg);
      }
      model.addElement(allItem);
    }
    else {
      allItem = null;
    }
    setModel(model);
    if (useAllItem) {
      selectAllItem();
    }
  }

  public boolean hasSelectedTypeCode() {
    return (getSelectedIndex() != -1);
  }

  public boolean isAllItemSelected() {
    if (allItem == null) {
      return false;
    }

    if (!hasSelectedTypeCode()) {
      return false;
    }

    Object item = getSelectedItem();
    return item.equals(allItem);
  }

  /**
   * Obtm o cdigo do tipo de arquivo de projetos que est selecionado.
   * 
   * @return O cdigo do tipo de arquivo de projetos ou <code>null</code>
   * caso nenhum esteja selecionado.
   */
  public String getSelectedTypeCode() {
    if (!hasSelectedTypeCode()) {
      return null;
    }

    if (isAllItemSelected()) {
      return null;
    }

    ProjectFileType type = (ProjectFileType) getSelectedItem();
    return type.getCode();
  }

  /**
   * Seleciona o cdigo do tipo informado.
   * 
   * @return <code>true</code> sucesso ou <code>false</code> caso o tipo 
   * no esteja nesta combobox. 
   */
  public boolean selectTypeCode(String typeCode) {
    for (int i = 0; i < getModel().getSize(); i++) {
      Object element = getModel().getElementAt(i);
      if (element instanceof ProjectFileType) {
        ProjectFileType type = (ProjectFileType) element;
        if (typeCode.equals(type.getCode())) {
          setSelectedItem(type);
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Seleciona o item "Todos".
   * 
   * @return <code>true</code> sucesso ou <code>false</code> caso o tipo 
   * no esteja nesta combobox. 
   */
  public boolean selectAllItem() {
    if (allItem == null) {
      return false;
    }
    setSelectedItem(allItem);
    return true;
  }
}
