/*
 * NotificationTranslator.java $Author: oikawa $ $Revision: 150777 $ - $Date:
 * 2006-12-07 15:53:32 -0200 (Thu, 07 Dec 2006) $
 */
package csbase.client.util;

import java.text.MessageFormat;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.logic.AlgorithmsReloadNotification;
import csbase.logic.CloseProjectNotification;
import csbase.logic.CommandEndNotification;
import csbase.logic.CommandErrorNotification;
import csbase.logic.CommandFailedNotification;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandKilledNotification;
import csbase.logic.CommandLostNotification;
import csbase.logic.CommandSuccessNotification;
import csbase.logic.CommandWithNoExitCodeNotification;
import csbase.logic.FileSystemSpaceNotification;
import csbase.logic.FileUpdateNotification;
import csbase.logic.Notification;
import csbase.logic.OpenProjectNotification;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectAllocationState;
import csbase.logic.ProjectNotification;
import csbase.logic.ProjectRecoveryFailureNotification;
import csbase.logic.ProjectRecoverySuccessNotification;
import csbase.logic.SGANotification;
import csbase.logic.SGASet;
import csbase.logic.ServerNotification;
import csbase.logic.UserNotification;
import csbase.logic.UsersNotification;
import csbase.logic.applicationservice.ApplicationsReloadNotification;

/**
 * Singleton do tradutor de notificaes vindas do servidor vindas do arquivo
 * <code> csbase/client/resources/properties/language/idiom_pt_BR.properties
 * </code>
 * 
 * @author Andr Luiz Clinio
 */
public class NotificationTranslator {
  /**
   * Prefixo utilizado nos nomes das chaves de traduo.
   */
  private static final String LNG_PREFIX = "NotificationTranslator.";

  /**
   * Traduo de comando sem classe definida. Retorna uma string de lixo para
   * depurao.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(Object data) {
    return LNG.get("notification.object.internal.error");
  }

  /**
   * Traduo de comando sem classe especfica de notificao definida. Retorna
   * uma string de lixo para depurao.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(Notification data) {
    return LNG.get("notification.untyped.object.internal.error");
  }

  /**
   * Traduo de notificao de usurio. Retorna uma string.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return .
   */
  public String translate(UserNotification data) {
    return data.toString();
  }

  /**
   * Traduo de reload de servio de algoritmos.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return uma string.
   */
  public String translate(AlgorithmsReloadNotification data) {
    return LNG.get("notification.data.algorithms.reload");
  }

  /**
   * Traduo de reload de servio de algoritmos.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return uma string.
   */
  public String translate(ApplicationsReloadNotification data) {
    return LNG.get("notification.data.applications.reload");
  }

  /**
   * Traduo de de/registro de SGA Retorna uma string.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return .
   */
  public String translate(SGANotification data) {
    SGASet info = data.getMainInfo();
    String sgaFormat = null;
    String sgaName = info.getName() + "/" + info.getPlatformId();
    if (data.getFlag() == SGANotification.SGA_START_UP) {
      sgaFormat = LNG.get("notification.data.sga.startup");
    }
    else if (data.getFlag() == SGANotification.SGA_SHUTDOWN) {
      sgaFormat = LNG.get("notification.data.sga.shutdown");
    }
    else if (data.getFlag() == SGANotification.SGA_REINIT) {
      sgaFormat = LNG.get("notification.data.sga.reinit");
    }
    else if (data.getFlag() == SGANotification.SGA_UPDATE) {
      sgaFormat = LNG.get("notification.data.sga.update");
    }
    else {
      return "!!!";
    }
    return MessageFormat.format(sgaFormat, new Object[] { sgaName });
  }

  /**
   * Traduo de start-up do SSI Retorna uma string.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return .
   */
  public String translate(ServerNotification data) {
    if (data.getState()) {
      final String startStr = LNG.get("notification.data.server.startup");
      return startStr;
    }
    final String shutdownStr = LNG.get("notification.data.server.shutdown");
    return shutdownStr;
  }

  /**
   * Traduo de comando terminado no SGA.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(CommandEndNotification data) {
    if (data.hasTimeInfo()) {
      return MessageFormat.format(
        LNG.get("notification.data.command.end"),
        new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
            formatInterval(data.getUserTime()),
            formatInterval(data.getCPUTime()),
            formatInterval(data.getElapsedTime()), data.getStartTime(),
            data.getEndTime(), data.getExecHost() });
    }

    return MessageFormat.format(
      LNG.get("notification.data.command.end.no.data"),
      new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
          data.getStartTime(), data.getEndTime(), data.getExecHost() });
  }

  /**
   * Traduo de comando terminado com erro no SGA.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(CommandErrorNotification data) {
    return MessageFormat.format(LNG.get("notification.data.command.error"),
      new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
          data.getStartTime(), data.getExecHost(),
          data.getFinalizationInfo().getExitCode() });

  }

  /**
   * Traduo de comando terminado no SGA, cujo cdigo de retorno no foi
   * encontrado.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(CommandWithNoExitCodeNotification data) {
    return MessageFormat.format(LNG.get("notification.data.command.no_code"),
      new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
          data.getStartTime(), data.getExecHost() });

  }

  /**
   * Traduo de comando terminado com sucesso no SGA.
   * 
   * @param data dados de fim de comando.
   * 
   * @return .
   */
  public String translate(CommandSuccessNotification data) {
    return MessageFormat.format(
      LNG.get("notification.data.command.success"),
      new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
          formatInterval(data.getUserTime()),
          formatInterval(data.getCPUTime()),
          formatInterval(data.getElapsedTime()), data.getStartTime(),
          data.getEndTime(), data.getExecHost() });

  }

  /**
   * Faz a formatao de um tempo em segundos
   * 
   * @param timesec o tempo
   * @return o texto
   */
  private String formatInterval(Integer timesec) {
    if (timesec == null || timesec < 0) {
      return "---";
    }
    return FormatUtils.formatInterval(timesec);
  }

  /**
   * Traduo de comando perdido no SGA.
   * 
   * @param data dados do comando.
   * 
   * @return .
   */
  public String translate(CommandLostNotification data) {
    return MessageFormat.format(LNG.get("notification.data.command.lost"),
      new Object[] { data.getCommandId(), data.getCmdDesc(), data.getTip(),
          data.getStartTime(), data.getExecHost() });
  }

  /**
   * Traduo de comando nao iniciado no SGA.
   * 
   * @param data dados do comando.
   * 
   * @return .
   */
  public String translate(CommandFailedNotification data) {
    CommandFinalizationInfo info = data.getFinalizationInfo();
    return MessageFormat.format(LNG.get("notification.data.command.failed"),
      new Object[] {
          data.getCommandId(),
          data.getCmdDesc(),
          data.getTip(),
          data.getStartTime(),
          data.getExecHost(),
          info.getFailureCause() == null ? info.getFinalizationType()
            .getDescription() : info.getFailureCause().getDescription() });
  }

  /**
   * Traduo de comando interrompido no SGA.
   * 
   * @param data dados do comando.
   * 
   * @return .
   */
  public String translate(CommandKilledNotification data) {
    CommandFinalizationInfo info = data.getFinalizationInfo();
    return MessageFormat.format(LNG.get("notification.data.command.killed"),
      new Object[] {
          data.getCommandId(),
          data.getCmdDesc(),
          data.getTip(),
          data.getStartTime(),
          data.getExecHost(),
          info.getFailureCause() == null ? info.getFinalizationType()
            .getDescription() : info.getFailureCause().getDescription() });
  }

  /**
   * Traduo de incluso/remoo de usurios em um projeto. Retorna uma string.
   * 
   * @param data objeto contedo de notificao
   * 
   * @return .
   */
  public String translate(UsersNotification data) {
    String text = null;
    Object projectAdmin = data.getUserId();
    String projectName = data.getProjectName();
    if (data.getType() == UsersNotification.INSERT) {
      text = LNG.get("notification.data.project.manager.user.inserted");
    }
    else if (data.getType() == UsersNotification.REMOVE) {
      text = LNG.get("notification.data.project.manager.user.removed");
    }
    else if (data.getType() == UsersNotification.CHANGE) {
      text = LNG.get("notification.data.project.manager.user.changed");
    }
    else if (data.getType() == UsersNotification.REMOVE_PROJECT) {
      text = LNG.get("notification.data.project.manager.prj.deleted");
    }
    else {
      return LNG.get("notification.data.project.manager.unknown");
    }
    return MessageFormat.format(text,
      new Object[] { projectName, projectAdmin });
  }

  /**
   * Traduo da notificao de administrao de projetos.
   * 
   * @param data O contedo de notificao
   * @return O texto correspondente  notificao que ser exibida ao usurio.
   */
  public String translate(ProjectNotification data) {
    final ProjectAdminInfo info = data.getProjectAdminInfo();
    final ProjectAllocationState state = info.getState();

    if (state == ProjectAllocationState.WAITING_AREA_ALLOCATION) {
      return MessageFormat.format(LNG.get("notification.data.area.alloc.wait"),
        new Object[] { info.getProjectName(), info.getOwnerId() });
    }
    if (state == ProjectAllocationState.UNLOCKED_WITH_AREA_ALLOCATED) {
      return MessageFormat.format(
        LNG.get("notification.data.area.unlock"),
        new Object[] { info.getProjectName(),
            new Long(info.getAreaRequestSize()),
            new Long(info.getAreaLockedSize()) });
    }
    if (state == ProjectAllocationState.WAITING_AREA_FREE) {
      return MessageFormat.format(LNG.get("notification.data.area.free.wait"),
        new Object[] { info.getProjectName(), info.getOwnerId() });
    }
    return LNG.get("notification.data.area.no.msg.error");
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * @return o texto
   */
  public String translate(ProjectRecoveryFailureNotification data) {
    String ownerLogin = data.getOwnerLogin();
    String ownerName = data.getOwnerName();
    String projectName = data.getProjectName();
    Object[] args = new Object[] { ownerLogin, ownerName, projectName };
    String fmt = LNG.get("notification.project.recovery.failure");
    String text = String.format(fmt, args);
    return text;
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * @return o texto
   */
  public String translate(ProjectRecoverySuccessNotification data) {
    String ownerLogin = data.getOwnerLogin();
    String ownerName = data.getOwnerName();
    String projectName = data.getProjectName();
    Object[] args = new Object[] { ownerLogin, ownerName, projectName };
    String fmt = LNG.get("notification.project.recovery.success");
    String text = String.format(fmt, args);
    return text;
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * @return o texto
   */
  public String translate(OpenProjectNotification data) {
    Object[] args = new Object[] { data.getProjectName() };
    String fmt = LNG.get("notification.data.project.open");
    return MessageFormat.format(fmt, args);
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * @return o texto
   */
  public String translate(FileSystemSpaceNotification data) {
    final String fsId = data.getFileSystemId();
    final String key = "notification.data.file.system." + fsId;
    final String fsText = !LNG.hasKey(key) ? fsId : LNG.get(key);
    final String msg = LNG.get("notification.data.file.system.space");
    final int scale = data.getScale();
    final double perc = data.getPercentage();
    final String text = String.format(msg, fsText, perc, scale);
    return text;
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * @return o texto
   */
  public String translate(CloseProjectNotification data) {
    return MessageFormat.format(LNG.get("notification.data.project.close"),
      new Object[] { data.getProjectName() });
  }

  /**
   * Faz a traduo de uma notificao.
   * 
   * @param data a notificao
   * 
   * @return o texto
   */
  public String translate(FileUpdateNotification data) {
    switch (data.getResult()) {
      case SUCCESS:
        return MessageFormat.format(
          LNG.get(LNG_PREFIX + "file.update.success"), data.getFilePath());
      case SOURCE_NOT_FOUND:
        return MessageFormat.format(
          LNG.get(LNG_PREFIX + "file.update.source.not.found"),
          data.getFilePath());
      case NO_PERMISSION:
        return MessageFormat
          .format(LNG.get(LNG_PREFIX + "file.update.no.permission"),
            data.getFilePath());
      case UNEXPECTED_ERROR:
        return MessageFormat.format(
          LNG.get(LNG_PREFIX + "file.update.unexpected.error"),
          data.getFilePath());
      case TEMPORARILY_UNAVAILABLE:
        return MessageFormat.format(
          LNG.get(LNG_PREFIX + "file.update.temporarily_unavailable"),
          data.getFilePath());
      case DISABLED:
        return MessageFormat.format(
          LNG.get(LNG_PREFIX + "file.update.disabled"), data.getFilePath());
      default:
        return "Resultado desconhecido." + data.toString();
    }
  }
}
