/**
 * $Id$
 */
package csbase.client.desktop;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Vector;

import csbase.client.externalresources.ExternalResources;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.PermissionProxy;
import csbase.exception.CSBaseRuntimeException;
import csbase.logic.LoginAsPermission;
import csbase.logic.Permission;
import csbase.logic.PreLoginData;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ServerEntryPoint;
import tecgraf.javautils.core.lng.LNG;

/**
 * Esta classe executa uma ao de logar como (sudo). Permite um usurio logar
 * como outro usurio, desde que tenha permisso para tal. Adiciona um atributo
 * ("realUser") na sesso (preLogin) com o usurio corrente. Com o token obtido
 * gera uma URL que  carregada para mostrar um segundo desktop logado como o
 * novo usurio. Esta action tem que ser desabilitada pela aplicao que for
 * utiliz-la como item de menu., caso no haja logins disponveis para o
 * usurio corrente logar como:
 * loginAsMenuItem.setEnabled(loginAsDesktopAction.hasAvailableLogins());
 *
 * @author Tecgraf
 */
public class LoginAsDesktopAction extends DesktopAction<DesktopFrame> {

  /** Chave do atributo do usurio real */
  public static final String REAL_USER_ATTRIBUTE = "realUser";

  /** Login para o qual se deseja logar */
  private String newLogin;

  /** Nome do sistema a ser aberto (nome da pgina html alternativa) */
  private String systemName;

  /** Lista de logins disponveis para aquele usurio */
  private String[] availableLogins;

  /**
   * Construtor
   *
   * @param systemName Nome do sistema a ser aberto (nome da pgina html
   *        alternativa)
   *
   */
  public LoginAsDesktopAction(String systemName) {
    super(DesktopFrame.getInstance());
    final ExternalResources extResources = ExternalResources.getInstance();
    availableLogins = getAvailableLogins();
    // Por alguma razo, est habilitando a action, mesmo com o cdigo abaixo
    setEnabled(extResources.isEnabled());
    this.systemName = systemName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent arg0) {
    // Abre dilogo para usurio escolher novo login
    final LoginAsDialog loginAsDialog =
      new LoginAsDialog(getDesktop(), availableLogins);
    loginAsDialog.center();
    loginAsDialog.setVisible(true);

    newLogin = loginAsDialog.getSelectedNewUser();
    // Se o usurio confirmou, vai ter um novo login selecionado
    if (newLogin != null) {
      loginAs();
    }
  }

  /**
   * Retorna a lista dos logins disponveis para o usurio correntemente logado,
   * ou null se no houver nenhum. Varre todas as permisses de "login as" e
   * consolida as listas de logins possveis de cada uma
   *
   * @return a lista dos logins disponveis para o usurio correntemente logado,
   *         ou null se no houver nenhum
   */
  private String[] getAvailableLogins() {
    String[] ret = null;
    /*
     * Se o loginInfo no foi inicializado (por exemplo, no caso de este j ser
     * um usurio que foi "logado como") no ser possvel logar como outros
     * usurios
     */
    if (!ClientRemoteMonitor.getInstance().isLoginInfoInitialized()) {
      return null;
    }
    ArrayList<String> retList = new ArrayList<String>();
    Vector<Permission> perms = PermissionProxy.getAllPermissions(
      getDesktop().getDesktopFrame(), LNG.get("IAS_USER_PERMISSIONS_TITLE"),
      LNG.get("IAS_WAITING_ALL_PERMISSIONS"));
    for (Permission permission : perms) {
      if (permission instanceof LoginAsPermission) {
        LoginAsPermission loginPerm = (LoginAsPermission) permission;
        retList.addAll(loginPerm
          .getAllowedLogins(ClientRemoteMonitor.getInstance().getLogin()));
      }
    }
    if (!retList.isEmpty()) {
      ret = retList.toArray(new String[retList.size()]);
    }
    return ret;
  }

  /**
   * Devolve true se houver pelo menos um login para o qual este usurio puder
   * logar como
   *
   * @return true se houver pelo menos um login para o qual este usurio puder
   *         logar como
   */
  public boolean hasAvailableLogins() {
    return availableLogins != null;
  }

  /**
   * Prepara a task e executa-a para que seja feito o login como
   *
   * @return true se executou a task com sucesso
   */
  protected boolean loginAs() {
    final ExternalResources extResources = ExternalResources.getInstance();
    if (!extResources.isEnabled()) {
      showErrorMessage(getString("no.external.resources.error"));
      return false;
    }

    final LoginAsTask task = new LoginAsTask(this);
    final DesktopFrame desktop = getDesktop();
    final DesktopComponentFrame frame = desktop.getDesktopFrame();
    final String title = frame.getTitle();
    final String message = getString("task.message");
    final boolean executed = task.execute(frame, title, message);
    if (!executed) {
      String causeText = "";
      final Exception exception = task.getError();
      if (exception != null) {
        causeText = exception.getMessage();
        System.out.println("Erro: " + causeText);
        exception.printStackTrace();
      }
      final String tag = "launch.error";
      final String err = getString(tag, causeText);
      showErrorMessage(err);
      return false;
    }
    return true;
  }

  /**
   * Devolve o login para o qual se deseja logar
   *
   * @return login para o qual se deseja logar
   */
  public String getNewLogin() {
    return newLogin;
  }

  /**
   * Atribui o valor do login para o qual se deseja logar
   *
   * @param newLogin login para o qual se deseja logar
   */
  public void setNewLogin(String newLogin) {
    this.newLogin = newLogin;
  }

  /**
   * Devolve o nome do sistema a ser aberto (nome da pgina html alternativa)
   *
   * @return nome do sistema a ser aberto (nome da pgina html alternativa)
   */
  public String getSystemName() {
    return systemName;
  }
}

/**
 * Task remota de clonagem do desktop.
 *
 * @author Tecgraf/PUC-Rio
 */
class LoginAsTask extends RemoteTask<Void> {

  /**
   * Ao
   */
  final private LoginAsDesktopAction action;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleError(final Exception error) {
    final String message = error.getMessage();
    action.showErrorMessage(message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleServerError(final CSBaseRuntimeException cserror) {
    final String message = cserror.getMessage();
    action.showErrorMessage(message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void performTask() throws Exception {
    final ServerEntryPoint serverEntryPoint = ClientRemoteLocator.server;
    final Locale locale = LNG.getLocale();
    final ClientRemoteMonitor clientRemoteMonitor =
      ClientRemoteMonitor.getInstance();
    final String userLogin = clientRemoteMonitor.getLogin();
    Map<String, Serializable> attributes =
      serverEntryPoint.getSessionAttributes(userLogin);
    if (attributes == null) {
      attributes = new HashMap<String, Serializable>();
    }
    // Para funcionar o sudo, a senha tem que ser vazia e o realUser escrito
    attributes.put(LoginAsDesktopAction.REAL_USER_ATTRIBUTE, userLogin);
    PreLoginData preLoginData =
      serverEntryPoint.preLogin(action.getNewLogin(), "", locale, attributes);
    String urlStr = serverEntryPoint.generateSystemURL(preLoginData.getToken(),
      locale, action.getSystemName());
    URL url = new URL(urlStr);
    final ExternalResources extResources = ExternalResources.getInstance();
    extResources.showDocument(url);
  }

  /**
   * Construtor
   *
   * @param action ao.
   */
  public LoginAsTask(final LoginAsDesktopAction action) {
    this.action = action;
  }
}
