/*
 * $Id: ProjectTreeNode.java 176230 2016-09-26 19:28:25Z jnlopes $
 */

package csbase.client.project;

import java.util.ArrayDeque;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.GetChildrenTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.NoHiddenFileFilter;
import csbase.logic.ProjectFileFilter;
import csbase.logic.ProjectFileType;

/**
 * Define um n de uma rvore de projetos.
 *
 * @author Rodrigo Carneiro Henrique (rodrigoh)
 */
public final class ProjectTreeNode implements TreeNode {
  /**
   * N-pai.
   */
  private ProjectTreeNode parent;
  /**
   * Arquivo associado ao n.
   */
  private ClientProjectFile file;
  /**
   * Lista de ns filhos.
   */
  private List<ProjectTreeNode> childList;
  /**
   * Indica se todos os filhos j foram obtidos recursivamente do servidor.
   */
  private boolean hasAllTree;
  /**
   * Filtro para exibio dos ns.
   */
  private ProjectFileFilter filter;
  /**
   * Comparador.
   */
  private Comparator<ProjectTreeNode> comparator;

  /**
   * Construtor.
   *
   * @param project projeto ao qual o n pertence
   * @param filter filtro para exibio dos ns
   * @param comparator comparador
   */
  ProjectTreeNode(CommonClientProject project, ProjectFileFilter filter,
    Comparator<ProjectTreeNode> comparator) {
    this(null, project.getRoot(), false, filter, comparator);
  }

  /**
   * Construtor.
   *
   * @param parent N pai.
   * @param file Arquivo associado ao n.
   * @param hasAllTree Indicativo de rvore completa no cliente.
   * @param filter Filtro para exibio dos ns.
   * @param comparator Comparador de ns.
   */
  private ProjectTreeNode(ProjectTreeNode parent, ClientProjectFile file,
    boolean hasAllTree, ProjectFileFilter filter,
    Comparator<ProjectTreeNode> comparator) {
    this.parent = parent;
    this.file = file;
    this.childList = null;
    this.hasAllTree = hasAllTree;
    this.filter = filter;
    this.comparator = comparator;
  }

  /**
   * Carrega os filhos do n corrente, recuperando os dados do servidor, se
   * necessrio. A recuperao dos dados do servidor pode ser forada atravs do
   * parmetro force. Mas note que mesmo que essa recuperao remota no seja
   * forada, ela pode ser feita se for necessrio.
   *
   * @param force Indica se deve forar o descarte das informaes locais e
   *        buscar os dados diretamente do servidor. Se for verdadeiro, as
   *        infores locais sero descartadas. Se for falso, utilizar as
   *        informaes locais, caso seja possvel.
   */
  void loadChildren(boolean force) {
    if (force) {
      childList = null;
      hasAllTree = false;
    }
    if (childList != null || !file.isDirectory()) {
      return;
    }
    ClientProjectFile[] children;
    if (!hasAllTree) {
      boolean bringAllTree = (filter != null || force);
      children = GetChildrenTask.runTask(file, bringAllTree, bringAllTree);
      hasAllTree = bringAllTree;
    }
    else {
      children = file.getLocalChildren();
    }
    setChildren(children);
  }

  /**
   * @param children
   */
  private void setChildren(final ClientProjectFile[] children) {
    this.childList = new LinkedList<>();
    if (children == null) {
      return;
    }
    NoHiddenFileFilter noDotFileFilter = NoHiddenFileFilter.getInstance();
    boolean showHiddenFiles =
      DesktopFrame.getInstance().shouldShowHiddenFiles();
    for (int i = 0; i < children.length; i++) {
      /*
       * se o nome do arquivo comea com ".", ele  ignorado
       */
      if (!showHiddenFiles && !noDotFileFilter.accept(children[i])) {
        continue;
      }
      if (filter == null || filter.accept(children[i])) {
        ProjectTreeNode treeNode = new ProjectTreeNode(this, children[i],
          hasAllTree, filter, comparator);
        this.childList.add(treeNode);
      }
    }
    Collections.sort(childList, comparator);
  }

  /**
   * @return {@link ClientProjectFile} associado ao n
   */
  public ClientProjectFile getClientProjectFile() {
    return this.file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Enumeration<ProjectTreeNode> children() {
    loadChildren(false);
    return new ChildEnumeration();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    ProjectTreeNode node = (ProjectTreeNode) obj;
    return this.file.equals(node.file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean getAllowsChildren() {
    return file.isDirectory() && !isRestrictedDataset();
  }

  /**
   * Obtm o n-filho do n corrente associado a um {@link ClientProjectFile}
   * especfico.
   *
   * @param child arquivo associado ao n-filho
   * @return n-filho associado ao arquivo, ou <code>null</code> caso este no
   *         exista
   */
  ProjectTreeNode getChild(ClientProjectFile child) {
    if (!getAllowsChildren()) {
      return null;
    }
    loadChildren(false);
    for (ProjectTreeNode node : childList) {
      if (node.file.equals(child)) {
        return node;
      }
    }
    return null;
  }

  /**
   * Obtm o n-filho do n corrente associado a um nome de arquivo.
   *
   * @param fileName nome do arquivo
   * @return n-filho associado ao arquivo com o nome especifica, ou
   *         <code>null</code> caso este no exista
   */
  ProjectTreeNode getChild(String fileName) {
    if (!getAllowsChildren()) {
      return null;
    }
    loadChildren(false);
    for (ProjectTreeNode node : childList) {
      if (node.file.getName().equals(fileName)) {
        return node;
      }
    }
    return null;
  }

  /**
   * Obtm o nome do n corrente.
   *
   * @return nome do n corrente
   */
  String getName() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TreeNode getChildAt(int childIndex) {
    if (!getAllowsChildren()) {
      return null;
    }
    loadChildren(false);
    return childList.get(childIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getChildCount() {
    if (!getAllowsChildren()) {
      return 0;
    }
    loadChildren(false);
    return childList.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TreeNode getParent() {
    return this.parent;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getIndex(TreeNode node) {
    if (!getAllowsChildren()) {
      return -1;
    }
    loadChildren(false);
    return this.childList.indexOf(node);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.file.hashCode();
  }

  /**
   * Indica se n deve ser exibido como dataset (com contedo oculto).
   *
   * @return Boolean que indica se n deve ser exibido como dataset (com
   *         contedo oculto).
   */
  public boolean isRestrictedDataset() {
    String type = file.getType();
    ProjectFileType fileType = ProjectFileType.getFileType(type);
    return fileType.isDirectory()
      && !fileType.getCode().equals(ProjectFileType.DIRECTORY_TYPE)
      && !DesktopFrame.getInstance().shouldExpandDataset();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLeaf() {
    return !getAllowsChildren() || !file.hasChildren();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return file.toString();
  }

  /**
   * Enumerao de ns.
   *
   * @author Tecgraf
   */
  private final class ChildEnumeration implements Enumeration<ProjectTreeNode> {
    /**
     * Iterador sobre a enumerao.
     */
    private Iterator<ProjectTreeNode> iterator;

    /**
     * Construtor.
     */
    ChildEnumeration() {
      this.iterator = ProjectTreeNode.this.childList.iterator();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean hasMoreElements() {
      return this.iterator.hasNext();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ProjectTreeNode nextElement() {
      return (this.iterator.next());
    }
  }

  /**
   * @return filtro para exibio dos ns
   */
  ProjectFileFilter getFilter() {
    return this.filter;
  }

  /**
   * Obtm os ndices dos ns dentre os filhos do n corrente.
   *
   * @param treeNodeArray array com ns filhos do n corrente
   * @return array com os ndices dos ns dentre os filhos do n corrente. Ns
   *         que no sejam filhos do n corrente tero ndice -1.
   */
  int[] getIndexes(ProjectTreeNode[] treeNodeArray) {
    int[] indexes = new int[treeNodeArray.length];
    for (int i = 0; i < indexes.length; i++) {
      indexes[i] = this.getIndex(treeNodeArray[i]);
    }
    return indexes;
  }

  /**
   * @return caminho da raiz at o n corrente, sob a forma de um
   *         {@link TreePath}
   */
  TreePath getTreePath() {
    ProjectTreeNode node = this;
    Deque<ProjectTreeNode> pathToRoot = new ArrayDeque<>();
    do {
      pathToRoot.addFirst(node);
      node = (ProjectTreeNode) node.getParent();
    } while (node != null);
    return new TreePath(
      pathToRoot.toArray(new ProjectTreeNode[pathToRoot.size()]));
  }

  /**
   * Redefine o arquivo associado ao n e invalida a lista de filhos.
   *
   * @param file - novo arquivo a ser associado ao n
   */
  public void setClientProjectFile(ClientProjectFile file) {
    this.file = file;
    childList = null;
    hasAllTree = false;
    if (parent != null) {
      // Reordena os filhos do pai pois o nome desse arquivo pode ter mudado.
      Collections.sort(parent.childList, comparator);
    }
  }

  /**
   * Adiciona um novo n  rvore, se esta operao for compatvel com o filtro
   * de visualizao atual. Arquivos com nome comeando com "." so exibidos
   * apenas se o usurio optou por visualizar tais arquivos.
   *
   * @param child - arquivo associado ao novo n
   * @return true se um novo n foi criado na rvore (i.e. se o arquivo era
   *         compatvel com os filtros)
   */
  public boolean addChild(ClientProjectFile child) {
    /*
     * Aqui  onde o controle de exibio de arquivos ocultos est.
     */
    boolean showHiddenFiles =
      DesktopFrame.getInstance().shouldShowHiddenFiles();
    NoHiddenFileFilter noDotFileFilter = NoHiddenFileFilter.getInstance();

    if (!showHiddenFiles && !noDotFileFilter.accept(child)) {
      return false;
    }

    if (isRestrictedDataset()) {
      return false;
    }

    if (filter == null || filter.accept(child)) {
      if (childList == null) {
        childList = new LinkedList<>();
      }
      childList
        .add(new ProjectTreeNode(this, child, hasAllTree, filter, comparator));
      Collections.sort(childList, comparator);
      file.setHasChildren(true);
      return true;
    }
    return false;
  }

  /**
   * Remove o n-filho associado a um determinado arquivo.
   *
   * @param child arquivo
   * @return <code>true</code> se o n foi removido
   */
  public boolean removeChild(ClientProjectFile child) {
    ProjectTreeNode treeNode = getChild(child);
    if (treeNode != null) {
      if (childList.remove(treeNode)) {
        Collections.sort(childList, comparator);
        file.setHasChildren(!childList.isEmpty());
        return true;
      }
    }
    return false;
  }

  /**
   * Redefine o filtro.
   *
   * @param filter novo filtro
   */
  void setFilter(ProjectFileFilter filter) {
    this.filter = filter;
    if (!getAllowsChildren()) {
      return;
    }
    childList = null;
  }

  /**
   * Redefine o comparador.
   *
   * @param comparator
   */
  void setComparator(Comparator<ProjectTreeNode> comparator) {
    this.comparator = comparator;
    if (!getAllowsChildren()) {
      return;
    }
    loadChildren(false);
    Collections.sort(this.childList, this.comparator);
    for (ProjectTreeNode child : childList) {
      child.setComparator(comparator);
    }
  }
}
