package csbase.client.rest;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.ServerSocket;
import java.net.URI;
import java.rmi.RemoteException;
import java.security.DigestException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import javax.ws.rs.core.Configuration;

import org.glassfish.grizzly.http.server.CLStaticHttpHandler;
import org.glassfish.grizzly.http.server.HttpServer;
import org.glassfish.jersey.grizzly2.httpserver.GrizzlyHttpServerFactory;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.model.ModelProcessor;
import org.glassfish.jersey.server.model.Resource;
import org.glassfish.jersey.server.model.ResourceMethod;
import org.glassfish.jersey.server.model.ResourceModel;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;

import csbase.client.Client;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.NotificationPanel;
import csbase.client.desktop.RemoteTask;
import csbase.client.login.LoginInterface;
import csbase.client.login.PreLogin;
import csbase.client.rest.ReloadFilter.RequestCounter;
import csbase.logic.SecureKey;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.RestServiceInterface;
import io.swagger.jaxrs.config.BeanConfig;
import tecgraf.javautils.core.lng.LNG;

/**
 * Controlador REST para o cliente.
 */
public class RestController {
  /**
   * Singleton
   */
  private static RestController instance;

  /**
   * Configurao de resources oferecidos.
   */
  private ResourceConfig resourceConfig;

  /**
   * Servidor HTTP
   */
  private HttpServer httpServer;

  /**
   * Container Jersey para atualizao dinmica de recursos
   */
  private Container container;

  /**
   * Token
   */
  private String token;

  /**
   * Usurio
   */
  private final LoginInterface userLogin;

  /**
   * Indicativo interno de developer mode (no demanda token)
   */
  private boolean developMode = false;

  /**
   * Pacotes de recursos da API rest
   */
  private final List<String> resourcePackages;

  /**
   * Conjunto de recursos por prefixo da aplicao (padro appId/instanceId).
   */
  private final Map<String, Set<Object>> instancesByPrefix;

  /**
   * Mapa de prefixos da aplicao por classe de recurso.
   */
  private final Map<Class<?>, String> prefixByResourceClass;

  /**
   * Diretrio de recursos para interface web da documentao
   */
  private String docFrontEndDirectory;

  /**
   * Bean de configurao de documentao Swagger.
   */
  private BeanConfig beanConfig;

  /**
   * Flag que inidica que o container HTTP deve ser reiniciado
   */
  private boolean shouldReload;

  /**
   * Define o diretrio de recursos para interface web da documentao
   *
   * @param docFrontEndDirectory diretrio.
   */
  public void setDocFrontEndDirectory(String docFrontEndDirectory) {
    if (httpServer != null && httpServer.isStarted()) {
      throw new IllegalStateException("Server is already started");
    }
    this.docFrontEndDirectory = docFrontEndDirectory;
  }

  /**
   * Pporta do servidor HTTP
   */
  private int httpPort;

  /**
   * Mtodo para recuperar a instncia do singleton do controlador REST.
   *
   * @return A instncia nica do controlador REST
   */
  public static RestController getInstance() {
    if (instance == null) {
      instance = new RestController();
    }
    return instance;
  }

  /**
   * Host padro para acesso.
   */
  private static final String DEFAULT_HOST = "127.0.0.1";

  /**
   * Porta padro do servidor REST
   */
  private static final int DEFAULT_PORT = 15000;

  /**
   * Determina uma porta TCP livre no host local
   *
   * @return Uma porta TCP que no est sendo usada
   */
  private static int getAvailablePort() {
    ServerSocket socket = null;
    int port = -1;
    try {
      socket = new ServerSocket(0);
      port = socket.getLocalPort();
    }
    catch (IOException ex) {
    }
    finally {
      if (socket != null) {
        try {
          socket.close();
        }
        catch (IOException e) {
          throw new RuntimeException("You should handle this error.", e);
        }
      }
    }

    return port;
  }

  /**
   * Construtor privado do controlador REST, somente acessvel via getInstance
   */
  private RestController() {
    this.userLogin = Client.getInstance().getLoginObject();

    //pacotes de recursos REST do cliente
    this.resourcePackages = new ArrayList<>();
    resourcePackages.add("csbase.client.rest.resources");

    //objetos de recursos de aplicaes
    instancesByPrefix = new HashMap<>();

    prefixByResourceClass = new HashMap<>();

    //Diretorio padro do frontend swagger para documentao
    this.docFrontEndDirectory = "csbase/client/rest/swagger-ui/";

    if (userLogin instanceof PreLogin) {
      this.token = ((PreLogin) userLogin).getClientInstanceId();
      if (this.token.startsWith("TOKENID:")) {
        this.token = this.token.substring("TOKENID:".length());
      }
    }
    else {
      SecureKey sessionKey = new SecureKey();
      try {
        this.token = sessionKey.digest();
      }
      catch (DigestException e) {
        e.printStackTrace();
        this.token = userLogin.getClientInstanceId();
      }
    }
  }

  /**
   * Recupera o token de autenticao do usurio.
   *
   * @return O token de login do usurio corrente.
   */
  public String getAuthToken() {
    return token;
  }

  /**
   * Verifica se o servidor est em modo Developer.
   *
   * @return Retorna true no caso de estar no modo de desenvolvimento, ou false
   * caso contrario
   */
  public boolean isDeveloperMode() {
    return developMode;
  }

  /**
   * Verifica se um token de usurio  vlido. No caso de estar no modo de
   * desenvolvimento, retorna true independentemente do token.
   *
   * @param checkToken O token a ser verificado
   * @return true se o token passado  vlido, ou false caso contrario
   */
  public boolean isTokenAuthorized(String checkToken) {
    return isDeveloperMode() || this.token.equals(checkToken);
  }

  /**
   * Recupera o login do usuario corrente.
   *
   * @return a interface de login
   */
  public LoginInterface getUserLogin() {
    return userLogin;
  }

  /**
   * Encerra o servidor REST.
   */
  public void stopServer() {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();

      RemoteTask<Void> task = new RemoteTask<Void>() {
        @Override
        protected void performTask() throws Exception {
          ClientRemoteLocator.restService.unregisterClientHttpServer(token);
        }
      };
      final String msg = LNG.get("csbase.client.rest.server.unregistering");
      task.execute(DesktopFrame.getInstance().getDesktopFrame(), null, msg);

      if (task.getError() != null) {
        task.getError().printStackTrace();
      }
    }
  }

  /**
   * Inicia o servidor REST na porta e host padro.
   */
  public void startServer() {
    try {
      startServer(DEFAULT_HOST, DEFAULT_PORT);
    }
    catch (javax.ws.rs.ProcessingException ex) {
      //Tentar uma porta disponivel
      int availablePort=getAvailablePort();
      startServer(DEFAULT_HOST, availablePort);
    }
  }

  /**
   * Inicia o servidor REST no host padro em uma porta especfica.
   *
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(int port) {
    startServer(DEFAULT_HOST, port);
  }

  /**
   * Inicia o servidor REST em uma porta e host especficos.
   *
   * @param host Host a ser usado pelo servidor.
   * @param port porta a ser usada pelo servidor
   */
  public void startServer(String host, int port) {
    this.httpPort = port;

    try {
      this.beanConfig = new BeanConfig();
      beanConfig.setVersion("1.0");
      beanConfig
        .setDescription(LNG.get("csbase.client.rest.server.description"));
      beanConfig.setBasePath("/");
      beanConfig.setResourcePackage(String.join(",", resourcePackages));
      beanConfig.setScan(true);

      this.resourceConfig = buildResourceConfig();

      this.resourceConfig.registerInstances(new ContainerLifecycleListener() {
        @Override
        public void onStartup(Container cont) {
          RestController.this.container = cont;
        }

        @Override
        public void onReload(Container cont) {
        }

        @Override
        public void onShutdown(Container cont) {
        }
      });

      final String baseURL = "http://" + host + ":" + port + "/";

      this.httpServer = GrizzlyHttpServerFactory
              .createHttpServer(URI.create(baseURL), this.resourceConfig);

      //Registro de UI do swagger
      CLStaticHttpHandler staticHttpHandler =
        new CLStaticHttpHandler(RestController.class.getClassLoader(),
          this.docFrontEndDirectory);
      httpServer.getServerConfiguration()
        .addHttpHandler(staticHttpHandler, "/docs/");

      RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
        @Override
        protected void performTask() throws Exception {
          final RestServiceInterface restService =
            ClientRemoteLocator.restService;
          restService.registerClientHttpServer(token, baseURL);
          boolean devMode = restService.isClientDeveloperMode();
          setResult(devMode);
        }
      };

      final String taskMsg = LNG.get("csbase.client.rest.server.registering");
      task.execute(DesktopFrame.getInstance().getDesktopFrame(), null, taskMsg);

      final Exception error = task.getError();
      if (error == null) {
        this.developMode = task.getResult();
      }
      else {
        final String errorMsg =
          LNG.get("csbase.client.rest.server.start.error") + " - " + error
            .getLocalizedMessage();

        addNotificationMessage(errorMsg);
      }

      final String logMsg = LNG.get(
        "csbase.client.rest.server.started") + " - " + port +
        (isDeveloperMode() ? " (dev mode)" : "");

      addNotificationMessage(logMsg);

      final String apiMsg = LNG.get(
        "csbase.client.rest.server.doc.notification") + " " + baseURL + "docs/";

      addNotificationMessage(apiMsg);

      logMessage("RestService", "Sevidor iniciado na porta " + port);

    }
    catch (Throwable ex) {

      ex.printStackTrace();

      //Repassar exceo de porta invalida
      if(ex instanceof javax.ws.rs.ProcessingException) {
        throw ex;
      }

      final String apiMsg =
        LNG.get("csbase.client.rest.server.error") + " " + ex
          .getLocalizedMessage();

      addNotificationMessage(apiMsg);



    }

  }

  /**
   * Metodo para adicionar uma mensagem ao notification panel
   *
   * @param msg Mensagem a ser impressa
   */
  private void addNotificationMessage(final String msg) {
    NotificationPanel notificationPanel =
            DesktopFrame.getInstance().getNotificationPanel();

    notificationPanel.addNotificationLine(notificationPanel
            .makeNotificationLine(new Date(), "ClientRestService", msg, true));
  }

  /**
   * Metodo para registro de recursos no servidor REST.
   *
   * @param classes Lista de classes de recursos a serem registradas.
   */
  public void registerService(Class<?>[] classes) {
    resourceConfig.registerClasses(classes);
  }

  /**
   * Adiciona uma lista de pacotes a lista de recursos a serem registrados no
   * servidor REST.
   *
   * @param packages Lista de pacotes com resursos REST a serem adicionados
   */
  public void addResourcePackages(String... packages) {
    resourcePackages.addAll(Arrays.asList(packages));

    beanConfig.setResourcePackage(String.join(",", resourcePackages));
    beanConfig.setScan(true);

    if (container != null) {
      this.resourceConfig = buildResourceConfig();
      container.reload(this.resourceConfig);
    }
  }

  /**
   * Adiciona uma lista de instncias de recursos a serem registrados no
   * servidor REST.
   *
   * @param prefix O prefixo da aplicao
   * @param config Cnfigurao de recursos REST a serem adicionados
   */
  public void addApplicationResources(String prefix, ResourceConfig config) {

    Set<Object> instances = config.getInstances();

    for (Object i : instances) {
      prefixByResourceClass.put(i.getClass(), prefix);
    }

    instancesByPrefix.put(prefix, instances);

    shouldReload = true;
    reloadContainer();
  }

  /**
   * Recarrega o container (ou registra um timer para fazer a recarga caso no
   * seja possvel recarregar no momento).
   */
  private void reloadContainer() {
    RequestCounter counter = ReloadFilter.getCounter();

    synchronized (counter.getLock()) {
      if (shouldReload) {
        // S recarrega efetivamente se no existirem requisies sendo
        // atendidas no momento.
        if (counter.isZero()) {
          if (container != null) {
            this.resourceConfig = buildResourceConfig();
            container.reload(this.resourceConfig);
          }
          shouldReload = false;
        }
        else {
          // Caso no seja possvel recarregar, registrar um timer para
          // tentar novamente.
          TimerTask timerTask = new TimerTask() {
            @Override
            public void run() {
              reloadContainer();
            }
          };
          Timer timer = new Timer("RestController reload timer");
          timer.schedule(timerTask, 300);
        }
      }
    }
  }

  /**
   * Remove uma lista de recursos registrados no servidor REST.
   *
   * @param prefix O prefixo da aplicao para retirada de recursos
   */
  public void removeApplicationResources(String prefix) {
    instancesByPrefix.remove(prefix);

    prefixByResourceClass.entrySet()
      .removeIf(entry -> entry.getValue().equals(prefix));

    shouldReload = true;
    reloadContainer();
  }

  /**
   * Consulta a porta REST usada pelo cliente.
   *
   * @return porta.
   */
  public int getPort() {
    if (httpServer == null || !httpServer.isStarted()) {
      return -999;
    }
    return httpPort;
  }

  /**
   * Metodo para registro de pacotes com recursos no servidor REST.
   *
   * @param packages Lista de packages de recursos a serem registradas.
   */
  public void registerPackages(String... packages) {
    if (httpServer != null && httpServer.isStarted()) {
      httpServer.shutdownNow();
    }
    resourcePackages.addAll(Arrays.asList(packages));
    this.startServer();
  }

  /**
   * Faz log no eventlog do servidor.
   *
   * @param type tipo
   * @param msg mensagem
   */
  private void logMessage(String type, String msg) {
    try {
      ClientRemoteLocator.eventLogService
        .addClientInformation(new String[] { "ClientRest" },
          new String[] { type, msg });
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
  }

  /**
   * Cria a configurao de recursos de acordo com os recursos atuais definidos
   *
   * @return Uma nova configurao de recursos
   */
  private ResourceConfig buildResourceConfig() {
    final ResourceConfig rc = new ResourceConfig();

    //Processador de recursos das instncias de aplicao
    rc.register(ApplicationApiModelProcessor.class);

    //Registro dos recursos REST do cliente
    rc.packages(true,
      resourcePackages.toArray(new String[resourcePackages.size()]));

    //Registra os recursos das aplicaes
    instancesByPrefix.entrySet()
      .forEach(e -> rc.registerInstances(e.getValue().toArray()));

    //Registro de recurso para serializao JSON
    rc.register(JacksonFeature.class);

    //Registro de recursos para tratamento de requisies e respostas
    rc.register(AuthFilter.class);
    rc.register(DebugMapper.class);
    rc.register(ReloadFilter.class);

    //Registro de classes do swagger
    rc.register(io.swagger.jersey.listing.ApiListingResourceJSON.class);
    rc.register(io.swagger.jaxrs.listing.ApiListingResource.class);
    rc.register(io.swagger.jaxrs.listing.SwaggerSerializers.class);
    return rc;
  }

  /**
   * Processador do modelo de recursos, usado para modificar os recursos
   * providos por instncias de aplicao. Os recursos do prprio cliente no
   * so modificados.
   */
  public static class ApplicationApiModelProcessor implements ModelProcessor {

    /**
     * Cria um novo modelo de recursos, separados os recursos comuns a mais de
     * uma instncia de aplicao em recursos prprios, prefixados pelo
     * identificador da instncia.
     *
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processResourceModel(ResourceModel resourceModel,
      Configuration configuration) {
      ResourceModel.Builder newModelBuilder = new ResourceModel.Builder(false);

      for (Resource res : resourceModel.getResources()) {
        boolean isAppResource = false;
        for (Map.Entry<Class<?>, String> prefixByClass : RestController
          .getInstance().prefixByResourceClass.entrySet()) {
          if (res.getHandlerClasses().contains(prefixByClass.getKey())) {
            isAppResource = true;
            Resource newRes = createResource(res, prefixByClass.getValue(),
              prefixByClass.getKey(), true);
            newModelBuilder.addResource(newRes);
          }
        }
        if (!isAppResource) {
          newModelBuilder.addResource(res);
        }
      }

      return newModelBuilder.build();
    }

    /**
     * Cria um novo recurso, adicionando somente os mtodos que pertencem 
     * instncia especificada.
     *
     * @param resource recurso original.
     * @param prefix prefixo da instncia de aplicao.
     * @param clazz a classe que implementa o recurso na instncia.
     * @param isParentResource verdadeiro se o recurso em questo  o recurso
     * pai.
     * @return o novo recurso.
     */
    private Resource createResource(Resource resource, String prefix,
      Class<?> clazz, boolean isParentResource) {
      final Resource.Builder resourceBuilder =
        Resource.builder().name(resource.getName());
      if (isParentResource) {
        resourceBuilder.path("/" + prefix + resource.getPath());
      }
      else {
        resourceBuilder.path(resource.getPath());
      }
      resource.getChildResources().forEach((r) -> resourceBuilder
        .addChildResource(createResource(r, prefix, clazz, false)));

      for (final ResourceMethod resourceMethod : resource
        .getResourceMethods()) {
        Method classMethod = getClassMethod(resourceMethod);
        if (classMethod.getDeclaringClass().equals(clazz)) {
          resourceBuilder.addMethod(resourceMethod);
        }
      }

      return resourceBuilder.build();
    }

    /**
     * Obtm o mtodo da classe que implementa o mtodo do recurso.
     *
     * @param resourceMethod o mtodo do recurso.
     * @return o mtodo da classe.
     */
    private Method getClassMethod(final ResourceMethod resourceMethod) {
      return resourceMethod.getInvocable().getDefinitionMethod();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ResourceModel processSubResource(ResourceModel resourceModel,
      Configuration configuration) {
      return resourceModel;
    }
  }

}
