/*
 * $Id: ProjectTreeModel.java 144791 2013-09-20 19:55:01Z mjulia $
 */
package csbase.client.project;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;

import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.DirRefreshedEvent;
import csbase.logic.NewProjectFileEvent;
import csbase.logic.NewProjectFilesEvent;
import csbase.logic.ProjectDeletedEvent;
import csbase.logic.ProjectEvent;
import csbase.logic.ProjectFileDeletedEvent;
import csbase.logic.ProjectFileFilter;
import csbase.logic.ProjectFileRenamedEvent;
import csbase.logic.ProjectFileStateChangedEvent;
import csbase.logic.ProjectFilesDeletedEvent;
import csbase.logic.ProjectInfoModifiedEvent;
import csbase.logic.TreeChangedEvent;

/**
 * Define um modelo de dados para uma rvore de projetos.
 * 
 * @author Rodrigo Carneiro Henrique (rodrigoh)
 */
public final class ProjectTreeModel implements TreeModel {
  /** O n raz da rvore de projetos. */
  private ProjectTreeNode rootNode;

  /** Coleo de ouvintes de eventos de alteraes do modelo. */
  private Set<TreeModelListener> listenerSet;

  /** Coleo de ouvintes de eventos de alteraes do modelo. */
  private Set<ProjectTreeStructureListener> listenerChangesSet;

  /** O projeto que  representado por este modelo. */
  private CommonClientProject clientProject;

  /**
   * Observador do projeto. Usado pelo modelo para se atualizar em relao a
   * mudanas no projeto.
   */
  private ProjectObserver projectObserver;

  /**
   * O filtro responsvel por definir que arquivos sero representados por esse
   * modelo.
   */
  private ProjectFileFilter filter;

  /**
   * Utilizado para ordenar a rvore. {@link ProjectTreeNodeExtensionComparator}
   * {@link ProjectTreeNodeNameComparator}
   */
  private Comparator<ProjectTreeNode> comparator;

  /**
   * Cria um modelo para a rvore de projetos que ter um filtro nulo, ou seja,
   * todos os arquivos do projeto sero representados na rvore. Alm disso,
   * ser utilizado um comparador que ordenar a rvore pelos nomes dos
   * arquivos.
   * 
   * @param clientProject O projeto que ser representado pelo modelo.
   */
  public ProjectTreeModel(CommonClientProject clientProject) {
    this(clientProject, null, ProjectTreeNodeNameComparator.getInstance());
  }

  /**
   * Cria um modelo para a rvore de projetos que ter um filtro nulo, ou seja,
   * todos os arquivos do projeto sero representados na rvore.
   * 
   * @param clientProject O projeto que ser representado pelo modelo.
   * @param comparator O comparador que ser utilizado para ordenar a rvore de
   *        projeto.
   */
  public ProjectTreeModel(CommonClientProject clientProject,
    Comparator<ProjectTreeNode> comparator) {
    this(clientProject, null, comparator);
  }

  /**
   * Cria um modelo para a rvore de projetos que ser ordenada pelos nomes dos
   * arquivos.
   * 
   * @param clientProject O projeto que ser representado pelo modelo.
   * @param filter O filtro que definir que arquivos de projeto sero
   *        representados pelo modelo.
   */
  public ProjectTreeModel(CommonClientProject clientProject,
    ProjectFileFilter filter) {
    this(clientProject, filter, ProjectTreeNodeNameComparator.getInstance());
  }

  /**
   * Cria um modelo para a rvore de projetos.
   * 
   * @param clientProject O projeto que ser representado pelo modelo.
   * @param filter O filtro que definir que arquivos de projeto sero
   *        representados pelo modelo.
   * @param comparator O comparador que ser utilizado para ordenar a rvore de
   *        projeto.
   */
  public ProjectTreeModel(CommonClientProject clientProject,
    ProjectFileFilter filter, Comparator<ProjectTreeNode> comparator) {
    this.listenerSet = new HashSet<TreeModelListener>();
    this.listenerChangesSet = new HashSet<ProjectTreeStructureListener>();
    this.projectObserver = new ProjectObserver();
    this.filter = filter;
    this.comparator = comparator;
    this.setProject(clientProject);
  }

  /**
   * Altera o modelo para que passe a representar um novo projeto.
   * 
   * @param clientProject O novo projeto que ser representado pelo modelo.
   */
  void setProject(CommonClientProject clientProject) {
    if (rootNode != null) {
      fireTreeWillChange(rootNode.getTreePath());
    }
    if (clientProject != null) {
      this.rootNode =
        new ProjectTreeNode(clientProject, this.filter, this.comparator);
      clientProject.addObserver(this.projectObserver);
    }
    else {
      this.rootNode = null;
    }
    if (this.clientProject != null) {
      this.clientProject.deleteObserver(this.projectObserver);
    }
    this.clientProject = clientProject;
    if (this.rootNode != null) {
      this.fireTreeStructureChanged(this.rootNode.getTreePath());
    }
    else {
      this.fireTreeStructureChanged(null);
    }
  }

  /**
   * Consulta o valor de clientProject
   * 
   * @return o valor
   */
  public final CommonClientProject getProject() {
    return clientProject;
  }

  /**
   * Altera o modelo para que passe a ser ordenado pelo novo comparador.
   * 
   * @param comparator O novo comparador do modelo.
   */
  void setComparator(Comparator<ProjectTreeNode> comparator) {
    if (rootNode != null) {
      fireTreeWillChange(rootNode.getTreePath());
    }
    this.comparator = comparator;
    if (rootNode != null) {
      rootNode.setComparator(comparator);
      fireTreeStructureChanged(this.rootNode.getTreePath());
    }
  }

  /**
   * Adiciona um listener de eventos de alterao na estrutura da rvore.
   * 
   * @param l O listener.
   */
  public void addTreeWillChangeStructureListener(ProjectTreeStructureListener l) {
    this.listenerChangesSet.add(l);
  }

  /**
   * Remove um listener de eventos de alterao na estrutura da rvore.
   * 
   * @param l O listener.
   */
  public void removeTreeWillChangeStructureListener(
    ProjectTreeStructureListener l) {
    this.listenerChangesSet.remove(l);
  }

  /**
   * Envia o evento de alterao na estrutura de uma sub-rvore para todos os
   * ouvintes cadastrados.
   * 
   * @param subTreeRoot A raiz da sub-rvore que teve sua estrutura alterada.
   */
  void fireTreeStructureChanged(TreePath subTreeRoot) {
    TreeModelEvent treeModelEvent = new TreeModelEvent(this, subTreeRoot);
    Iterator<TreeModelListener> listenerIterator = this.listenerSet.iterator();
    while (listenerIterator.hasNext()) {
      TreeModelListener treeModelListener = listenerIterator.next();
      treeModelListener.treeStructureChanged(treeModelEvent);
    }
    this.fireTreeWasChanged(subTreeRoot);
  }

  /**
   * Envia o evento de que a estrutura da rvore vai ser alterada. Esse evento
   * s deve ser disparado <b>antes</b> de uma alterao.
   * 
   * @param subTreeRoot A raiz da sub-rvore que teve sua estrutura alterada.
   */
  void fireTreeWillChange(TreePath subTreeRoot) {
    TreeModelEvent event = new TreeModelEvent(this, subTreeRoot);
    Iterator<ProjectTreeStructureListener> listenerChangesIterator =
      this.listenerChangesSet.iterator();
    while (listenerChangesIterator.hasNext()) {
      ProjectTreeStructureListener listener = listenerChangesIterator.next();
      listener.treeWillChange(event);
    }
  }

  /**
   * Envia o evento de que a estrutura da rvore foi alterada.
   * 
   * @param subTreeRoot A raiz da sub-rvore que teve sua estrutura alterada.
   */
  private void fireTreeWasChanged(TreePath subTreeRoot) {
    TreeModelEvent event = new TreeModelEvent(this, subTreeRoot);
    Iterator<ProjectTreeStructureListener> listenerChangesIterator =
      this.listenerChangesSet.iterator();
    while (listenerChangesIterator.hasNext()) {
      ProjectTreeStructureListener listener = listenerChangesIterator.next();
      listener.treeWasChanged(event);
    }
  }

  /**
   * Envia o evento de insero de ns para todos os ouvintes cadastrados.
   * 
   * @param parentTreePath O n-raz (pai) onde os ns foram inseridos.
   * @param insertedIndexes Os ndices dos ns na lista de filhos do n pai.
   * @param insertedNodes Os ns que foram inseridos.
   */
  private void fireTreeNodesInserted(TreePath parentTreePath,
    int[] insertedIndexes, ProjectTreeNode[] insertedNodes) {
    this.sort(insertedIndexes, insertedNodes);
    TreeModelEvent treeModelEvent =
      new TreeModelEvent(this, parentTreePath, insertedIndexes, insertedNodes);
    Iterator<TreeModelListener> listenerIterator = this.listenerSet.iterator();
    while (listenerIterator.hasNext()) {
      TreeModelListener treeModelListener = listenerIterator.next();
      try {
        treeModelListener.treeNodesInserted(treeModelEvent);
      }
      catch (Throwable t) {
        t.printStackTrace();
      }
    }
  }

  /**
   * Envia o evento de remoo de ns para todos os ouvintes cadastrados.
   * 
   * @param parentTreePath O n-raz (pai) de onde os ns foram removidos.
   * @param removedIndexes Os ndices dos ns na lista de filhos do n pai.
   * @param removedNodes Os ns que foram removidos.
   */
  private void fireTreeNodesRemoved(TreePath parentTreePath,
    int[] removedIndexes, ProjectTreeNode[] removedNodes) {
    this.sort(removedIndexes, removedNodes);
    TreeModelEvent treeModelEvent =
      new TreeModelEvent(this, parentTreePath, removedIndexes, removedNodes);
    Iterator<TreeModelListener> listenerIterator = this.listenerSet.iterator();
    while (listenerIterator.hasNext()) {
      TreeModelListener treeModelListener = listenerIterator.next();
      try {
        treeModelListener.treeNodesRemoved(treeModelEvent);
      }
      catch (Throwable t) {
        t.printStackTrace();
      }
    }
  }

  /**
   * Envia o evento de alterao de ns para todos os ouvintes cadastrados.
   * 
   * @param parentTreePath O n-raz (pai) de onde os ns foram alterados.
   * @param changedIndexes Os ndices dos ns na lista de filhos do n pai.
   * @param changedNodes Os ns que foram alterados.
   */
  private void fireTreeNodesChanged(TreePath parentTreePath,
    int[] changedIndexes, ProjectTreeNode[] changedNodes) {
    this.sort(changedIndexes, changedNodes);
    TreeModelEvent treeModelEvent =
      new TreeModelEvent(this, parentTreePath, changedIndexes, changedNodes);
    Iterator<TreeModelListener> listenerIterator = this.listenerSet.iterator();
    while (listenerIterator.hasNext()) {
      TreeModelListener treeModelListener = listenerIterator.next();
      try {
        treeModelListener.treeNodesChanged(treeModelEvent);
      }
      catch (Throwable t) {
        t.printStackTrace();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getRoot() {
    return this.rootNode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getChildCount(Object parent) {
    ProjectTreeNode projectTreeNode = (ProjectTreeNode) parent;
    return projectTreeNode.getChildCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLeaf(Object node) {
    ProjectTreeNode projectTreeNode = (ProjectTreeNode) node;
    return projectTreeNode.isLeaf();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTreeModelListener(TreeModelListener l) {
    this.listenerSet.add(l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeTreeModelListener(TreeModelListener l) {
    this.listenerSet.remove(l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getChild(Object parent, int index) {
    ProjectTreeNode projectTreeNode = (ProjectTreeNode) parent;
    return projectTreeNode.getChildAt(index);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getIndexOfChild(Object parent, Object child) {
    ProjectTreeNode projectTreeNode = (ProjectTreeNode) parent;
    return projectTreeNode.getIndex((ProjectTreeNode) child);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void valueForPathChanged(TreePath path, Object newValue) {
  }

  /**
   * Retorna o caminho na rvore onde um determinado arquivo se encontra.
   * 
   * @param file O arquivo do qual se deseja o caminho na rvore.
   * 
   * @return O caminho para o arquivo na rvore, ou null, caso o arquivos no
   *         exista na rvore.
   */
  public TreePath getTreePath(ClientProjectFile file) {
    String[] path = file.getPath();
    ProjectTreeNode[] pathToRootArray = new ProjectTreeNode[path.length + 1];
    ProjectTreeNode node = this.rootNode;
    pathToRootArray[0] = node;
    for (int i = 0; i < path.length; i++) {
      node = node.getChild(path[i]);
      if (node == null) {
        return null;
      }
      pathToRootArray[i + 1] = node;
    }
    return new TreePath(pathToRootArray);
  }

  /**
   * Verifica se o path existe no modelo.
   * 
   * @param path O path a ser verificado.
   * 
   * @return true, se o path existe no modelo, ou false, caso contrrio.
   */
  public boolean exists(TreePath path) {
    if (this.rootNode == null) {
      return false;
    }
    Object[] pathComponents = path.getPath();
    ProjectTreeNode cursor = rootNode;
    /*
     * a partir da raiz verificamos se cada um dos nveis do path especificado
     * existem
     */
    for (int i = 0; i < pathComponents.length; i++) {
      ProjectTreeNode target = (ProjectTreeNode) pathComponents[i];
      if (!target.equals(cursor)) {
        return false;
      }
      cursor = cursor.getChild(target.getName());
    }
    return true;
  }

  /**
   * Responsvel por observar mudanas no projeto e atualizar o modelo em
   * relao  essas mudanas.
   * 
   * @author Rodrigo Carneiro Henrique (rodrigoh)
   */
  private class ProjectObserver implements Observer {
    /**
     * {@inheritDoc}
     */
    @Override
    public void update(Observable o, Object arg) {
      final ProjectEvent event = (ProjectEvent) arg;
      Runnable code = new Runnable() {
        @Override
        public void run() {
          try {
            switch (event.event) {
              case ProjectEvent.INFO_MODIFIED:
                handleEvent((ProjectInfoModifiedEvent) event);
                break;
              case ProjectEvent.PROJECT_DELETED:
                handleEvent((ProjectDeletedEvent) event);
                break;
              case ProjectEvent.NEW_FILE:
                handleEvent((NewProjectFileEvent) event);
                break;
              case ProjectEvent.NEW_FILES:
                handleEvent((NewProjectFilesEvent) event);
                break;
              case ProjectEvent.FILE_DELETED:
                handleEvent((ProjectFileDeletedEvent) event);
                break;
              case ProjectEvent.FILES_DELETED:
                handleEvent((ProjectFilesDeletedEvent) event);
                break;
              case ProjectEvent.NEW_FILE_NAME:
                handleEvent((ProjectFileRenamedEvent) event);
                break;
              case ProjectEvent.FILE_STATE_CHANGED:
                handleEvent((ProjectFileStateChangedEvent) event);
                break;
              case ProjectEvent.DIR_REFRESHED:
                handleEvent((DirRefreshedEvent) event);
                break;
              case ProjectEvent.TREE_CHANGED:
                handleEvent((TreeChangedEvent) event);
                break;
              default:
                throw new IllegalStateException("Evento desconhecido. Nmero: "
                  + event.event + " - Classe: " + event.getClass());
            }
          }
          catch (Throwable t) {
            t.printStackTrace();
          }
        }
      };
      if (SwingThreadDispatcher.isEventDispatchThread()) {
        code.run();
      }
      else {
        SwingThreadDispatcher.invokeLater(code);
      }
    }
  }

  /**
   * Trata o evento de modificao nas informaes do projeto.
   * 
   * @param event informaes sobre o evento de modificao das informaes do
   *        projeto
   */
  private void handleEvent(ProjectInfoModifiedEvent event) {
    // Nada a fazer. As informaes do projeto no afetam a rvore.
  }

  /**
   * Trata o evento de remoo do projeto corrente.
   * 
   * @param event informaes sobre o evento de remoo de um projeto
   */
  private void handleEvent(ProjectDeletedEvent event) {
    Runnable handler = new Runnable() {
      @Override
      public void run() {
        ProjectTreeModel.this.setProject(null);
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Trata o evento de insero de um novo arquivo no modelo.
   * 
   * @param event informaes sobre o evento de criao de arquivo
   */
  private void handleEvent(NewProjectFileEvent event) {
    final ClientProjectFile insertedFile = event.getFile();
    Runnable handler = new Runnable() {
      @Override
      public void run() {
        addNewNode(insertedFile);
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Trata o evento de arquivos criados no servidor, atualizando a rvore no
   * cliente com os novos arquivos.
   * 
   * @param event informaes sobre o evento de criao de arquivos.
   */
  private void handleEvent(NewProjectFilesEvent event) {
    final ClientProjectFile[] files = event.getFiles();
    if (files == null) {
      throw new IllegalArgumentException("files == null");
    }

    Runnable handler = new Runnable() {
      @Override
      public void run() {
        for (int inx = 0; inx < files.length; inx++) {
          addNewNode(files[inx]);
        }
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Adiciona um novo n na rvore de projeto.
   * 
   * @param file Arquivo representando o novo n a ser inserido.
   */
  private void addNewNode(ClientProjectFile file) {
    // Sempre existe um pai, nem que seja a raiz:
    TreePath parentTreePath = getTreePath(file.getParent());
    if (parentTreePath == null) {
      return;
    }
    ProjectTreeNode parentTreeNode =
      (ProjectTreeNode) parentTreePath.getLastPathComponent();
    boolean wasLeaf = parentTreeNode.isLeaf();
    if (!parentTreeNode.addChild(file)) {
      return;
    }
    // Se o pai era uma folha (porque estava vazio),  necessrio disparar um
    // evento de mudana de estrutura para o pai dele. Note que nesse caso o
    // evento treeWillChange pode ser enviado aps a alterao ter sido feita
    // porque a incluso de um novo n no afeta o processamento feito para
    // manter a seleo e demais aspectos visuais.
    if (wasLeaf) {
      ClientProjectFile grandDir = file.getParent().getParent();
      if (grandDir == null) {
        // Nesse caso, o evento foi na raiz e ela estava vazia. Usa a raiz
        // como base da mudana estrutural.
        grandDir = file.getParent();
      }
      TreePath grandTreePath = getTreePath(grandDir);
      fireTreeWillChange(grandTreePath);
      fireTreeStructureChanged(grandTreePath);
    }
    // Se o pai no era uma folha, basta disparar o evento de incluso de n.
    else {
      ProjectTreeNode treeNode = parentTreeNode.getChild(file);
      ProjectTreeNode[] insertedNodes = new ProjectTreeNode[] { treeNode };
      int[] insertedIndexes = parentTreeNode.getIndexes(insertedNodes);
      fireTreeNodesInserted(parentTreePath, insertedIndexes, insertedNodes);
    }
  }

  /**
   * Trata o evento de remoo de um arquivo.
   * 
   * @param event informaes sobre o evento de remoo de arquivo
   */
  private void handleEvent(ProjectFileDeletedEvent event) {
    final ClientProjectFile removedFile = event.getRemovedFile();
    final TreePath parentTreePath = getTreePath(removedFile.getParent());
    if (parentTreePath == null) {
      return;
    }
    final ProjectTreeNode parentTreeNode =
      (ProjectTreeNode) parentTreePath.getLastPathComponent();
    final TreePath fileTreePath = getTreePath(removedFile);
    if (fileTreePath == null) {
      return;
    }

    Runnable handler = new Runnable() {
      @Override
      public void run() {
        ProjectTreeNode fileTreeNode =
          (ProjectTreeNode) fileTreePath.getLastPathComponent();
        ProjectTreeNode[] removedNodes = new ProjectTreeNode[] { fileTreeNode };
        int[] removedIndexes = parentTreeNode.getIndexes(removedNodes);
        if (!parentTreeNode.removeChild(removedFile)) {
          return;
        }
        fireTreeNodesRemoved(parentTreePath, removedIndexes, removedNodes);
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Trata o evento de remoo de vrios arquivos.
   * 
   * @param event informaes sobre o evento de remoo de vrios arquivos
   */
  private void handleEvent(ProjectFilesDeletedEvent event) {
    ClientProjectFile[] removedFiles = event.getRemovedFiles();
    final Map<TreePath, List<TreePath>> parent2Children =
      new HashMap<TreePath, List<TreePath>>();

    for (ClientProjectFile file : removedFiles) {
      TreePath parentTreePath = getTreePath(file.getParent());
      if (parentTreePath == null) {
        continue;
      }
      TreePath fileTreePath = getTreePath(file);
      if (fileTreePath == null) {
        continue;
      }
      List<TreePath> list = parent2Children.get(parentTreePath);
      if (list == null) {
        list = new ArrayList<TreePath>();
        parent2Children.put(parentTreePath, list);
      }
      list.add(fileTreePath);
    }

    if (parent2Children.size() == 0) {
      return;
    }

    Runnable handler = new Runnable() {
      @Override
      public void run() {
        for (TreePath parent : parent2Children.keySet()) {
          List<TreePath> children = parent2Children.get(parent);
          ProjectTreeNode[] nodes = new ProjectTreeNode[children.size()];
          for (int i = 0; i < nodes.length; i++) {
            nodes[i] = (ProjectTreeNode) children.get(i).getLastPathComponent();
          }
          ProjectTreeNode parentNode =
            (ProjectTreeNode) parent.getLastPathComponent();
          int[] removedIndexes = parentNode.getIndexes(nodes);
          for (int i = 0; i < nodes.length; i++) {
            parentNode.removeChild(nodes[i].getClientProjectFile());
          }
          fireTreeNodesRemoved(parent, removedIndexes, nodes);
        }
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Trata o evento de novo nome para um arquivo.
   * 
   * @param event informaes sobre o evento de novo nome para um arquivo
   */
  private void handleEvent(ProjectFileRenamedEvent event) {
    updateFile(event.getFile());
  }

  /**
   * Trata o evento de alterao do estado do arquivo.
   * 
   * @param event informaes sobre o evento de alterao do estado de um
   *        arquivo
   * @see ProjectFileStateChangedEvent
   */
  private void handleEvent(ProjectFileStateChangedEvent event) {
    updateFile(event.getFile());
  }

  /**
   * Trata o evento de atualizao de diretrio. Atualiza o n da rvore
   * representando o diretrio selecionado com as informaes oriundas do
   * servidor.
   * 
   * @param event informaes sobre o evento (caminho e diretrio atualizado).
   */
  private void handleEvent(DirRefreshedEvent event) {
    updateFile(event.getDir());
  }

  /**
   * Trata genericamente uma alterao em um arquivo.
   * 
   * @param changedFile O arquivo que foi alterado.
   */
  private void updateFile(final ClientProjectFile changedFile) {
    final TreePath parentTreePath = getTreePath(changedFile.getParent());
    if (parentTreePath == null) {
      return;
    }
    final TreePath treePath = getTreePath(changedFile);
    if (treePath == null) {
      return;
    }

    Runnable handler = new Runnable() {
      @Override
      public void run() {
        fireTreeWillChange(treePath);
        ProjectTreeNode treeNode =
          (ProjectTreeNode) treePath.getLastPathComponent();
        treeNode.setClientProjectFile(changedFile);
        fireTreeStructureChanged(parentTreePath);
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Trata o evento de refresh na rvore.
   * 
   * @param event informaes sobre o evento de refresh na rvore
   */
  private void handleEvent(final TreeChangedEvent event) {
    if (rootNode == null) {
      return;
    }

    Runnable handler = new Runnable() {
      @Override
      public void run() {
        fireTreeWillChange(rootNode.getTreePath());
        rootNode.setClientProjectFile(event.getNewTree());
        fireTreeStructureChanged(rootNode.getTreePath());
      }
    };

    // Executa o tratador do evento na EDT.
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      handler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(handler);
    }
  }

  /**
   * Ordena um array de ndices e seu respectivo array de ns.
   * <p>
   * Os eventos de alterao no modelo que recebem ndice, precisam receber
   * esses ndices ordenados. Os ns devem manter sua posio de array de acordo
   * com a posio de seu ndice.
   * </p>
   * 
   * @param indexes O array com os ndices.
   * @param nodes O array com os ns.
   * 
   * @throws IllegalArgumentException Caso os arrays possuam tamanhos
   *         diferentes.
   */
  private void sort(int[] indexes, ProjectTreeNode[] nodes) {
    if (indexes.length != nodes.length) {
      throw new IllegalArgumentException(
        "Os arrays no podem ter tamanhos diferentes.");
    }
    for (int i = 0; i < indexes.length - 1; i++) {
      for (int j = i + 1; j < indexes.length; j++) {
        if (indexes[i] > indexes[j]) {
          int auxInt = indexes[i];
          indexes[i] = indexes[j];
          indexes[j] = auxInt;
          ProjectTreeNode auxNode = nodes[i];
          nodes[i] = nodes[j];
          nodes[j] = auxNode;
        }
      }
    }
  }

  /**
   * Obtm o filtro utilizado para definir que arquivos podero fazer parte do
   * modelo.
   * 
   * @return O filtro do modelo.
   */
  ProjectFileFilter getFilter() {
    return this.filter;
  }

  /**
   * Altera o filtro do modelo.
   * 
   * @param filter O novo filtro do modelo.
   */
  void setFilter(ProjectFileFilter filter) {
    if (rootNode != null) {
      fireTreeWillChange(rootNode.getTreePath());
    }
    this.filter = filter;
    if (this.rootNode != null) {
      rootNode.setFilter(filter);
      this.fireTreeStructureChanged(this.rootNode.getTreePath());
    }
  }
}
