package csbase.client.util;

import java.util.logging.Level;
import java.util.logging.LogRecord;

/**
 * Classe utilitria que facilita a criao de {@link LogRecord} para logar o
 * incio e fim de um bloco de cdigo.
 * 
 * @author Tecgraf
 */
public class CodeBlockLog {

  /**
   * Modelo de mensagem a ser logada no incio de bloco.
   */
  private static final String BEGIN_LOG_MESSAGE = "[Thread:%s][BEGIN] %s";
  /**
   * Modelo de mensagem a ser logada no fim de um bloco.
   */
  private static final String FINISHED_LOG_MESSAGE =
    "[Thread:%s][FINISHED:%.3fs] %s";

  /**
   * Nvel do log.
   */
  private Level level;
  /**
   * Informao a ser logada.
   */
  private String message;
  /**
   * Data de incio do bloco.
   */
  private long begin;

  /**
   * Construtor.
   * 
   * @param level Nvel do log.
   * @param messageTemplate modelo de informao a ser logada no formato
   *        {@link String#format(String, Object...)}.
   * @param args argumentos da informao a ser logada.
   */
  public CodeBlockLog(Level level, String messageTemplate, Object... args) {
    this.level = level;
    this.message = String.format(messageTemplate, args);
  }

  /**
   * Cria um {@link LogRecord} contendo uma mensagem indicando o incio de um
   * bloco.
   * 
   * @return um {@link LogRecord} contendo uma mensagem indicando o incio de um
   *         bloco.
   */
  public LogRecord begin() {
    begin = System.currentTimeMillis();
    return new LogRecord(level, String.format(BEGIN_LOG_MESSAGE, Thread
      .currentThread().getName(), message));
  }

  /**
   * Cria um {@link LogRecord} contendo uma mensagem indicando o fim de um
   * bloco.
   * 
   * @return um {@link LogRecord} contendo uma mensagem indicando o incio de um
   *         bloco.
   */
  public LogRecord finished() {
    return new LogRecord(level, String.format(FINISHED_LOG_MESSAGE, Thread
      .currentThread().getName(),
      (System.currentTimeMillis() - begin) * 1.0 / 1000.0, message));
  }
}
