/*
 * $Id$
 */
package csbase.client.applications.filetransferclient;

import java.util.ArrayList;
import java.util.List;

import csbase.client.applications.ApplicationFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.exception.ServiceFailureException;
import csbase.logic.User;
import csbase.logic.filetransferservice.FileTransferConnection;
import csbase.logic.filetransferservice.FileTransferElement;
import csbase.logic.filetransferservice.FileTransferPredefinedConnection;
import csbase.logic.filetransferservice.FileTransferRequest;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.FileTransferServiceInterface;

/**
 * Classe que modela um proxy para o {@link FileTransferServiceInterface}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FileTransferProxy {

  /**
   * Aplicao.
   */
  final private FileTransferClient application;

  /**
   * Indicativo de conexo acessvel.
   * 
   * @param connection conexo.
   * @return indicativo
   */
  public boolean checkConnection(final FileTransferConnection connection) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return false;
    }

    final RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        final boolean check = service.checkConnection(connection);
        setResult(check);
      }

      @Override
      protected void handleError(Exception error) {
        setResult(false);
        // No faz nada, pois o mtodo retornar false,
        // o que vai indicar falha na conexo.
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    final String userName = connection.getUserName();
    final String serverName = connection.getServerName();
    final String connectionText = userName + "@" + serverName;
    task.execute(frame, "", connectionText);

    final Exception exception = task.getError();
    if (exception != null) {
      return false;
    }

    if (task.wasCancelled()) {
      return false;
    }

    final Boolean result = task.getResult();
    return result;
  }

  /**
   * Consulta o service {@link FileTransferServiceInterface}
   * 
   * @return o servico ou null
   */
  static public FileTransferServiceInterface getFileTransferService() {
    final FileTransferServiceInterface service =
      ClientRemoteLocator.fileTransferService;
    return service;
  }

  /**
   * Consulta o service {@link FileTransferServiceInterface}
   * 
   * @return o servico ou null
   */
  private FileTransferServiceInterface getService() {
    final FileTransferServiceInterface service = getFileTransferService();
    if (service == null) {
      final String tag = "no.service.found.msg";
      final String prefix = this.getClass().getSimpleName();
      final String err = application.getString(prefix + "." + tag);
      application.showError(err);
      return null;
    }
    return service;
  }

  /**
   * Download
   * 
   * @param connection conexo
   * @param remotePath path
   * @param localPath path
   * @param size tamanho.
   */
  public void download(final FileTransferConnection connection,
    final String remotePath, final String[] localPath, final long size) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return;
    }

    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        service.download(connection, remotePath, localPath, size);
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", "\u2190 " + remotePath);
  }

  /**
   * Upload
   * 
   * @param connection conexo
   * @param remotePath path
   * @param localPath path
   * @param size tamanho.
   */
  public void upload(final FileTransferConnection connection,
    final String remotePath, final String[] localPath, final long size) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return;
    }

    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        service.upload(connection, remotePath, localPath, size);
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", "\u2192 " + remotePath);
  }

  /**
   * Lista um diretrio remoto.
   * 
   * @param connection a conexo
   * @param remoteDirPath nome do arquivo remoto usando o separador padro.
   * 
   * @return a lista
   * @throws ServiceFailureException em caso de falha.
   */
  public List<FileTransferElement> listContent(
    final FileTransferConnection connection, final String remoteDirPath) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return new ArrayList<FileTransferElement>();
    }

    final RemoteTask<List<FileTransferElement>> task =
      new RemoteTask<List<FileTransferElement>>() {
        @Override
        protected void performTask() throws Exception {
          final List<FileTransferElement> list =
            service.listContent(connection, remoteDirPath);
          setResult(list);
        }
      };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", remoteDirPath);

    final Exception exception = task.getError();
    if (exception != null) {
      return new ArrayList<FileTransferElement>();
    }

    final List<FileTransferElement> result = task.getResult();
    return result;
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public FileTransferProxy(FileTransferClient application) {
    this.application = application;
  }

  /**
   * Lista requisies do usurio ao serivo
   * 
   * @param user usurio
   * @return a lista
   * @see FileTransferServiceInterface#getAllRequests(User)
   */
  public final List<FileTransferRequest> getAllRequests(final User user) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return new ArrayList<FileTransferRequest>();
    }

    final RemoteTask<List<FileTransferRequest>> task =
      new RemoteTask<List<FileTransferRequest>>() {
        @Override
        protected void performTask() throws Exception {
          final List<FileTransferRequest> list = service.getAllRequests(user);
          setResult(list);
        }
      };
    task.setProgressDialogDelay(10);
    final ApplicationFrame frame = application.getApplicationFrame();
    final String userName = user != null ? user.getName() : "-";
    task.execute(frame, "", userName);

    final Exception exception = task.getError();
    if (exception != null) {
      return new ArrayList<FileTransferRequest>();
    }

    final List<FileTransferRequest> result = task.getResult();
    return result;
  }

  /**
   * Faz cancelamento.
   * 
   * @param requests lista de requisices.
   */
  public void interruptRequests(final List<String> requests) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return;
    }

    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        service.interruptRequest(requests);
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", "XX");

    final Exception exception = task.getError();
    if (exception != null) {
      return;
    }
  }

  /**
   * Faz cancelamento.
   * 
   * @param requests lista de requisices.
   */
  public void removeRequests(final List<String> requests) {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return;
    }

    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        service.removeRequest(requests);
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", "---");

    final Exception exception = task.getError();
    if (exception != null) {
      return;
    }
  }

  /**
   * Busca o nome do servidor para exibio.
   * @return o nome
   */
  public static String getServerName() {
    String serverName = null;
    final RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      public void performTask() throws Exception {
        setResult(ClientRemoteLocator.server.getSystemName());
      }
    };
    // Esse dilogo no deve aparecer rapidamente pois dar uma impresso
    // ruim ao usurio no start-up. Ele somente aparecer caso
    // haja alguma demora real na conexo com o servidor.
    task.setProgressDialogDelay(10);
    task.execute(DesktopFrame.getInstance().getDesktopFrame(), "", "... ...");
    if (!task.getStatus()) {
      serverName = "ERROR";
    }
    serverName = task.getResult();
    if (serverName == null) {
      serverName = "NULL";
    }
    return serverName;
  }
  
  /**
   * Consulta a lista (j adaptada) de conexes predefinidas.
   * 
   * @return a lista
   */
  public List<FileTransferPredefinedConnection> getPedefinedConnections() {
    final FileTransferServiceInterface service = getService();
    if (service == null) {
      return new ArrayList<FileTransferPredefinedConnection>();
    }

    final RemoteTask<List<FileTransferPredefinedConnection>> task = new RemoteTask<List<FileTransferPredefinedConnection>>() {
      @Override
      protected void performTask() throws Exception {
        setResult(service.getPredefinedConnections());
      }
    };
    final ApplicationFrame frame = application.getApplicationFrame();
    task.execute(frame, "", "---");

    final Exception exception = task.getError();
    if (exception != null) {
      return null;
    }
    return task.getResult();
  }
}
