/*
 * $Id$
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Observable;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;

import csbase.client.util.SwingObserver;
import csbase.logic.AdministrationEvent;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * A classe <code>ManagerPanel</code> representa um painel de administrao de
 * informaes do sistema. Os itens gerenciados so apresentados em uma tabela
 * com seleo simples e que permite a ordenao a partir da seleo da coluna
 * no cabealho. Possui botes para incluso, alterao e remoo.  um
 * observador de mudanas em janelas-filhas de cadastro.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public abstract class ManagerPanel extends JPanel implements SwingObserver {
  /**
   * ndice da coluna que representa o identificador do objeto-linha, a qual 
   * invisvel para o usurio. IMPORTANTE: O modelo fornecido deve estar
   * compatvel com esta premissa.
   */
  protected static final int IDENTIFIER_INDEX = 0;

  /**
   * ndice da coluna que serve de base para a ordenao inicial da tabela
   */
  private static final int INITIAL_SORTED_COLUMN_INDEX = 1;

  /**
   * tabela ordenvel, listando os itens atualmente persistidos em arquivo
   */
  private SortableTable table;

  /**
   * boto de alterao
   */
  private JButton modifyButton;

  /**
   * boto de remoo
   */
  private JButton removeButton;

  /**
   * boto de adio
   */
  private JButton addButton;

  /**
   * Listener de seleo.
   */
  private EnablerListSelectionListener selectionListener;

  /**
   * Atualiza a lista interna de itens e recarrega a tabela, para que esta
   * reflita eventuais modificaes na exibio dos elementos.
   * 
   * @param items lista de itens modificada
   */
  protected void setItems(Vector items) {
    if (items == null) {
      throw new IllegalArgumentException("items == null");
    }
    if (table == null) {
      throw new IllegalStateException("table == null");
    }
    ObjectTableModel model = (ObjectTableModel) table.getModel();
    model.setRows(items);
  }

  /**
   * Cria um painel de gerncia de dados no sistema, mostrando os itens em uma
   * tabela, cujo modelo  passado como parmetro.
   * 
   * @param model modelo de dados para a tabela. IMPORTANTE: assume-se que a
   *        primeira coluna do modelo corresponde ao identificador da linha, e
   *        portanto a mesma ser invisvel para o usurio.
   */
  public void make(TableModel model) {
    make(model, null, false);
  }

  /**
   * Cria um painel de gerncia de dados no sistema, mostrando os itens em uma
   * tabela, cujo modelo  passado como parmetro.
   * 
   * @param model modelo de dados para a tabela. IMPORTANTE: assume-se que a
   *        primeira coluna do modelo corresponde ao identificador da linha, e
   *        portanto a mesma ser invisvel para o usurio.
   * @param hasSearchField indica se o painel ter campo de busca nas colunas.
   */
  public void make(TableModel model, boolean hasSearchField) {
    make(model, null, hasSearchField);
  }

  /**
   * Cria um painel de gerncia de dados no sistema, mostrando os itens em uma
   * tabela, cujo modelo  passado como parmetro.
   * 
   * @param model modelo de dados para a tabela. IMPORTANTE: assume-se que a
   *        primeira coluna do modelo corresponde ao identificador da linha, e
   *        portanto a mesma ser invisvel para o usurio.
   * @param comparators Array de comparadores de cada coluna.
   * @param hasSearchField indica se o painel ter campo de busca nas colunas.
   * 
   */
  public void make(TableModel model, Comparator[] comparators,
    boolean hasSearchField) {
    if (model == null) {
      throw new IllegalArgumentException("model == null");
    }
    setLayout(new BorderLayout());
    if (comparators == null) {
      add(makeTable(model), BorderLayout.CENTER);
    }
    else {
      add(makeTable(model, comparators), BorderLayout.CENTER);
    }
    add(makeButtons(hasSearchField), BorderLayout.SOUTH);
  }

  /**
   * Constri uma tabela de dados cadastrados no sistema. As linhas da tabela
   * so ordenadas de acordo com a coluna selecionada. A tabela oferece seleo
   * simples e permite fazer drag and drop das colunas para mudar sua posio.
   * 
   * @param model Modelo da tabela.
   * 
   * @return a tabela com informaes sobre dados cadastrados no sistema.
   */
  protected JComponent makeTable(TableModel model) {
    table = new SortableTable(model);
    table.removeColumn(table.getColumnModel().getColumn(IDENTIFIER_INDEX));
    table.sort(INITIAL_SORTED_COLUMN_INDEX, SortOrder.ASCENDING);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setPreferredScrollableViewportSize(new Dimension(500, 70));
    table.getSelectionModel().addListSelectionListener(selectionListener);
    JScrollPane scrollpane = new JScrollPane(table);
    return scrollpane;
  }

  /**
   * Constri uma tabela de dados cadastrados no sistema. As linhas da tabela
   * so ordenadas de acordo com a coluna selecionada. A tabela oferece seleo
   * simples e permite fazer drag and drop das colunas para mudar sua posio.
   * 
   * @param model Modelo da tabela.
   * @param comparators Array com o Comparator de cada coluna.
   * 
   * @return a tabela com informaes sobre dados cadastrados no sistema.
   */
  protected JComponent makeTable(TableModel model, Comparator[] comparators) {
    table = new SortableTable(model);
    table.sort(INITIAL_SORTED_COLUMN_INDEX, SortOrder.ASCENDING);
    table.setComparators(comparators);
    table.removeColumn(table.getColumnModel().getColumn(IDENTIFIER_INDEX));
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setPreferredScrollableViewportSize(new Dimension(500, 70));
    table.getSelectionModel().addListSelectionListener(selectionListener);
    JScrollPane scrollpane = new JScrollPane(table);
    return scrollpane;
  }

  /**
   * Constri os botes do painel de gerncia de usurios. Os botes representam
   * as seguintes aes:
   * <ul>
   * <li>Incluso</li>
   * <li>Alterao</li>
   * <li>Remoo</li>
   * </ul>
   * 
   * @return um painel com os botes criados
   */
  protected JPanel makeButtons() {
    JPanel panel = new JPanel();
    this.createModifyButton(panel, false);
    this.createRemoveButton(panel, false);
    this.createAddButton(panel, true);
    final JComponent[] buttons =
      new JComponent[] { addButton, removeButton, modifyButton };
    GUIUtils.matchPreferredSizes(buttons);
    return panel;
  }

  /**
   * Cria o painel com os botes da interface e um campo de busca se o argumento
   * hasSearchField seja true. Este mtodo deve ser chamado, somente depois que
   * a tabela tiver sido criada.
   * 
   * @param hasSearchField indica se o painel dos botes ter o campo de busca.
   * 
   * @return o painel
   */
  private JPanel makeButtons(boolean hasSearchField) {
    if (hasSearchField) {
      JPanel panel = new JPanel(new BorderLayout());
      JPanel searchPanel = table.createSearchPanel(LNG.get(
        "IAS_SEARCH_IN_SELECTED_COLUMN") + ":", null, LNG.get("IAS_PREVIOUS"),
          LNG.get("IAS_NEXT"), 5, true);
      panel.add(searchPanel, BorderLayout.NORTH);
      panel.add(makeButtons(), BorderLayout.SOUTH);
      return panel;
    }
    return makeButtons();
  }

  private void createAddButton(JPanel panel, boolean enabled) {
    addButton = new JButton(LNG.get("IAS_ADD"));
    addButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        add();
      }
    });
    addButton.setEnabled(enabled);
    panel.add(addButton);
  }

  private void createRemoveButton(JPanel panel, boolean enabled) {
    removeButton = new JButton(LNG.get("IAS_REMOVE"));
    removeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int row = table.getSelectedRow();
        Object id = getRowId(row);
        delete(id);
      }
    });
    this.removeButton.setEnabled(enabled);
    panel.add(removeButton);
    this.addButtonToSelectionListener(removeButton);
  }

  private void createModifyButton(JPanel panel, boolean enabled) {
    modifyButton = new JButton(LNG.get("IAS_UPDATE"));
    modifyButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int row = table.getSelectedRow();
        Object id = getRowId(row);
        modify(id);
      }
    });
    this.modifyButton.setEnabled(enabled);
    panel.add(modifyButton);
    this.addButtonToSelectionListener(modifyButton);
  }

  /**
   * Implementa o mtodo invocado pelo objeto que est sendo observado. Quando
   * alguma alterao ocorre, esse mtodo  chamado para atualizar as
   * informaes apresentadas na tabela. Os eventos de interesse desse objeto
   * so a incluso, remoo e alterao de items. Repassa para o modelo essa
   * notificao.
   * <p>
   * OBS: Este mtodo s deve ser chamado no thread do Swing (EDT - Event
   * Dispatcher Thread).
   * </p>
   * 
   * @param observable o objeto que est sendo observado.
   * @param arg objeto contendo informaes sobre o evento ocorrido.
   */
  @Override
  public void doUpdate(Observable observable, Object arg) {
    AdministrationEvent action = (AdministrationEvent) arg;
    Object item = action.item;
    ObjectTableModel model = (ObjectTableModel) table.getModel();
    switch (action.type) {
      case AdministrationEvent.CREATE:
        model.add(item);
        break;
      case AdministrationEvent.MODIFY:
        model.modify(item);
        break;
      case AdministrationEvent.DELETE:
        model.remove(item);
        break;
    }
  }

  /**
   * Mtodo abstrato que define a insero de novos elementos na tabela.
   */
  public abstract void add();

  /**
   * Mtodo abstrato que define a modificao de elementos na tabela.
   * 
   * @param id identificador da linha a ser modificada.
   */
  public abstract void modify(Object id);

  /**
   * Mtodo abstrato que define a remoo de elementos na tabela.
   * 
   * @param id identificador da linha a ser removida.
   */
  public abstract void delete(Object id);

  /**
   * Mtodo abstrato que define os procedimentos necessrios antes de fechar a
   * janela.
   */
  public abstract void beforeClose();

  protected ManagerPanel() {
    this.selectionListener = new EnablerListSelectionListener();
  }

  protected SortableTable getTable() {
    return this.table;
  }

  protected void addButtonToSelectionListener(JButton button) {
    this.selectionListener.addButton(button);
  }

  protected void removeButtonFromSelectionListener(JButton button) {
    this.selectionListener.removeButton(button);
  }

  /**
   * Obtm o identificador de um objeto-linha especfico.<br>
   * O identificador  dado pelo valor da coluna zero no modelo. Essa coluna 
   * ocultada quando a tabela  desenhada.
   * 
   * @param row linha
   * @return um objeto representando o identificador da linha.
   */
  protected Object getRowId(int row) {
    return table.getModel().getValueAt(table.convertRowIndexToModel(row),
      IDENTIFIER_INDEX);
  }

  private class EnablerListSelectionListener implements ListSelectionListener {
    private ArrayList buttons;

    public EnablerListSelectionListener() {
      this.buttons = new ArrayList();
    }

    public void addButton(JButton button) {
      this.buttons.add(button);
    }

    public void removeButton(JButton button) {
      this.buttons.remove(button);
    }

    @Override
    public void valueChanged(ListSelectionEvent e) {
      Iterator buttonsIterator = buttons.iterator();
      boolean enabled;
      if (table.getSelectedRowCount() == 0) {
        enabled = false;
      }
      else {
        enabled = true;
      }
      while (buttonsIterator.hasNext()) {
        JButton button = (JButton) buttonsIterator.next();
        button.setEnabled(enabled);
      }
    }
  }
}
