/*
 * UserGroupManagerPanel.java
 * 
 * $Author$ $Revision$ - $Date: 2006-08-16 15:17:41 -0300
 * (Wed, 16 Aug 2006) $
 */
package csbase.client.ias;

import java.text.MessageFormat;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.SwingObserverWrapper;
import csbase.logic.UserGroup;

/**
 * A classe <code>UserGroupManagerPanel</code> representa o painel de
 * administrao de grupos de usurios do sistema. Mostra uma tabela com a
 * identificao e a descrio de cada grupo. O administrador pode incluir,
 * remover e modificar os dados de um grupo de usurio qualquer.
 * 
 * @author Joaquim Cleto de Sousa
 * @version $Revision$
 */
public class UserGroupManagerPanel extends ManagerPanel {
  /** A janela onde esse painel est sendo usado. */
  protected JFrame parent;

  // Constante utilizada para testes de condies de erro.
  // 0: funcionamento normal
  // 1: exibe a mensagem de comunicao com o servidor e dispara o tratamento 
  //    para grupo de usurio no encontrado
  private static final int debug = 0;

  /**
   * Remove o observador de Grupos de Usurio.
   */
  @Override
  public void beforeClose() {
    UserGroup.deleteObserver(new SwingObserverWrapper(this));
  }

  /**
   * Exibe a janela de incluso de Grupos de Usurio.
   */
  @Override
  public void add() {
    new UserGroupInfoDialog(parent);
  }

  /**
   * Exibe a janela de modificao de Grupos de Usurio.
   * 
   * @param id identificador do Grupo de Usurio a ser modificado.
   */
  @Override
  public void modify(Object id) {
    UserGroup userGroup = getUserGroup(id);
    if (userGroup == null) {
      return;
    }
    new UserGroupInfoDialog(parent, userGroup);
  }

  /**
   * Obtm o grupo de usurio identificado pelo parmetro especificado. Caso o
   * grupo de usurio no seja encontrado, exibe mensagem de erro adequada.
   * 
   * @param id identificador do grupo de usurio.
   * 
   * @return grupo de usurio ou <code>null</code> caso este no seja
   *         encontrado.
   */
  protected UserGroup getUserGroup(Object id) {
    UserGroup userGroup =
      UserGroupProxy.getUserGroup(parent, parent.getTitle(),
        LNG.get("IAS_USERGROUPMGR_WAITING_USERGROUP"), id);
    if (debug == 1) { //@debug
      displayMessage( //@debug
        parent.getTitle(), LNG.get("IAS_USERGROUPMGR_WAITING_USERGROUP"));
      userGroup = null; // @debug
    } //@debug
    if (userGroup == null) {
      displayError(LNG.get("IAS_USERGROUPMGR_USERGROUP_NOT_FOUND"));
    }
    return userGroup;
  }

  /**
   * Exibe a janela de remoo de Grupos de Usurio.
   * 
   * @param id identificador do Grupo de Usurio a ser removido.
   */
  @Override
  public void delete(Object id) {
    UserGroup userGroup = getUserGroup(id);
    if (userGroup == null) {
      return;
    }
    String userGroupName = userGroup.getName();
    int res =
      StandardDialogs.showYesNoDialog(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_USERGROUPMGR_USERGROUP_REMOVAL_CONFIRMATION"),
          new Object[] { userGroupName }));
    if (res == JOptionPane.YES_OPTION) {
      UserGroupProxy.deleteUserGroup(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_USERGROUPMGR_WAITING_REMOVE_USERGROUP"),
          new Object[] { userGroupName }), userGroup.getId());
    }
  }

  /**
   * Mostra uma janela com uma mensagem.
   * 
   * @param title ttulo da janela de mensagem
   * @param message mensagem a ser exibida
   */
  protected void displayMessage(String title, String message) {
    StandardDialogs.showInfoDialog(parent, title, message);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param message mensagem de erro a ser exibida
   */
  protected void displayError(String message) {
    StandardErrorDialogs.showErrorDialog(parent, parent.getTitle(), message);
  }

  /**
   * Cria um painel de gerncia de Grupos de Usurio cadastrados no sistema
   * 
   * @param parent A janela onde esse painel est sendo usado.
   */
  public UserGroupManagerPanel(JFrame parent) {
    this.parent = parent;
    Vector<UserGroup> userGroups =
      UserGroupProxy.getAllUserGroups(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (userGroups == null) {
      userGroups = new Vector<UserGroup>();
    }
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        UserGroup userGroup = (UserGroup) item;
        if (userGroup == null) {
          return null;
        }
        return new Object[] { userGroup.getId(), userGroup.getName(),
            userGroup.getDesc() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { null, LNG.get("IAS_USERGROUP"),
            LNG.get("IAS_USERGROUP_DESCRIPTION") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { Object.class, String.class, String.class };
      }
    };
    TableModel model = new ObjectTableModel<UserGroup>(userGroups, provider);
    Comparator<?>[] comparators =
      { null, ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase() };
    make(model, comparators, false);
    UserGroup.addObserver(new SwingObserverWrapper(this));
  }
}
