/*
 * $Id$
 */
package csbase.client.project.action;

import java.awt.event.ActionEvent;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.ClientServerManager;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTree;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.PermissionException;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Ao que atualiza a rvore de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectCloseAction extends ProjectAction implements WindowListener {

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    close();
  }

  /**
   * Executa a ao relativa a janela
   * 
   * @param e evento.
   */
  @Override
  public void windowClosing(WindowEvent e) {
    close();
  }

  /**
   * Tenta salvar as modificaes no projeto?
   * 
   * @return false Caso tenha ocorrido algum erro ao fechar o projeto.
   */
  @SuppressWarnings("unchecked")
  public boolean close() {
    final CommonClientProject project = getCurrentProject();
    if (project == null) {
      return true;
    }
    /*
     * Tenta fechar cada uma das aplicaes em execuo. Caso alguma no
     * termine, cancela o fechamento do projeto.
     */
    final ApplicationManager appMgr = ApplicationManager.getInstance();
    final Hashtable<String, Vector<ApplicationType>> runningApps =
      appMgr.getRunningApplications();
    final Enumeration<String> appKeys = runningApps.keys();
    while (appKeys.hasMoreElements()) {
      final String key = appKeys.nextElement();
      final Vector<ApplicationType> originalVector = runningApps.get(key);
      /*
       *  medida que as aplicaes vo sendo fechadas, o vetor vai diminuindo;
       * para evitar problemas no loop, trabalha com um clone.
       */
      final Vector<ApplicationType> copyVector =
        (Vector<ApplicationType>) originalVector.clone();
      if (copyVector != null) {
        for (int i = 0; i < copyVector.size(); i++) {
          final ApplicationType app = copyVector.get(i);
          final ApplicationRegistry appReg =
            appMgr.getApplicationRegistry(app.getId());
          if (appReg.requireProject()) {
            if (!app.closeApplication()) {
              return false;
            }
          }
        }
      }
    }
    /* Notifica observadores do fechamento do projeto */
    closeProject();
    /*
     * se o usurio tem permisso de escrita no projeto, grava modificaes
     */
    Object ownerID = User.getLoggedUser().getId();
    CommonClientProject currentProject = getCurrentProject();
    if (currentProject.userHasAccessRW(ownerID)) {
      if (!saveProjectModifications()) {
        return false;
      }
    }
    /*
     * agora fecha efetivamente o projeto
     */
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        project.close(true);
      }

      @Override
      public void handleServerError(CSBaseRuntimeException cserror) {
        if (cserror instanceof PermissionException) {
          // Usurio perdeu acesso ao projeto depois de te-lo aberto.
        }
        else {
          super.handleServerError(cserror);
        }
      }
    };
    if (!task.execute(getWindow(), LNG.get("PRJ_PROJECT_CLOSE_TITLE"),
      LNG.get("PRJ_WAITING_CLOSE_PROJECT"))) {
      Exception e = task.getError();
      if (!(e instanceof PermissionException)) {
        return false;
      }
      // else: Usurio perdeu acesso ao projeto depois de te-lo aberto.
    }
    this.getProjectTree().resetProject();
    return true;
  }

  /**
   * Persiste eventuais modificaes nas propriedades do projeto.
   * 
   * @return true se a gravao foi bem-sucedida
   */
  private boolean saveProjectModifications() {
    RemoteTask<Void> saveProjectTask = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        try {
          getCurrentProject().modify();
        }
        finally {
          ClientRemoteLocator.server.setServerSessionProperty(
            ClientServerManager.getInstance().getSession().getKey(),
            ProjectServiceInterface.CURRENT_PROJECT_SESSION_PROPERTY, null);
        }
      }
    };
    if (!saveProjectTask.execute(getWindow(),
      LNG.get("PRJ_PROJECT_CLOSE_TITLE"), LNG.get("PRJ_WAITING_CLOSE_PROJECT"))) {
      Exception e = saveProjectTask.getError();
      if (!(e instanceof PermissionException)) {
        return false;
      }
      // else: Usurio perdeu acesso ao projeto depois de te-lo aberto.
    }
    return true;
  }

  /**
   * Obtm o nome da ao.
   * 
   * @return .
   */
  @Override
  public String getName() {
    return LNG.get("PRJ_CLOSE");
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowActivated(java.awt.event.WindowEvent)
   */
  @Override
  public void windowActivated(WindowEvent e) {
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowClosed(java.awt.event.WindowEvent)
   */
  @Override
  public void windowClosed(WindowEvent e) {
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowDeactivated(java.awt.event.WindowEvent)
   */
  @Override
  public void windowDeactivated(WindowEvent e) {
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowDeiconified(java.awt.event.WindowEvent)
   */
  @Override
  public void windowDeiconified(WindowEvent e) {
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowIconified(java.awt.event.WindowEvent)
   */
  @Override
  public void windowIconified(WindowEvent e) {
  }

  /**
   * {@inheritDoc}
   * 
   * @see java.awt.event.WindowListener#windowOpened(java.awt.event.WindowEvent)
   */
  @Override
  public void windowOpened(WindowEvent e) {
  }

  /**
   * Constri a ao.
   * 
   * @param treeView .
   */
  public ProjectCloseAction(ProjectTree treeView) {
    super(treeView);
  }
}
