/**
 * $Id$
 */
package csbase.client.project.dialogs;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.util.user.UserPanel;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;
import csbase.logic.UserOutline;

/**
 * Tela para gerenciamento das permisses de compartilhamento dos projetos.
 * Permite controlar o tipo de acesso ao projeto (privado, pblico RO, pblico
 * RW, seletivo). Para acesso seletivo, permite definio pontual de permisses
 * de acesso RO ou RW para cada usurio.
 * 
 * @author Tecgraf
 */
public class PublishProjectDialog extends JDialog {
  /**
   * Radio button para projeto privado.
   */
  protected JRadioButton choicePrivate;
  /**
   * Radio button para projeto compartilhado.
   * 
   * @see #choicePublicRO
   * @see #choicePublicRW
   */
  protected JRadioButton choicePublic;
  /**
   * Radio button para projeto pblico em modo RO.
   * 
   * @see #choicePublicRW
   */
  protected JRadioButton choicePublicRO;
  /**
   * Radio button para projeto pblico em modo RW.
   * 
   * @see #choicePublicRO
   */
  protected JRadioButton choicePublicRW;
  /**
   * Radio button para projeto compartilhado seletivamente (listas de
   * permisses).
   */
  protected JRadioButton choiceSharedPartial;
  /**
   * Boto de confirmao da tela.
   */
  protected JButton confirmButton;
  /**
   * Boto de cancelamento da tela.
   */
  protected JButton cancelButton;
  /**
   * Flag que indica se o usurio confirmou (true) ou cancelou (false) a edio.
   */
  protected boolean userConfirmed;

  /**
   * Painel de compartilhamento de projeto com usurios.
   */
  protected UserPanel sharingPanel;
  /**
   * Usurios com acesso RO ao projeto (preenchido pelo mtodo {@link #close()}
   * ).
   */
  protected Set<Object> usersRO;
  /**
   * Usurios com acesso RW ao projeto (preenchido pelo mtodo {@link #close()}
   * ).
   */
  protected Set<Object> usersRW;
  /**
   * Tipo de compartilhamento do projeto.
   */
  protected SharingType sharingType;
  /**
   * Flag que indica se o dilogo j foi fechado. Se sim, algumas informaes j
   * esto disponveis (como p.ex. as listas de usurios).
   */
  private boolean isClosed;

  /**
   * Construtor.
   * 
   * @param parent - janela-me
   * @param project - projeto a ser configurado
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO - lista com os usurios que tm acesso RO ao projeto
   * @param usersRW - lista com os usurios que tm acesso RW ao projeto
   */
  public PublishProjectDialog(JFrame parent, CommonClientProject project,
    List<UserOutline> allUsers, Set<Object> usersRO, Set<Object> usersRW) {
    this(parent, project.getUserId(), project.getName(), project
      .getSharingType(), allUsers, usersRO, usersRW);
  }

  /**
   * Construtor.
   * 
   * @param parent - janela-me
   * @param ownerID - identificador do usurio corrente
   * @param prjName - nome do projeto
   * @param sharingType - tipo do compartilhamento
   * @param allUsers - lista com todos os usurios disponveis para
   *        compartilhamento
   * @param usersRO - lista com os usurios que tm acesso RO ao projeto
   * @param usersRW - lista com os usurios que tm acesso RW ao projeto
   */
  public PublishProjectDialog(JFrame parent, Object ownerID, String prjName,
    SharingType sharingType, List<UserOutline> allUsers, Set<Object> usersRO,
    Set<Object> usersRW) {
    super(parent,
      String.format(LNG.get("PublishProjectDialog.title"), prjName), true);
    this.sharingType = sharingType;
    filterUsers(allUsers, ownerID);
    sharingPanel = new UserPanel(allUsers, usersRO, usersRW, true);
  }

  /**
   * Mtodo privado para testes.
   */
  private PublishProjectDialog() {
    this(null, null, "TESTE", SharingType.PARTIAL,
      new ArrayList<UserOutline>(), new HashSet<Object>(),
      new HashSet<Object>());
  }

  /**
   * Removemos da lista de todos os usurios o dono do projeto e o
   * administrador.
   * 
   * @param allUsers - lista de todos os usurios do sistema
   * @param ownerID - ID do usurio corrente
   */
  private void filterUsers(List<UserOutline> allUsers, Object ownerID) {
    List<UserOutline> toRemove = new ArrayList<UserOutline>();
    for (UserOutline userOutline : allUsers) {
      Object id = userOutline.getId();
      if (id.equals(User.getAdminId()) || id.equals(ownerID)) {
        toRemove.add(userOutline);
      }
      if (toRemove.size() == 2) {
        // s precisamos filtrar o admin e o dono do projeto
        break;
      }
    }
    allUsers.removeAll(toRemove);
  }

  /**
   * Cria a interface.
   */
  protected void createGUI() {
    /*
     * interceptamos tentativas de fechar a janela
     */
    setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        cancel();
      }
    });

    setLayout(new GridBagLayout());

    // adiciona verticalmente 1 ou mais painis para entrada de dados
    int y = addInputPanels(0);

    configureRadioButtons();

    // painel inferior com os botes da tela
    JPanel buttonsPanel = createButtonsPanel();
    GBC gbc = new GBC(0, y++).northwest().bottom(5).horizontal();
    add(buttonsPanel, gbc);
  }

  /**
   * Adiciona um ou mais painis para entrada de dados, verticalmente, de cima
   * para baixo, a partir de uma "linha" especfica de um
   * <code>GridBagLayout</code>.
   * 
   * @param yArg linha inicial do layout a partir da qual devem ser inseridos os
   *        painis
   * @return y + (nmero de painis adicionados), para ser usado como referncia
   *         para insero de novos painis
   */
  protected int addInputPanels(int yArg) {
    int y = yArg;
    // adiciona o painel principal
    GBC gbc = new GBC(0, y++).both();
    add(createMainPanel(), gbc);
    return y;
  }

  /**
   * Cria o painel principal.
   * 
   * @return painel principal
   */
  protected JPanel createMainPanel() {
    JPanel panel = new JPanel(new GridBagLayout());

    // --------------------------------
    // radio-buttons
    choicePrivate =
      new JRadioButton(LNG.get("PublishProjectDialog.option.private"));
    choicePublic =
      new JRadioButton(LNG.get("PublishProjectDialog.option.public"));
    choicePublicRO =
      new JRadioButton(LNG.get("PublishProjectDialog.option.public.ro"));
    choicePublicRW =
      new JRadioButton(LNG.get("PublishProjectDialog.option.public.rw"));
    choiceSharedPartial =
      new JRadioButton(LNG.get("PublishProjectDialog.option.partial"));

    // acesso privado
    int y = 0;
    GBC gbc = new GBC(0, y++).northwest().top(10).left(10);
    panel.add(choicePrivate, gbc);

    // acesso pblico
    gbc = new GBC(0, y++).northwest().top(0).left(10);
    panel.add(choicePublic, gbc);

    // acesso pblico RW
    gbc = new GBC(0, y++).northwest().top(0).left(30);
    panel.add(choicePublicRW, gbc);

    // acesso pblico RO
    gbc = new GBC(0, y++).northwest().top(0).left(30);
    panel.add(choicePublicRO, gbc);

    // acesso seletivo
    gbc = new GBC(0, y++).northwest().top(0).left(10);
    panel.add(choiceSharedPartial, gbc);

    JPanel sharedUsersPanel = createSharedUsersPanel();
    gbc = new GBC(0, y++).northwest().insets(5, 35, 10, 15).both();
    panel.add(sharedUsersPanel, gbc);

    return panel;
  }

  /**
   * Cria e configura os radio-buttons.
   */
  private void configureRadioButtons() {
    /*
     * o radio que torna o projeto pblico deve habilitar os que definem se o
     * acesso ser RO ou RW
     */
    choicePublic.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        boolean isSelected = e.getStateChange() == ItemEvent.SELECTED;
        choicePublicRO.setEnabled(isSelected);
        choicePublicRW.setEnabled(isSelected);
      }
    });

    choiceSharedPartial.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        boolean isSelected = e.getStateChange() == ItemEvent.SELECTED;
        enableSharedUsersPanel(isSelected);
      }
    });

    ButtonGroup mainRadiosGroup = new ButtonGroup();
    mainRadiosGroup.add(choicePrivate);
    mainRadiosGroup.add(choicePublic);
    mainRadiosGroup.add(choiceSharedPartial);

    ButtonGroup publicRadiosGroup = new ButtonGroup();
    publicRadiosGroup.add(choicePublicRO);
    publicRadiosGroup.add(choicePublicRW);

    /*
     * faz com que os radio-buttons espelhem o tipo de compartilhamento do
     * projeto
     */
    switch (sharingType) {
      case PRIVATE:
        choicePrivate.setSelected(true);
        choicePublicRO.setEnabled(false);
        choicePublicRW.setEnabled(false);
        choicePublicRW.setSelected(true);
        enableSharedUsersPanel(false);
        break;

      case ALL_RO:
        choicePublic.setSelected(true);
        choicePublicRO.setSelected(true);
        enableSharedUsersPanel(false);
        break;

      case ALL_RW:
        choicePublic.setSelected(true);
        choicePublicRW.setSelected(true);
        enableSharedUsersPanel(false);
        break;

      case PARTIAL:
        choiceSharedPartial.setSelected(true);
        choicePublicRO.setEnabled(false);
        choicePublicRW.setEnabled(false);
        choicePublicRW.setSelected(true);
        break;
    }
  }

  /**
   * Habilita / Desabilita tabela do painel de compartilhamento.
   * 
   * @param enabled - estado do painel
   */
  public void enableSharedUsersPanel(final boolean enabled) {
    sharingPanel.enableSharedUsersPanel(enabled);
  }

  /**
   * Cria o painel para configurao pontual do acesso de cada usurio no modo
   * de compartilhamento seletivo .
   * 
   * @return painel para configurao pontual do acesso de cada usurio
   */
  private JPanel createSharedUsersPanel() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(BorderFactory.createTitledBorder(""));

    GBC gbc = new GBC(0, 1).both().insets(10, 10, 0, 10).width(3);
    panel.add(sharingPanel, gbc);
    return panel;
  }

  /**
   * Cria o painel inferior com os botes "Confirmar" e "Cancelar".
   * 
   * @return painel inferior com os botes "Confirmar" e "Cancelar"
   */
  private JPanel createButtonsPanel() {
    JPanel panel = new JPanel();
    confirmButton = new JButton(LNG.get("PublishProjectDialog.button.confirm"));
    confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (canConfirm()) {
          confirm();
        }
      }
    });
    cancelButton = new JButton(LNG.get("PublishProjectDialog.button.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        cancel();
      }
    });
    panel.add(confirmButton);
    panel.add(cancelButton);

    final JComponent[] buttons =
      new JComponent[] { confirmButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);
    return panel;
  }

  /**
   * Aes associadas ao boto "Confirmar".
   */
  protected void confirm() {
    userConfirmed = true;
    close();
  }

  /**
   * Avalia se a tela pode ser confirmada.
   * 
   * @return true se no houver nenhuma inconsistncia e a tela pode ser
   *         confirmada
   */
  protected boolean canConfirm() {
    // TODO implementar
    return true;
  }

  /**
   * Aes associadas ao boto "Cancelar".
   */
  protected void cancel() {
    if (publishChanged()) {
      int option =
        StandardDialogs.showYesNoDialog(this,
          LNG.get("PublishProjectDialog.confirmation.dialog.title"),
          LNG.get("PublishProjectDialog.confirmation.dialog.message"));
      if (option == JOptionPane.NO_OPTION) {
        return;
      }
    }
    userConfirmed = false;
    close();
  }

  /**
   * Verifica se o usurio fez alguma alterao nas permisses de
   * compartilhamento.
   * 
   * @return True, se o usurio fez alguma alterao. False, caso contrrio.
   */
  private boolean publishChanged() {
    SharingType selectedSharingType = getSelectedSharingType();

    if (sharingType != selectedSharingType) {
      return true;
    }
    if (selectedSharingType == SharingType.PARTIAL) {
      return sharingPanel.hasUsersPermissionsChanged();
    }
    return false;
  }

  /**
   * Retorna o tipo de compartilhamento selecionado (estado dos radiobuttons).
   * 
   * @return tipo de compartilhamento selecionado (estado dos radiobuttons)
   */
  private SharingType getSelectedSharingType() {
    if (choicePrivate.isSelected()) {
      return SharingType.PRIVATE;
    }
    else if (choicePublic.isSelected()) {
      if (choicePublicRO.isSelected()) {
        return SharingType.ALL_RO;
      }
      else {
        return SharingType.ALL_RW;
      }
    }
    else {
      return SharingType.PARTIAL;
    }
  }

  /**
   * Aes associadas ao fechamento da janela. Alm de fech-la, preenche os
   * conjuntos de acesso RO e RW.
   */
  protected void close() {
    sharingPanel.fillUserSets();
    usersRO = sharingPanel.getUsersRO();
    usersRW = sharingPanel.getUsersRW();
    /*
     * atualiza o tipo de compartilhamento de acordo com o que o usurio
     * selecionou
     */
    sharingType = getSelectedSharingType();

    setVisible(false);
    dispose();
    isClosed = true;
  }

  /**
   * Exibe a tela.
   * 
   * @return true se o usurio confirmou a edio, false caso tenha cancelado
   */
  public boolean execute() {
    createGUI();
    setSize(new Dimension(600, 700));
    setVisible(true);
    return userConfirmed;
  }

  /**
   * Retorna o tipo de compartilhamento definido pelo usurio (estado dos
   * radiobuttons).
   * 
   * @return tipo de compartilhamento definido pelo usurio (estado dos
   *         radiobuttons)
   */
  public SharingType getSharingType() {
    checkIsClosed();
    return sharingType;
  }

  /**
   * Obtm o conjunto de usurios com acesso RO ao projeto.
   * 
   * @return conjunto de usurios com acesso RO ao projeto
   */
  public Set<Object> getUsersRO() {
    checkIsClosed();
    return usersRO;
  }

  /**
   * Garante que o dilogo j foi fechado.
   */
  protected void checkIsClosed() {
    if (!isClosed) {
      throw new IllegalStateException(
        LNG.get("PublishProjectDialog.not.closed.msg"));
    }
  }

  /**
   * Obtm o conjunto de usurios com acesso RW ao projeto.
   * 
   * @return conjunto de usurios com acesso RW ao projeto
   */
  public Set<Object> getUsersRW() {
    checkIsClosed();
    return usersRW;
  }

  /**
   * Mtodo para testes.
   * 
   * @param args argumentos de teste.
   */
  public static void main(String[] args) {
    PublishProjectDialog dialog = new PublishProjectDialog();
    dialog.execute();
  }
}
