/**
 * $Id$
 */

package csbase.client.remote;

import java.rmi.RemoteException;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.AdministrationEvent;
import csbase.logic.Permission;
import csbase.logic.Platform;
import csbase.logic.Role;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * A classe <code>AdministrationObserver</code> recebe os eventos disparados
 * pelo servio de administrao. Possui apenas uma nica instncia. Interage
 * com as classes {@link csbase.logic.User User}, {@link csbase.logic.UserGroup
 * UserGroup}, {@link csbase.logic.Role Role} e {@link csbase.logic.Platform
 * Platform} para manter as informaes dos objetos atualizadas.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class AdministrationObserver {
  /**
   * Instncia nica do observador
   */
  private static AdministrationObserver instance = null;

  /**
   * O ouvinte de mensagens com o corpo contendo objetos da classe
   * AdministrationEvent.
   */
  private IMessageListener listener = new IMessageListener() {
    @Override
    public void onMessagesReceived(Message... messages) throws Exception {
      for (Message message : messages) {
        AdministrationEvent event = (AdministrationEvent) message.getBody();
        try {
          if (event.item instanceof User) {
            User.update(event);
          }
          else if (event.item instanceof UserGroup) {
            UserGroup.update(event);
          }
          else if (event.item instanceof Role) {
            Role.update(event);
          }
          else if (event.item instanceof Platform) {
            Platform.update(event);
          }
          else if (event.item instanceof Permission) {
            Permission.update(event);
          }
        }
        catch (Exception e) {
          StandardErrorDialogs.showErrorDialog(null, LNG.get("ERRO"), e);
        }
      }
    }
  };

  /**
   * Inicializa os servios desta classe: cria o nico objeto deste tipo e o
   * registra como observador do AdministrationService.
   * 
   * @throws Exception em caso de falha.
   */
  public static void start() throws Exception {
    if (instance != null) {
      return;
    }
    instance = new AdministrationObserver();
  }

  /**
   * @throws RemoteException em caso de falha RMI.
   */
  public static void stop() throws RemoteException {
    if (instance == null) {
      return;
    }
    instance.uninstall();
  }

  /**
   * Registra no servidor o observador de eventos administrativos.
   */
  public void install() {
    MessageProxy.addListener(listener, AdministrationEvent.class);
  }

  /**
   * Remove do servidor o observador de eventos administrativos.
   */
  public void uninstall() {
    MessageProxy.removeListener(listener);
  }

  /**
   * Cria um observador remoto de eventos administrativos.
   * 
   * @throws Exception em caso de falha.
   */
  protected AdministrationObserver() throws Exception {
    install();
  }
}
