/**
 * $Id$
 */

package csbase.client.remote;

import java.rmi.RemoteException;
import java.util.Observable;

import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.logic.AdministrationEvent;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectUserEvent;
import csbase.logic.SharedProjectEvent;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * A classe <code>ProjectAdminObserver</code> recebe os eventos enviados pelo
 * servio de administrao de projetos. Possui apenas uma nica instncia.
 * Interage com a classe {@link csbase.logic.ProjectAdminInfo ProjectAdminInfo}
 * para manter as informaes de projetos atualizadas.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectAdminObserver extends Observable {
  /**
   * Instncia nica do observador
   */
  private static ProjectAdminObserver instance = null;

  /**
   * Ouvinte de eventos administrativos.
   */
  private IMessageListener adminstrationEventListener;
  /**
   * Ouvinte de eventos de projetos.
   */
  private IMessageListener projectUserEventListener;
  /**
   * Ouvinte de eventos de mudanas no compartilhamento de projetos.
   */
  private IMessageListener sharedProjectEventListener;

  /**
   * Inicializa os servios desta classe: cria o nico objeto deste tipo e o
   * registra como observador do ProjectService e do ServerMonitor.
   * 
   * @throws RemoteException no caso de falha de RMI.
   */
  public static void start() throws RemoteException {
    if (instance != null) {
      return;
    }
    instance = new ProjectAdminObserver();
  }

  /**
   * @throws RemoteException no caso de falha de RMI.
   */
  public static void stop() throws RemoteException {
    if (instance == null) {
      return;
    }
    instance.uninstall();
  }

  /**
   * Retorna a nica instancia desse Objeto
   * 
   * @return .
   */
  public static ProjectAdminObserver getInstance() {
    return instance;
  }

  /**
   * Registra o ouvinte de eventos administrativos.
   */
  private void registerAdminstrationEventListener() {
    adminstrationEventListener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          final AdministrationEvent event =
            (AdministrationEvent) message.getBody();
          ProjectAdminInfo.update(event);
        }
      }
    };

    MessageProxy.addListener(adminstrationEventListener,
      AdministrationEvent.class);
  }

  /**
   * Registra o ouvinte de eventos de projetos.
   */
  private void registerProjectUserEventListener() {
    projectUserEventListener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          final ProjectUserEvent event = (ProjectUserEvent) message.getBody();
          setChanged();
          notifyObservers(event);
        }
      }
    };
    MessageProxy.addListener(projectUserEventListener, ProjectUserEvent.class);
  }

  /**
   * Registra o ouvinte de eventos de mudanas no compartilhamento de projetos.
   */
  private void registerSharedProjectEventListener() {
    sharedProjectEventListener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          final SharedProjectEvent event =
            (SharedProjectEvent) message.getBody();
          setChanged();
          notifyObservers(event);
        }
      }
    };
    MessageProxy.addListener(sharedProjectEventListener,
      SharedProjectEvent.class);
  }

  /**
   * Registra no servidor este objeto como observador de projectAdmin.
   */
  public void install() {
    registerAdminstrationEventListener();
    registerProjectUserEventListener();
    registerSharedProjectEventListener();
  }

  /**
   * Remove do servidor o registro desse observador.
   */
  public void uninstall() {
    MessageProxy.removeListener(adminstrationEventListener);
    MessageProxy.removeListener(projectUserEventListener);
    MessageProxy.removeListener(sharedProjectEventListener);
  }

  /**
   * Cria um observador remoto do servio de administrao de projetos. Exporta
   * esse objeto para o sistema rmi.
   * 
   * @throws RemoteException no caso de falha de RMI.
   */
  protected ProjectAdminObserver() throws RemoteException {
    super();
    install();
  }
}
