package csbase.client.applications.flowapplication.graph;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Window;
import java.awt.datatransfer.Transferable;
import java.awt.geom.Dimension2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.algorithms.validation.ViewValidationResult;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.applications.flowapplication.FlowTransferable;
import csbase.client.applications.flowapplication.messages.PasteMessage;
import csbase.client.applications.flowapplication.messages.PickGraphMessage;
import csbase.client.applications.flowapplication.messages.RemoveElementsMessage;
import csbase.client.applications.flowapplication.messages.ResetMessage;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowLink;
import csbase.logic.algorithms.flows.FlowNode;
import csbase.logic.algorithms.flows.LinkParameter;
import csbase.logic.algorithms.flows.NodeParameter;
import csbase.logic.algorithms.flows.Point;
import csbase.logic.algorithms.parameters.AbstractFileParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.remote.ClientRemoteLocator;
import tecgraf.vix.Group;
import tecgraf.vix.TypeMessage;
import tecgraf.vix.TypeVO;

/**
 * <p>
 * O grafo  a representao grfica de um fluxo de algoritmo. Os ns do grafo
 * so a representao grfica das verses dos algoritmos. As arestras so as
 * ligaes entre os algoritmos. A ligao  unidirecional e representa o
 * encameamento de um arquivo de sada com um arquivo de entrada.
 * </p>
 *
 * @author lmoreira
 */
public final class Graph extends Group implements GraphElementListener,
  ViewValidator {

  /**
   * Espaamento padro para posicionamento de novos n relativos aos ns
   * existentes.
   */
  static final int NODE_INSET = Grid.DISTANCE * 2;

  /**
   * Ponto inicial para posicionamento de novos n.
   */
  protected static final java.awt.Point INITIAL_POINT = new java.awt.Point(100,
    50);

  /**
   * Lista de observadores do grafo.
   */
  private final List<GraphListener> graphListenerList;
  /**
   * Janela onde o grafo ser desenhado.
   */
  private final Window window;
  /**
   * Nome do fluxo que foi utilizado para gerar o grafo.
   */
  private String flowName;

  /**
   * Descrio do fluxo representado pelo grafo.
   */
  private String flowDescription;

  /**
   * Indica se a informao de verso do algoritmo deve ser mostrada no n.
   */
  private boolean versionInfoVisible;

  /**
   * Indica se o rtulo do n deve ser exibido.
   */
  private boolean labelEnabled;

  /**
   * Constri um grafo.
   *
   * @param window A janela onde o grafo ser desenhado.
   */
  public Graph(final Window window) {
    if (window == null) {
      throw new IllegalArgumentException("O parmetro window est nulo.");
    }
    this.graphListenerList = new ArrayList<>();
    this.window = window;
    this.versionInfoVisible = false;
    this.labelEnabled = true;
  }

  /**
   * Constri um grafo a partir de um fluxo.
   *
   * @param window A janela onde o grafo ser desenhado.
   * @param flow O fluxo .
   * @throws ParseException em caso de erro na atribuio dos valores dos
   *         parmetros dos ns do fluxo.
   * @throws RemoteException em caso de erro de comunicao com o servidor.
   */
  public Graph(final Window window, final Flow flow) throws ParseException,
    RemoteException {
    this(window);
    flowName = flow.getName();
    flowDescription = flow.getDescription();
    createGraphNodes(flow);
    createGraphLinks(flow);
  }

  /**
   * Adiciona um observador de grafo.
   *
   * @param listener O observador.
   */
  public void addGraphListener(final GraphListener listener) {
    this.graphListenerList.add(listener);
  }

  /**
   * Adiciona um conjunto de observadores de grafo ao mesmo tempo.
   *
   * @param listenerList O conjunto de observadores de grafo.
   */
  public void addGraphListeners(final List<GraphListener> listenerList) {
    this.graphListenerList.addAll(listenerList);
  }

  /**
   * Remove todos os observadores de grafo.
   */
  public void clearGraphListeners() {
    this.graphListenerList.clear();
  }

  /**
   * Cria uma nova conexo no grafo entre um arquivo de sada de um n e um de
   * entrada de outro n, com o identificador especificado.
   *
   * @param desiredId Identificador desejado para a conexo (pode no estar
   *        disponvel e outro id ser usado).
   * @param inputFileDescriptor Arquivo de entrada da conexo.
   * @param outputFileDescriptor Arquivo de sada da conexo.
   * @param pointList Lista de pontos que a conexo percorre.
   * @return a conexo criada.
   */
  public GraphLink createGraphLink(final int desiredId,
    final GraphFileDescriptor inputFileDescriptor,
    final GraphFileDescriptor outputFileDescriptor,
    final List<Point2D> pointList) {
    final GraphLink link = new GraphLink(this, generateId(desiredId));
    link.attach();
    if (outputFileDescriptor != null) {
      link.increase(new Point2D.Double());
      link.setStart(outputFileDescriptor);
    }
    for (Point2D point : pointList) {
      link.increase(point);
    }
    if (inputFileDescriptor != null) {
      link.increase(new Point2D.Double());
      link.setEnd(inputFileDescriptor);
    }
    return link;
  }

  /**
   * Cria uma nova conexo no grafo entre um arquivo de sada de um n e um de
   * entrada de outro n.
   *
   * @param inputFileDescriptor Arquivo de entrada da conexo.
   * @param outputFileDescriptor Arquivo de sada da conexo.
   * @param pointList Lista de pontos que a conexo percorre.
   * @return nova conexo no grafo entre um arquivo de sada de um n e um de
   *         entrada de outro n.
   */
  public GraphLink createGraphLink(
    final GraphFileDescriptor inputFileDescriptor,
    final GraphFileDescriptor outputFileDescriptor,
    final List<Point2D> pointList) {
    return createGraphLink(generateId(null), inputFileDescriptor,
      outputFileDescriptor, pointList);
  }

  /**
   * Gera um id vlido para um elemento do grafo. Caso seja especificado e
   * esteja disponvel, o id desejado  escolhido.
   * 
   * @param desiredId Nmero do id a ser gerado.
   *
   * @return o id.
   */
  private int generateId(Integer desiredId) {
    int id;
    if (desiredId != null) {
      id = desiredId;
    }
    else {
      id = 0;
    }
    while (!isIdAvailable(id)) {
      id++;
    }
    return id;
  }

  /**
   * Cria um novo n no grafo a partir do configurador do algoritmo que ele
   * representa.
   *
   * @param algorithmConfiguratorView A viso do configurador do algoritmo
   *        representado pelo n.
   * @param label label para o n (aceita {@code null}).
   * @param point Ponto onde o n deve ser colocado no grafo.
   * @param bypassed Indica se o n deve ser desviado.
   * @return O n criado.
   */
  public GraphNode createGraphNode(
    final AlgorithmConfiguratorView algorithmConfiguratorView,
    final String label, final Point2D point, final boolean bypassed) {
    final GraphNode node = new GraphNode(this, generateId(null), label,
      algorithmConfiguratorView, point, bypassed, versionInfoVisible,
      labelEnabled);
    node.attach();
    return node;
  }

  /**
   * Cria um novo n no grafo a partir do configurador do algoritmo que ele
   * representa.
   *
   * @param algorithmConfiguratorView A viso do configurador do algoritmo
   *        representado pelo n.
   * @param label label para o n (aceita {@code null}).
   * @param point Ponto onde o n deve ser colocado no grafo.
   * @param size Tamanho que o n deve ter.
   * @param bypassed Indica se o n deve ser desviado.
   * @return O n criado.
   */
  public GraphNode createGraphNode(
    final AlgorithmConfiguratorView algorithmConfiguratorView,
    final String label, final Point2D point, final Dimension2D size,
    final boolean bypassed) {
    final GraphNode node = new GraphNode(this, generateId(null), label,
      algorithmConfiguratorView, point, size, bypassed, versionInfoVisible,
      labelEnabled);
    node.attach();
    return node;
  }

  /**
   * Cria um novo n no grafo a partir do configurador do algoritmo que ele
   * representa, com o id especificado.
   *
   * @param desiredId Identificador desejado para a conexo (pode no estar
   *        disponvel e outro id ser usado).
   * @param label label para o n (aceita {@code null}).
   * @param configuratorView A viso do configurador do algoritmo representado
   *        pelo n.
   * @param point Ponto onde o n deve ser colocado no grafo.
   * @param size Tamanho que o n deve ter.
   * @param bypassed Indica se o n deve ser desviado.
   * @return o n criado.
   */
  public GraphNode createGraphNode(final int desiredId, final String label,
    final AlgorithmConfiguratorView configuratorView, final Point2D point,
    final Dimension2D size, final boolean bypassed) {
    final GraphNode node = new GraphNode(this, generateId(desiredId), label,
      configuratorView, point, size, bypassed, versionInfoVisible,
      labelEnabled);
    node.attach();
    return node;
  }

  /**
   * Cria um novo n no grafo a partir de informaes do algoritmo que ele
   * representa, com o id especificado. Usado nos casos que no  possvel obter
   * o configurador do algoritmo.
   *
   * @param desiredId Identificador desejado para a conexo (pode no estar
   *        disponvel e outro id ser usado).
   * @param algorithmInfo Informaes sobre o algoritmo.
   * @param label label para o n (aceita {@code null}).
   * @param versionId Informaes sobre a verso do algoritmo.
   * @param point Ponto onde o n deve ser colocado no grafo.
   * @param size Tamanho que o n deve ter.
   * @param bypassed Indica se o n deve ser desviado.
   * @return o n criado.
   */
  public GraphNode createGraphNode(final int desiredId,
    final AlgorithmInfo algorithmInfo, final String label,
    final AlgorithmVersionId versionId, final Point2D point,
    final Dimension2D size, final boolean bypassed) {
    final GraphNode node = new GraphNode(this, generateId(desiredId),
      algorithmInfo, label, versionId, point, size, bypassed,
      versionInfoVisible, labelEnabled);
    node.attach();
    return node;
  }

  /**
   * Cria um novo n no grafo a partir de informaes do algoritmo que ele
   * representa, com o id especificado. Usado nos casos que no  possvel obter
   * o configurador do algoritmo.
   *
   * @param desiredId Identificador desejado para a conexo (pode no estar
   *        disponvel e outro id ser usado).
   * @param algorithmName Nome do algoritmo.
   * @param label label para o n (aceita {@code null}).
   * @param versionId Informaes sobre a verso do algoritmo.
   * @param point Ponto onde o n deve ser colocado no grafo.
   * @param size Tamanho que o n deve ter.
   * @param bypassed Indica se o n deve ser desviado.
   * @return o n criado.
   */
  public GraphNode createGraphNode(final int desiredId,
    final String algorithmName, final String label,
    final AlgorithmVersionId versionId, final Point2D point,
    final Dimension2D size, final boolean bypassed) {
    final GraphNode node = new GraphNode(this, generateId(desiredId),
      algorithmName, label, versionId, point, size, bypassed,
      versionInfoVisible, labelEnabled);
    node.attach();
    return node;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    if (this.getVOs().isEmpty()) {
      return null;
    }
    TypeVO typeVO = getVOs().get(0);
    Rectangle2D bounds = (Rectangle2D) typeVO.getBounds2D().clone();
    for (int i = 1; i < getVOs().size(); i++) {
      typeVO = getVOs().get(i);
      bounds = bounds.createUnion(typeVO.getBounds2D());
    }
    bounds = Grid.includeMargin(bounds);
    return bounds;
  }

  /**
   * Retorna o conjunto de elementos do grafo.
   *
   * @return o conjunto de elementos do grafo.
   */
  public Collection<GraphElement> getElementCollection() {
    return Collections.unmodifiableCollection(getVOs());
  }

  /**
   * Obtm a coleo dos elementos do grafo (<code>GraphElement</code>) que
   * contenham o ponto fornecido.
   *
   * @param pt O ponto.
   * @return A coleo de elementos do grafo. Caso no haja elementos que
   *         contenham o ponto fornecido, retornar uma coleo vazia.
   */
  public Collection<GraphElement> getElementCollection(final Point2D pt) {
    final Collection<GraphElement> elementCollection = new LinkedList<>();
    for (GraphElement element : getVOs()) {
      if (element.contains(pt)) {
        elementCollection.add(element);
      }
    }
    return elementCollection;
  }

  /**
   * Obtm a coleo dos elementos do grafo (<code>GraphElement</code>) que
   * esto contidos na regio limitada pelo retngulo fornecido.
   *
   * @param bounds O retngulo.
   * @return A coleo de elementos do grafo. Caso no haja elementos que
   *         estejam contidos no retngulo fornecido, retornar uma coleo
   *         vazia.
   */
  public Collection<GraphElement> getElementCollection(
    final Rectangle2D bounds) {
    final Collection<GraphElement> elementCollection = new LinkedList<>();
    for (GraphElement element : getVOs()) {
      if (bounds.contains(element.getBounds2D())) {
        elementCollection.add(element);
      }
    }
    return elementCollection;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Graphics2D getGraphics2D() {
    if (getVS() == null) {
      return null;
    }
    return getVS().getGraphics2D();
  }

  /**
   * Retorna uma lista de observadores do grafo.
   *
   * @return A lista de observadores do grafo.
   */
  public List<GraphListener> getGraphListenerList() {
    return Collections.unmodifiableList(this.graphListenerList);
  }

  /**
   * Retorna a conexo com o identificador especificado ou nulo, caso no exista
   * conexo com esse identificador.
   *
   * @param id O identificador da conexo.
   * @return A conexo com o identificador especificado ou nulo, caso no exista
   *         conexo com esse identificador.
   */
  public GraphLink getLink(final int id) {
    for (final GraphLink link : getLinkCollection()) {
      if (link.getId() == id) {
        return link;
      }
    }
    return null;
  }

  /**
   * Obtm a coleo dos conexes do grafo (<code>GraphLink</code>).
   *
   * @return A coleo de conexes do grafo. Caso no haja conexes, retornar
   *         uma coleo vazia.
   */
  public Collection<GraphLink> getLinkCollection() {
    final Collection<GraphLink> linkCollection = new LinkedList<>();
    for (GraphElement element : getVOs()) {
      if (element instanceof GraphLink) {
        final GraphLink graphLink = (GraphLink) element;
        linkCollection.add(graphLink);
      }
    }
    return linkCollection;
  }

  /**
   * Obtm o nome do fluxo representado por este grafo.
   *
   * @return o nome do fluxo representado por este grafo. Pode ser {@code null}.
   */
  public String getName() {
    return flowName;
  }

  /**
   * Retorna a descrio do fluxo representado por este grafo.
   *
   * @return flowDescription a descrio do fluxo.
   */
  public String getDescription() {
    return flowDescription;
  }

  /**
   * Retorna o n com o identificador especificado ou nulo, caso no exista n
   * com esse identificador.
   *
   * @param id O identificador do n.
   * @return O n com o identificador especificado ou nulo, caso no exista n
   *         com esse identificador.
   */
  public GraphNode getNode(final int id) {
    final Collection<GraphNode> nodeCollection = getNodeCollection();
    for (GraphNode node : nodeCollection) {
      if (node.getId() == id) {
        return node;
      }
    }
    return null;
  }

  /**
   * Obtm um determinado n do grafo, a partir do nome do algoritmo que ele
   * representa.
   *
   * @param nodeName nome do algoritmo do n
   * @param nodeVersionId o identificador da verso do algoritmo do n
   * @return o n procurado
   */
  public GraphNode getNode(final String nodeName,
    final AlgorithmVersionId nodeVersionId) {
    final Collection<GraphNode> nodeCollection = getNodeCollection();
    for (GraphNode node : nodeCollection) {
      if (node.getAlgorithmName().equals(nodeName) && node
        .getAlgorithmVersionId().equals(nodeVersionId)) {
        return node;
      }
    }
    return null;
  }

  /**
   * Obtm a coleo dos ns do grafo (<code>GraphNode</code>).
   *
   * @return A coleo de ns do grafo. Caso no haja ns, retornar uma coleo
   *         vazia.
   */
  public Collection<GraphNode> getNodeCollection() {
    final Collection<GraphNode> nodeCollection = new LinkedList<>();
    for (GraphElement obj : getVOs()) {
      if (obj instanceof GraphNode) {
        final GraphNode node = (GraphNode) obj;
        nodeCollection.add(node);
      }
    }
    return nodeCollection;
  }

  /**
   * Retorna a janela onde o grafo  desenhado.
   *
   * @return A janela onde o grafo  desenhado.
   */
  public Window getParentWindow() {
    return this.window;
  }

  /**
   * Retorna o conjunto de itens do grafo que esto selecionados.
   *
   * @return O conjunto de itens selecionados.
   */
  public Set<GraphElement> getSelectedElements() {
    return getSelectedElements(getVOs());
  }

  /**
   * Retorna o conjunto de ns do grafo que esto selecionados.
   *
   * @return O conjunto de ns selecionados.
   */
  public Set<GraphNode> getSelectedNodes() {
    return getSelectedElements(getNodeCollection());
  }

  /**
   * Retorna o conjunto de ligaes do grafo que esto selecionadas.
   *
   * @return O conjunto de ligaes selecionadas.
   */
  public Set<GraphLink> getSelectedLinks() {
    return getSelectedElements(getLinkCollection());
  }

  /**
   * Filtra um conjunto de itens do grafo, retornando os que esto selecionados.
   * 
   * @param <T> Tipo do elemento do grafo.
   * @param elements o conjunto a ser filtrado.
   * @return O conjunto de itens selecionados.
   */
  private <T extends GraphElement> Set<T> getSelectedElements(
    Collection<T> elements) {
    final Set<T> selectedElements = new HashSet<>();
    for (final T element : elements) {
      if (element.isSelected()) {
        selectedElements.add(element);
      }
    }
    return Collections.unmodifiableSet(selectedElements);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public List<GraphElement> getVOs() {
    return super.getVOs();
  }

  /**
   * Verifica se o grafo possui elementos.
   *
   * @return <code>true</code> se o grafo possui elementos ou <code>false</code>
   *         caso contrrio.
   */
  public boolean hasElements() {
    return !getVOs().isEmpty();
  }

  /**
   * Verifica se o grafo possui elementos selecionados.
   *
   * @return <code>true</code> se o grafo possui elementos selecionados ou
   *         <code>false</code> caso contrrio.
   */
  public boolean hasSelectedElements() {
    for (GraphElement element : getVOs()) {
      if (element.isSelected()) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se o grafo representa um fluxo de algoritmos que pode ser
   * executado.
   *
   * Condies para um grafo ser executvel:
   * <ul>
   * <li>Ter ns;</li>
   * <li>Ter apenas um n de entrada (N que no possue arestas de
   * chegada);</li>
   * <li>Todas as arestas devem estar bem formadas (Todas as arestas comeam em
   * um arquivo de sada e terminam em um arquivo de entrada);</li>
   * </ul>
   *
   * @return {@code true} se  executvel ou {@code false} caso contrrio.
   */
  public boolean isExecutable() {
    if (getVOs().isEmpty()) {
      return false;
    }
    for (final GraphLink link : getLinkCollection()) {
      if (!link.isWellFormed()) {
        return false;
      }
    }
    for (final GraphNode node : getNodeCollection()) {
      if (!node.isWellFormed()) {
        return false;
      }
    }
    return true;
  }

  /**
   * Verifica se o grafo est pronto para ser executado. Condies para um grafo
   * estar pronto para ser executado:
   * <ul>
   * <li>Ser executvel;</li>
   * <li>Todos os ns tm estar prontos para serem executados
   * (parametrizados)</li>
   * </ul>
   *
   * @return {@code true} se est pronto para ser executado ou {@code false}
   *         caso contrrio.
   */
  public boolean isReadyToExecute() {
    if (!isExecutable()) {
      return false;
    }
    for (Object element : getVOs()) {
      if (element instanceof GraphNode) {
        final GraphNode node = (GraphNode) element;
        if (!node.isReadyToExecute()) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Faz a juno deste grafo com o outro especificado.
   *
   * @param graph O grafo que deve ser combinado com esse.
   * @return A coleo de elementos dos dois grafos combinados.
   */
  public Collection<GraphElement> merge(final Graph graph) {
    if (graph == null) {
      throw new IllegalArgumentException("O parmetro graph est nulo.");
    }
    final List<GraphElement> elements = new LinkedList<>();
    final Map<GraphFileDescriptor, GraphFileDescriptor> graphFileDescriptors =
      new HashMap<>();
    for (final GraphNode node : graph.getNodeCollection()) {
      int desiredId = node.getId();
      final AlgorithmConfiguratorView algorithmConfiguratorView = node
        .getAlgorithmConfiguratorView();
      final Point2D.Double nodePoint = new Point2D.Double(node.getX(), node
        .getY());
      final Dimension nodeSize = new Dimension();
      nodeSize.setSize(node.getWidth(), node.getHeight());
      final boolean bypassed = node.isBypassed();
      GraphNode newNode;
      if (node.isWellFormed()) {
        newNode = createGraphNode(desiredId, node.getLabel(),
          algorithmConfiguratorView, nodePoint, nodeSize, bypassed);
      }
      else if (node.getAlgorithm() != null) {
        newNode = createGraphNode(desiredId, node.getAlgorithm(), node
          .getLabel(), node.getAlgorithmVersionId(), nodePoint, nodeSize,
          bypassed);
      }
      else {
        newNode = createGraphNode(desiredId, node.getAlgorithmName(), node
          .getLabel(), node.getAlgorithmVersionId(), nodePoint, nodeSize,
          bypassed);
      }
      for (final GraphFileDescriptor fileDescriptor : node
        .getInputFileDescriptorCollection()) {
        GraphFileDescriptor newFileDescriptor = newNode.getInputFileDescriptor(
          fileDescriptor.getParameterName());
        if (newFileDescriptor == null) {
          newFileDescriptor = new GraphFileDescriptor(fileDescriptor
            .getParameterName(), fileDescriptor.getParameterLabel(), false,
            newNode);
          newNode.addInputFileDescriptor(newFileDescriptor);
        }
        graphFileDescriptors.put(fileDescriptor, newFileDescriptor);
      }
      for (final GraphFileDescriptor fileDescriptor : node
        .getOutputFileDescriptorCollection()) {
        GraphFileDescriptor newFileDescriptor = newNode.getOutputFileDescriptor(
          fileDescriptor.getParameterName());
        if (newFileDescriptor == null) {
          newFileDescriptor = new GraphFileDescriptor(fileDescriptor
            .getParameterName(), fileDescriptor.getParameterLabel(), true,
            newNode);
          newNode.addOutputFileDescriptor(newFileDescriptor);
        }
        graphFileDescriptors.put(fileDescriptor, newFileDescriptor);
      }
      try {
        final Map<String, String> parameterValuesByName = new HashMap<>();
        for (final String parameterName : node.getParameterNames()) {
          final String parameterLabel = node.getParameterLabel(parameterName);
          final String parameterType = node.getParameterType(parameterName);
          final String parameterValue = node.getParameterValue(parameterName);
          newNode.addParameter(parameterName, parameterLabel, parameterType);
          parameterValuesByName.put(parameterName, parameterValue);
        }
        newNode.setParameterValuesByName(Collections.unmodifiableMap(
          parameterValuesByName));
      }
      catch (final ParameterNotFoundException | ParseException e1) {
        throw new IllegalStateException(e1.getLocalizedMessage(), e1);
      }
      elements.add(newNode);
    }
    for (final GraphLink link : graph.getLinkCollection()) {
      int desiredId = link.getId();
      final GraphFileDescriptor inputFileDescriptor = graphFileDescriptors.get(
        link.getInputFileDescriptor());
      final GraphFileDescriptor outputFileDescriptor = graphFileDescriptors.get(
        link.getOutputFileDescriptor());
      final List<Point2D> pointList = link.getPointList();
      final List<Point2D> points = new ArrayList<>(pointList);
      if (inputFileDescriptor != null) {
        points.remove(points.size() - 1);
      }
      if (outputFileDescriptor != null) {
        points.remove(0);
      }
      final GraphLink newLink = createGraphLink(desiredId, inputFileDescriptor,
        outputFileDescriptor, points);
      elements.add(newLink);
    }
    return elements;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVO(final TypeMessage msg) {
    if (msg instanceof ResetMessage) {
      handleResetMessage();
      return true;
    }
    if (msg instanceof RemoveElementsMessage) {
      return handleRemoveElementsMessage();
    }
    if (msg instanceof PickGraphMessage) {
      handlePickGraphMessage((PickGraphMessage) msg);
      return true;
    }
    if (msg instanceof PasteMessage) {
      handlePasteMessage((PasteMessage) msg);
      return true;
    }
    return super.msgHandlerVO(msg);
  }

  /**
   * Notifica os observadores sobre mudanas no workspace.
   */
  public void notifyChangedWorkspace() {
    for (final GraphListener graphListener : this.graphListenerList) {
      graphListener.wasChangedWorkspace(this);
    }
  }

  /**
   * Remove todos os observadores do grafo cadastrados.
   */
  public void removeAllListeners() {
    this.graphListenerList.clear();
    for (final Object obj : getElementCollection()) {
      final GraphElement element = (GraphElement) obj;
      element.removeAllListeners();
    }
  }

  /**
   * Remove um observador de grafo.
   *
   * @param listener O observador.
   */
  public void removeGraphListener(final GraphListener listener) {
    this.graphListenerList.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void repaint() {
    if (getVS() != null) {
      super.repaint();
    }
  }

  /**
   * Gera um fluxo que corresponde a este grafo.
   *
   * @return O fluxo gerado.
   */
  public Flow toFlow() {
    return toFlow(flowName, flowDescription, getNodeCollection(),
      getLinkCollection());
  }

  /**
   * Gera um fluxo a partir dos dados de um grafo.
   *
   * @param flowName o nome do fluxo.
   * @param flowDescription a descrio do fluxo.
   * @param nodes conjunto de ns do grafo.
   * @param links conjunto de ligaes do grafo.
   * @return O fluxo gerado.
   */
  public static Flow toFlow(String flowName, String flowDescription,
    Collection<GraphNode> nodes, Collection<GraphLink> links) {
    final Set<FlowNode> flowNodes = new HashSet<>();
    for (final GraphNode graphNode : nodes) {
      final Set<NodeParameter> flowNodeParameters = new HashSet<>();
      for (final String parameterName : graphNode.getParameterNames()) {
        String parameterLabel;
        try {
          parameterLabel = graphNode.getParameterLabel(parameterName);
          final String parameterType = graphNode.getParameterType(
            parameterName);
          final String parameterValue = graphNode.getParameterValue(
            parameterName);
          flowNodeParameters.add(new NodeParameter(parameterName,
            parameterLabel, parameterType, parameterValue));
        }
        catch (final ParameterNotFoundException e) {
          throw new IllegalStateException(e);
        }

      }
      FlowNode flowNode;
      if (graphNode.getAlgorithmConfiguratorView() == null) {
        flowNode = new FlowNode(graphNode.getId(), graphNode.getAlgorithmName(),
          graphNode.getLabel(), graphNode.getAlgorithmVersionId(),
          flowNodeParameters, (int) Math.round(graphNode.getX()), (int) Math
            .round(graphNode.getY()), (int) Math.round(graphNode.getWidth()),
          (int) Math.round(graphNode.getHeight()), graphNode.isBypassed(),
          graphNode.hasExitCode());
      }
      else {
        flowNode = new FlowNode(graphNode.getId(), graphNode
          .getAlgorithmConfiguratorView().getConfigurator(), graphNode
            .getLabel(), flowNodeParameters, (int) Math.round(graphNode.getX()),
          (int) Math.round(graphNode.getY()), (int) Math.round(graphNode
            .getWidth()), (int) Math.round(graphNode.getHeight()), graphNode
              .isBypassed());
      }
      flowNode.setStandardOutputFile(graphNode.getStandardOutputFile());
      flowNode.setExitCodeLogFile(graphNode.getExitCodeLogFile());
      flowNode.setWarningsFile(graphNode.getWarningsFile());
      flowNode.setHasExitCode(graphNode.hasExitCode());
      flowNodes.add(flowNode);
    }
    final Set<FlowLink> flowLinks = new HashSet<>();
    for (final GraphLink graphLink : links) {
      final List<Point> points = new LinkedList<>();
      for (final Point2D point2D : graphLink.getPointList()) {
        points.add(new Point((int) Math.round(point2D.getX()), (int) Math.round(
          point2D.getY())));
      }
      LinkParameter inputLinkParameter = null;
      final GraphFileDescriptor inputFileDescriptor = graphLink
        .getInputFileDescriptor();
      if (inputFileDescriptor != null) {
        final GraphNode inputGraphNode = inputFileDescriptor.getNode();
        if (inputGraphNode != null) {
          inputLinkParameter = new LinkParameter(inputGraphNode.getId(),
            inputFileDescriptor.getParameterName(), inputFileDescriptor
              .getParameterLabel());
          points.remove(points.size() - 1);
        }
      }
      LinkParameter outputLinkParameter = null;
      final GraphFileDescriptor outputFileDescriptor = graphLink
        .getOutputFileDescriptor();
      if (outputFileDescriptor != null) {
        final GraphNode outputGraphNode = outputFileDescriptor.getNode();
        if (outputGraphNode != null) {
          outputLinkParameter = new LinkParameter(outputGraphNode.getId(),
            outputFileDescriptor.getParameterName(), outputFileDescriptor
              .getParameterLabel());
          points.remove(0);
        }
      }
      final FlowLink flowLink = new FlowLink(graphLink.getId(),
        outputLinkParameter, inputLinkParameter, points);
      flowLinks.add(flowLink);
    }
    return new Flow(flowName, flowDescription, flowNodes, flowLinks);
  }

  /**
   * Valida a configurao do grafo, procurando erros e inconsistncias.
   *
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   * @return O resultado da validao.
   */
  @Override
  public ViewValidationResult validate(final ValidationMode mode) {
    final Collection<GraphNode> nodeCollection = getNodeCollection();
    if (nodeCollection.isEmpty()) {
      LocalizedMessage message = new LocalizedMessage(Graph.class,
        "error_empty_graph");
      return new ViewValidationResult(new ValidationError(message), Graph.this);
    }
    return nodeValidation(mode, false);
  }

  /**
   * Valida os ns do grafo.
   *
   * @param mode o modo de validao.
   * @param highlight determina se o resultado da validao deve ser indicado na
   *        interface grfica.
   * @return o resultado da validao.
   */
  public ViewValidationResult nodeValidation(final ValidationMode mode,
    boolean highlight) {

    ViewValidationResult result = new ViewValidationResult(Graph.this);

    final Collection<GraphNode> nodeCollection = getNodeCollection();
    for (GraphNode node : nodeCollection) {
      final ViewValidationResult nodeValidation = node.validate(mode);
      if (highlight) {
        node.highlightValidationResult(nodeValidation);
      }
      result.addChild(nodeValidation);
    }

    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasAnchored(final GraphLink link,
    final GraphFileDescriptor fileDescriptor) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasLinkAnchored(this, link, fileDescriptor);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasDragged(final GraphElement element, final double tx,
    final double ty) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementDragged(this, element, tx, ty);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasDragged(final GraphElement element, final Point2D startPoint,
    final Point2D endPoint) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementDragged(this, element, startPoint, endPoint);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasDropped(final GraphElement element, final Point2D point) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementDropped(this, element, point);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasIncreased(final GraphLink link) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasLinkIncreased(this, link);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasLinkStatusChanged(final GraphLink link) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasLinkStatusChanged(this, link);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasParameterSetEnabled(final GraphNode node,
    final String parameterName, final boolean isEnabled) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasParameterSetEnabled(this, node, parameterName,
        isEnabled);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasParameterSetVisible(final GraphNode node,
    final String parameterName, final boolean isVisible) {
    for (final GraphListener listener : graphListenerList) {
      listener.wasParameterSetVisible(this, node, parameterName, isVisible);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasParametrized(final GraphElement element) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementParametrized(this, element);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasRenamed(final GraphNode node) {
    // ignora este tipo de evento
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasResized(final GraphNode node) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasNodeResized(this, node);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasSelected(final GraphElement element) {
    if (element.isSelected()) {
      bringToFront(element);
    }
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementSelected(this, element);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void wasUnanchored(final GraphLink link,
    final GraphFileDescriptor fileDescriptor) {
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasLinkUnanchored(this, link, fileDescriptor);
    }
  }

  /**
   * Adiciona o elemento especificado ao grafo.
   *
   * @param element O elemento a ser adicionado.
   * @return verdadeiro se o elemento foi adicionado ou falso, caso o elemento
   *         j esteja no grafo.
   */
  boolean addElement(final GraphElement element) {
    if (getVOs().contains(element)) {
      return false;
    }
    insertElement(element);
    element.addListener(this);
    for (final GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementCreated(this, element);
    }
    return true;
  }

  /**
   * <p>
   * Traz para frente.
   * </p>
   * <p>
   * Faz com que o elemento do grafo fornecido fique na frente de todos os
   * outros.
   * </p>
   *
   * @param element O elemento.
   */
  void bringToFront(final GraphElement element) {
    getVOs().remove(element);
    insertElement(element);
  }

  /**
   * Cria uma aresta.
   *
   * @param outputFileType O arquivo de sada que origina esta conexo.
   * @return A aresta.
   */
  GraphLink createLink(final GraphFileDescriptor outputFileType) {
    final GraphLink link = new GraphLink(this, generateId(null));
    link.setSelected(true);
    link.attach();
    link.increase(new Point2D.Double());
    link.start(outputFileType);
    return link;
  }

  /**
   * Remove um elemento do grafo.
   *
   * @param element elemento do grafo.
   */
  void removeElement(final GraphElement element) {
    getVOs().remove(element);
    for (final GraphListener graphListener : this.graphListenerList) {
      graphListener.wasElementRemoved(this, element);
    }
  }

  /**
   * Cria uma conexo no grafo com as mesmas caractersticas da conexo de fluxo
   * especificada.
   *
   * @param flowLink A conexo de fluxo.
   * @return a conexo criada.
   */
  private GraphLink createGraphLink(final FlowLink flowLink) {
    GraphFileDescriptor inputFileDescriptor = null;
    final LinkParameter inputLinkParameter = flowLink.getInput();
    if (inputLinkParameter != null) {
      final GraphNode inputGraphNode = getNode(inputLinkParameter.getNodeId());
      if (inputGraphNode != null) {
        inputFileDescriptor = inputGraphNode.getInputFileDescriptor(
          inputLinkParameter.getName());
        if (inputFileDescriptor == null) {
          inputFileDescriptor = new GraphFileDescriptor(inputLinkParameter
            .getName(), inputLinkParameter.getLabel(), false, inputGraphNode);
          inputGraphNode.addInputFileDescriptor(inputFileDescriptor);
        }
        /*
         * Seta o valor do parmetro como nulo, para evitar que se tenha o link
         * e um arquivo escolhido manualmente no configurador ao mesmo tempo,
         * conflitando.
         */
        final AbstractFileParameter fileParameter = inputFileDescriptor
          .getFileParameter();
        if (fileParameter != null) {
          fileParameter.setValue(null);
        }
      }
    }
    GraphFileDescriptor outputFileDescriptor = null;
    final LinkParameter outputLinkParameter = flowLink.getOutput();
    if (outputLinkParameter != null) {
      final GraphNode outputGraphNode = getNode(outputLinkParameter
        .getNodeId());
      if (outputGraphNode != null) {
        outputFileDescriptor = outputGraphNode.getOutputFileDescriptor(
          outputLinkParameter.getName());
        if (outputFileDescriptor == null) {
          outputFileDescriptor = new GraphFileDescriptor(outputLinkParameter
            .getName(), outputLinkParameter.getLabel(), true, outputGraphNode);
          outputGraphNode.addOutputFileDescriptor(outputFileDescriptor);
        }
        /*
         * Seta o valor do parmetro como nulo, para evitar que se tenha o link
         * e um arquivo escolhido manualmente no configurador ao mesmo tempo,
         * conflitando.
         */
        final AbstractFileParameter fileParameter = outputFileDescriptor
          .getFileParameter();
        if (fileParameter != null) {
          fileParameter.setValue(null);
        }
      }
    }
    final List<Point2D> points = new LinkedList<>();
    for (final Point point : flowLink.getPoints()) {
      points.add(new Point2D.Double(point.getX(), point.getY()));
    }
    return createGraphLink(flowLink.getId(), inputFileDescriptor,
      outputFileDescriptor, points);
  }

  /**
   * Cria as conexes do grafo com as mesmas caractersticas das conexes do
   * fluxo especificado.
   *
   * @param flow O fluxo.
   */
  private void createGraphLinks(final Flow flow) {
    for (final FlowLink flowLink : flow.getLinks()) {
      createGraphLink(flowLink);
    }
  }

  /**
   * Cria um n no grafo que tenha as mesmas caractersticas do n especificado
   * de fluxo.
   *
   * @param flowNode O n do fluxo.
   * @return o n criado.
   * @throws ParseException em caso de erro na atribuio dos valores dos
   *         parmetros do n.
   * @throws RemoteException em caso de erro de comunicao com o servidor.
   * @throws ParameterNotFoundException Caso no exista um parmetro com o nome
   *         fornecido.
   */
  private GraphNode createGraphNode(final FlowNode flowNode)
    throws ParseException, RemoteException, ParameterNotFoundException {
    final Point2D nodePoint = new Point2D.Double(flowNode.getX(), flowNode
      .getY());
    final Dimension2D nodeDimension = new Dimension(flowNode.getWidth(),
      flowNode.getHeight());
    final AlgorithmInfo nodeAlgorithm = ClientRemoteLocator.algorithmService
      .getInfo(flowNode.getAlgorithmName());
    final boolean bypassed = flowNode.isBypassed();
    GraphNode newNode;
    if (nodeAlgorithm == null) {
      newNode = createGraphNode(flowNode.getId(), flowNode.getAlgorithmName(),
        flowNode.getLabel(), flowNode.getAlgorithmVersionId(), nodePoint,
        nodeDimension, bypassed);
    }
    else {
      final AlgorithmVersionInfo nodeAlgorithmVersion = nodeAlgorithm
        .getVersionInfo(flowNode.getAlgorithmVersionId());
      if (nodeAlgorithmVersion == null) {
        newNode = createGraphNode(flowNode.getId(), nodeAlgorithm, flowNode
          .getLabel(), flowNode.getAlgorithmVersionId(), nodePoint,
          nodeDimension, bypassed);
      }
      else {
        final AlgorithmConfiguratorView nodeConfiguratorView =
          AlgorithmConfiguratorFactory.getInstance().createConfigurationView(
            this.window, nodeAlgorithmVersion,
            ValidationMode.ALLOW_EMPY_VALUES);
        if (nodeConfiguratorView == null) {
          return null;
        }
        newNode = createGraphNode(flowNode.getId(), flowNode.getLabel(),
          nodeConfiguratorView, nodePoint, nodeDimension, bypassed);
      }
    }
    // Atribui o arquivo de sada.
    newNode.setStandardOutputFile(flowNode.getStandardOutputFile());
    newNode.setHasExitCode(flowNode.hasExitCode());
    newNode.setExitCodeLogFile(flowNode.getExitCodeLogFile());
    newNode.setWarningsFile(flowNode.getWarningsFile());

    final Map<String, String> parameterValues = new HashMap<>();
    for (final NodeParameter nodeParameter : flowNode.getParameters()) {
      final String parameterName = nodeParameter.getName();
      final AlgorithmConfiguratorView configuratorView = newNode
        .getAlgorithmConfiguratorView();
      if (configuratorView == null) {
        final String parameterLabel = nodeParameter.getLabel();
        final String parameterType = nodeParameter.getType();
        newNode.addParameter(parameterName, parameterLabel, parameterType);
      }
      final String parameterValue = nodeParameter.getValue();
      parameterValues.put(parameterName, parameterValue);
    }
    newNode.setParameterValuesByName(Collections.unmodifiableMap(
      parameterValues));
    return newNode;
  }

  /**
   * Cria os ns do grafo com as mesmas caractersticas dos ns do fluxo
   * especificado.
   *
   * @param flow O fluxo.
   * @throws ParseException em caso de erro na atribuio dos valores dos
   *         parmetros dos ns.
   * @throws RemoteException em caso de erro de comunicao com o servidor.
   */
  private void createGraphNodes(final Flow flow) throws ParseException,
    RemoteException {
    for (final FlowNode flowNode : flow.getNodes()) {
      try {
        createGraphNode(flowNode);
      }
      catch (ParameterNotFoundException e) {
        // Deixa a criao do restante dos ns continuar
        e.printStackTrace();
      }
    }
  }

  /**
   * Trata mensagens pegar grafo.
   *
   * @param message A mensagem.
   */
  private void handlePickGraphMessage(final PickGraphMessage message) {
    message.setGraph(this);
  }

  /**
   * Trata mensagens remover elementos selecionados.
   *
   * @return <code>true</code> se conseguiu remover pelo menos um elemento ou
   *         <code>false</code> caso contrrio.
   */
  private boolean handleRemoveElementsMessage() {
    final List<GraphElement> elementToRemoveList = new LinkedList<>();
    for (GraphElement element : getVOs()) {
      if (element.isSelected()) {
        elementToRemoveList.add(element);
      }
    }
    if (elementToRemoveList.isEmpty()) {
      return false;
    }
    for (GraphElement elementToRemove : elementToRemoveList) {
      elementToRemove.deattach();
    }
    repaint();
    return true;
  }

  /**
   * Trata mensagens reiniciar.
   */
  private void handleResetMessage() {
    int ix = 0;
    while (ix < getVOs().size()) {
      final GraphElement element = getVOs().get(ix);
      element.reset();
      ix++;
    }
    for (GraphListener graphListener : this.graphListenerList) {
      graphListener.wasReseted(this);
    }
    repaint();
  }

  /**
   * Trata a mensagem do tipo "colar".
   *
   * @param pasteMessage a mensagem.
   */
  private void handlePasteMessage(PasteMessage pasteMessage) {
    final Transferable transferable = pasteMessage.getTransferable();
    if (transferable instanceof FlowTransferable) {
      FlowTransferable flowTransferable = (FlowTransferable) transferable;
      try {
        Flow flow = flowTransferable.getTransferData(
          FlowTransferable.DATA_FLAVOR);
        Graph copyGraph = new Graph(getParentWindow(), flow);
        int gridSize = Grid.DISTANCE;
        copyGraph.translate(new Point2D.Float(gridSize, gridSize));
        Collection<GraphElement> elements = merge(copyGraph);
        pasteMessage.setAffectedElements(elements);
      }
      catch (Exception e) {
        StandardErrorDialogs.showErrorDialog(getParentWindow(), e);
      }
    }
  }

  /**
   * Insere o elemento especificado no grafo.
   *
   * @param element O elemento a ser inserido.
   */
  private void insertElement(final GraphElement element) {
    if (element instanceof GraphLink) {
      getVOs().add(0, element);
    }
    else if (element instanceof GraphNode) {
      if (getVOs().isEmpty()) {
        getVOs().add(0, element);
      }
      else {
        for (int i = 0; i < getVOs().size(); i++) {
          final Object vo = getVOs().get(i);
          if (vo instanceof GraphNode) {
            getVOs().add(i, element);
            break;
          }
        }
      }
    }
    else {
      final String errorMessage = MessageFormat.format(
        "O elemento {0}  de um tipo desconhecido ({1}).", element, element
          .getClass().getName());
      throw new IllegalArgumentException(errorMessage);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean highlightValidationResult(ViewValidationResult result) {
    ViewValidator element = result.getElement();
    if (element != null && !element.equals(this)) {
      return element.highlightValidationResult(result);
    }
    return false;
  }

  /**
   * Determina se a informao de verso do algoritmo deve ser mostrada nos ns.
   *
   * @param visible verdadeiro se a informao deve ser mostrada ou falso, caso
   *        contrrio.
   */
  public void setVersionInfoVisible(boolean visible) {
    this.versionInfoVisible = visible;
    Collection<GraphNode> nodeCollection = getNodeCollection();
    for (GraphNode graphNode : nodeCollection) {
      graphNode.setVersionInfoVisible(visible);
    }
  }

  /**
   * Indica se a informao de verso do algoritmo deve ser mostrada nos ns.
   *
   * @return verdadeiro se a informao est sendo mostrada ou falso, caso
   *         contrrio.
   */
  protected boolean isVersionInfoVisible() {
    return versionInfoVisible;
  }

  /**
   * Habilita/desabilita a exibio de rtulos para os ns do fluxo.
   * 
   * @param isLabelEnabled {@code true}, para habilitar a exibio dos rtulos,
   *        {@code false} para exibir o nome do algoritmo em cada n.
   */
  public void setLabelEnabled(boolean isLabelEnabled) {
    this.labelEnabled = isLabelEnabled;
    Collection<GraphNode> nodeCollection = getNodeCollection();
    for (GraphNode graphNode : nodeCollection) {
      graphNode.setLabelEnabled(isLabelEnabled);
    }
  }

  /**
   * Indica se um determinado identificador est disponvel para ser usado no
   * grafo.
   *
   * @param id o indetificador.
   * @return verdadeiro, se o identificador est disponvel ou falso, caso
   *         contrrio.
   */
  protected boolean isIdAvailable(int id) {
    return getNode(id) == null && getLink(id) == null;
  }

  /**
   * Encontra uma posio para um n a partir da localizao dos ns j
   * existentes, caso existam. O novo n  posicionado abaixo dos j existentes,
   * alinhado de forma centralizada. Se for o primeiro n a ser adicionado, este
   *  colocado no canto superior esquerdo do diagrama.
   *
   * @return nova posio, relativa aos ns pr-existentes.
   */
  public java.awt.Point findPositionForNewNode() {
    if (!hasElements()) {
      return INITIAL_POINT;
    }
    else {
      Rectangle2D bounds = getBounds2D();
      java.awt.Point p = new java.awt.Point();
      p.setLocation(bounds.getCenterX(), bounds.getMaxY() + NODE_INSET);
      return p;
    }
  }

  /**
   * Translada os elementos do grafo.
   *
   * @param point o tamanho do translado.
   */
  public void translate(final Point2D point) {
    for (GraphNode graphNode : getNodeCollection()) {
      graphNode.translate(point);
    }
    for (GraphLink graphLink : getLinkCollection()) {
      graphLink.translate(point);
    }
  }
}
