/*
 * $Id$
 */

package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.AllocateAreaTask;

/**
 * Ao que trata alterao na alocao de projetos. Projetos com espao alocado
 * que sero desalocados e projeto esperando alocao que sero alocados.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class AllocateAreaAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public AllocateAreaAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final boolean ret = runTask(projects);
    if (ret) {
      updateStatusOfLocalProjects(projectList);
      projectsManager.refreshProjectsTable();
    }
    else {
      final String errMsg =
        getString("AllocateAreaAction.error.allocation.message");
      final String errTitle =
        getString("AllocateAreaAction.error.allocation.title");
      StandardDialogs.showErrorDialog(appFrame, errTitle, errMsg);
    }
  }

  /**
   * Define o status de alocao dos projetos como
   * <code>ProjectSpaceAllocation.ALLOCATED</code> e o espao de alocao.
   * 
   * @param projectList Lista de projetos.
   */
  private static void updateStatusOfLocalProjects(
    final List<ProjectsManagerData> projectList) {
    for (ProjectsManagerData pmd : projectList) {
      pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.ALLOCATED);
      final long requestedSpace = pmd.getRequestedSpace();
      pmd.setAllocatedSpace(requestedSpace);
      pmd.setScope(ProjectsManagerScope.PRIVATE);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ProjectsManagerUI.ALLOCATE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {
    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      if (pmd.getProjectSpaceAllocation() != ProjectSpaceAllocation.WAITING_ALLOCATION) {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
      else {
        i++;
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("AllocateAreaAction.project.selection.denied.message")
          + getString("AllocateAreaAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("AllocateAreaAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("AllocateAreaAction.project.selection.some.denied.message")
          + getString("AllocateAreaAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("AllocateAreaAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(List<ProjectsManagerData> projects) throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    AllocateAreaTask aat =
      new AllocateAreaTask(projectsManager, projects, true);
    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    aat.execute(appFrame, getString("AllocateAreaAction.title"),
      getString("AllocateAreaAction.message"));
    if (aat.wasCancelled()) {
      final String err = getString("AllocateAreaAction.cancelled.message");
      aat.showError(err);
      return false;
    }
    if (aat.getStatus() != true) {
      final Exception exception = aat.getError();
      throw exception;
    }
    return aat.getResult();
  }
}
