package csbase.client.preferences.editors;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import csbase.client.applicationmanager.resourcehelpers.AlgorithmSelectionDialog;
import csbase.client.applications.ApplicationImages;
import csbase.client.preferences.PreferenceEditor;
import csbase.client.preferences.PreferenceValue;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import tecgraf.javautils.gui.GBC;

/**
 * Editor de algoritmos favoritos.
 *
 * @author Tecgraf/PUC-Rio
 */
public class FavoriteAlgorithmsEditor extends PreferenceEditor<List<String>> {

  /**
   * Construtor
   *
   * @param pv lista de valores de preferncia.
   */
  public FavoriteAlgorithmsEditor(PreferenceValue<List<String>> pv) {
    super(pv);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createComponent() {
    AbstractTableModel model = new AbstractTableModel() {
      @Override
      public Object getValueAt(int rowIndex, int columnIndex) {
        return getValue().get(rowIndex);
      }

      @Override
      public int getRowCount() {
        return getValue().size();
      }

      @Override
      public String getColumnName(int column) {
        return null;
      }

      @Override
      public int getColumnCount() {
        return 1;
      }
    };

    JTable table = new JTable();
    table.setModel(model);
    table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    table.setShowHorizontalLines(false);
    table.setDefaultRenderer(Object.class, new InternalRenderer());
    JScrollPane pane = new JScrollPane(table);
    pane.setPreferredSize(new Dimension(400, 150));

    JToolBar toolbar = buildToolbar(table);

    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(pane, new GBC(0, 0).both());
    panel.add(toolbar, new GBC(1, 0).vertical());
    setTitledBorder(panel);
    return panel;
  }

  /**
   * Constroi a toolbar.
   *
   * @param table tabela.
   *
   * @return toolbar
   */
  private JToolBar buildToolbar(JTable table) {
    JToolBar toolbar = new JToolBar(JToolBar.VERTICAL);
    toolbar.setFloatable(false);

    AbstractAction addAction = buildAddAction(table);
    AbstractAction delAction = buildDelAction(table);

    addAction.setEnabled(isEditable());
    delAction.setEnabled(isEditable());

    toolbar.add(addAction);
    toolbar.add(delAction);

    return toolbar;
  }

  /**
   * Constroi uma ao.
   *
   * @param table tabela.
   * @return ao
   */
  private AbstractAction buildDelAction(final JTable table) {
    AbstractAction delAction = new AbstractAction(null,
      ApplicationImages.ICON_ROWDELETE_16) {
      @Override
      public void actionPerformed(ActionEvent e) {
        int[] rowIds = table.getSelectedRows();
        if (rowIds == null || rowIds.length == 0) {
          return;
        }

        // Salvar objetos para remoo
        List<String> value = FavoriteAlgorithmsEditor.this.getValue();
        List<String> algorithmsToRemove = new ArrayList<String>(rowIds.length);
        for (int rowId : rowIds) {
          algorithmsToRemove.add(value.get(rowId));
        }

        value.removeAll(algorithmsToRemove);
        updateTable(table);
        notifyListeners();
      }
    };
    return delAction;
  }

  /**
   * Constroi uma ao.
   *
   * @param table tabela.
   * @return ao
   */
  private AbstractAction buildAddAction(final JTable table) {
    AbstractAction addAction = new AbstractAction(null,
      ApplicationImages.ICON_ADD_16) {
      @Override
      public void actionPerformed(ActionEvent e) {
        List<AlgorithmInfo> algorithms = getOthersAlgorithmsRegistries();
        AlgorithmSelectionDialog dialog = new AlgorithmSelectionDialog(null,
          algorithms);
        dialog.setTitle(getLabel());
        dialog.setVisible(true);
        List<String> ids = dialog.getSelectedAlgorithms();
        List<String> value = FavoriteAlgorithmsEditor.this.getValue();

        for (String id : ids) {
          if (!value.contains(id)) {
            value.add(id);
          }
        }
        updateTable(table);
        notifyListeners();
      }
    };
    return addAction;
  }

  /**
   * Atualiza a tabela.
   *
   * @param table tabela
   */
  private void updateTable(JTable table) {
    ((AbstractTableModel) (table.getModel())).fireTableDataChanged();
  }

  /**
   * Consulta a lista de registros passveis de serem seleceionados.
   *
   * @return lista
   */
  private ArrayList<AlgorithmInfo> getOthersAlgorithmsRegistries() {
    AlgorithmInfo[] allInfos = AlgorithmManagementProxy.getAllAlgorithmInfos(
      null, AlgorithmManagementProxy.AlgorithmOperation.EXECUTE_ALGORITHM);
    ArrayList<AlgorithmInfo> algorithms = new ArrayList<AlgorithmInfo>();
    for (int i = 0; i < allInfos.length; i++) {
      AlgorithmInfo info = allInfos[i];
      String id = info.getId();
      if (!getValue().contains(id)) {
        algorithms.add(info);
      }
    }
    return algorithms;
  }

  /**
   * Renderer.
   *
   * @author Tecgraf
   */
  private class InternalRenderer extends DefaultTableCellRenderer {

    /** Cor usada em alternncia com {@link Color#WHITE}. */
    private final Color LIGHT = new Color(0xE8F2FE);

    /**
     * {@inheritDoc}
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
      boolean isSelected, boolean hasFocus, int row, int column) {
      Component cmp = super.getTableCellRendererComponent(table, value,
        isSelected, hasFocus, row, column);

      JLabel label = (JLabel) cmp;
      label.setText(null);
      label.setIcon(null);
      label.setHorizontalAlignment(SwingConstants.LEADING);

      String id = (String) value;
      if (id == null) {
        return label;
      }

      AlgorithmInfo algorithm = AlgorithmManagementProxy.getAlgorithmInfoById(
        id, null,
        AlgorithmManagementProxy.AlgorithmOperation.EXECUTE_ALGORITHM);

      if (algorithm == null) {
        label.setText("!!!");
      }
      else {
        label.setText(algorithm.getName());
        ImageIcon icon = getIconForVersion(algorithm.getLastVersion());
        label.setIcon(icon);
      }

      if (!isSelected) {
        label.setBackground((row % 2 == 0) ? Color.WHITE : LIGHT);
      }

      return label;
    }
  }

  /**
   * Obtm o cone para a verso especificada.
   *
   * @param algorithmVersion a verso do algoritmo.
   * @return o cone que representa a verso especificada.
   */
  public static ImageIcon getIconForVersion(
    AlgorithmVersionInfo algorithmVersion) {
    if (algorithmVersion != null) {
      ConfiguratorType type = algorithmVersion.getType();
      if (type == ConfiguratorType.SIMPLE) {
        return ApplicationImages.ICON_FLOW_NODE_16;
      }
      if (type == ConfiguratorType.FLOW) {
        return ApplicationImages.ICON_FLOW_16;
      }
    }
    return ApplicationImages.ICON_BLANK_16;
  }
}
