/*
 * $Id: LightEffect.java 111171 2010-10-08 22:55:52Z clinio $
 */
package csbase.client.applications.imageviewer.effects;

import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.LookupOp;
import java.awt.image.ShortLookupTable;
import java.util.Hashtable;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.imageviewer.ImageViewer;

/**
 * Efeitos de luz.
 * 
 * @author Tecgraf/PUC-Rio
 */
final public class RGBEffect extends AbstractEffect {

  /**
   * Slider
   */
  final JSlider rSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Slider
   */
  final JSlider gSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Slider
   */
  final JSlider bSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Default
   */
  private final int DEFAULT = 0;

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public RGBEffect(ImageViewer application) {
    super(application);
    rSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        RGBEffect.this.updatePreview();
      }
    });

    gSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        RGBEffect.this.updatePreview();
      }
    });

    bSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        RGBEffect.this.updatePreview();
      }
    });

    final JSlider[] sliders = new JSlider[] { rSlider, bSlider, gSlider };
    for (JSlider slider : sliders) {
      final int min = -100;
      final int max = 100;
      final int step = 10;
      slider.setMinimum(min);
      slider.setMaximum(max);
      slider.setValue(DEFAULT);
      slider.setMajorTickSpacing(step);
      slider.setPaintTicks(true);
      slider.setPaintLabels(true);

      Hashtable<Integer, JLabel> table = new Hashtable<Integer, JLabel>();
      final JLabel minLabel = new JLabel(min + "%");
      table.put(new Integer(min), minLabel);
      final JLabel maxLabel = new JLabel(max + "%");
      table.put(new Integer(max), maxLabel);
      final JLabel defLabel = new JLabel(DEFAULT + "%");
      table.put(new Integer(DEFAULT), defLabel);
      slider.setLabelTable(table);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JPanel getParameterPanel() {
    final JLabel redLabel = new JLabel(getString("RGBEffect.red.label"));
    final JLabel greenLabel = new JLabel(getString("RGBEffect.green.label"));
    final JLabel blueLabel = new JLabel(getString("RGBEffect.blue.label"));
    final JComponent[] rRow = new JComponent[] { redLabel, rSlider };
    final JComponent[] gRow = new JComponent[] { greenLabel, gSlider };
    final JComponent[] bRow = new JComponent[] { blueLabel, bSlider };
    final JComponent[][] rows = new JComponent[][] { rRow, gRow, bRow };
    return GUIUtils.createBasicGridPanel(rows);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public BufferedImage transformImage(BufferedImage image) {
    float red = rSlider.getValue() * 2.5F;
    float green = gSlider.getValue() * 2.5F;
    float blue = bSlider.getValue() * 2.5F;

    short[] lookupR = new short[256];
    short[] lookupG = new short[256];
    short[] lookupB = new short[256];
    for (int i = 0; i < 256; i++) {
      final double rd = i + red;
      final double gd = i + green;
      final double bd = i + blue;
      short rv = (short) Math.round(rd);
      short gv = (short) Math.round(gd);
      short bv = (short) Math.round(bd);
      if (rv < 0) {
        rv = 0;
      }
      if (rv > 255) {
        rv = 255;
      }
      if (gv < 0) {
        gv = 0;
      }
      if (gv > 255) {
        gv = 255;
      }

      if (bv < 0) {
        bv = 0;
      }
      if (bv > 255) {
        bv = 255;
      }

      lookupR[i] = rv;
      lookupG[i] = gv;
      lookupB[i] = bv;
    }
    final short[][] array = new short[][] { lookupR, lookupG, lookupB };
    final ShortLookupTable lookupTable = new ShortLookupTable(0, array);
    BufferedImageOp lightOp = new LookupOp(lookupTable, null);
    BufferedImage newImage = lightOp.filter(image, null);
    return newImage;

    //    final RescaleOp lightOp = new RescaleOp(contrast, offset, null);
    //    BufferedImage newImage = lightOp.filter(image, null);
    //    return newImage;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetParameters() {
    rSlider.setValue(DEFAULT);
    gSlider.setValue(DEFAULT);
    bSlider.setValue(DEFAULT);
  }
}
