package csbase.client.applications.jobmonitor.filters;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tecgraf.javautils.core.filter.IFilter;
import csbase.client.applications.jobmonitor.rowmodel.JobInfoRow;

/**
 * Filtro de objetos do tipo {@link JobInfoRow}.
 * 
 * @author Tecgraf
 */
public class JobInfoFilter implements IFilter<JobInfoRow> {

  /**
   * Expresso regular usada para filtrar as linhas.
   */
  private Pattern pattern;

  /**
   * Lista com o nome das colunas que sero consideradas na filtragem.
   */
  private List<String> columnsToFilter;

  /**
   * Flag que define se o padro deve casar com a palavra inteira.
   */
  private boolean wholeWord;

  /**
   * Construtor padro.
   */
  public JobInfoFilter() {
    // pattern que casa com tudo.
    pattern = Pattern.compile("");
    columnsToFilter = new ArrayList<String>();
  }

  /**
   * Define a lista com o nome das colunas que sero consideradas na filtragem.
   * 
   * @param columnsToFilter - lista com o nome das colunas que sero
   *        consideradas na filtragem.
   */
  public void setColumnsToFilter(List<String> columnsToFilter) {
    this.columnsToFilter = columnsToFilter;
  }

  /**
   * Define a expresso regular que ser usada para filtrar as linhas.
   * 
   * @param pattern - expresso regular.
   */
  public void setPattern(Pattern pattern) {
    if (pattern != null) {
      this.pattern = pattern;
    }
  }

  /**
   * Define se o padro deve casar com a palavra inteira.
   * 
   * @param wholeWord - true se for para casar a palavra inteira, false caso
   *        contrrio.
   */
  public void setWholeWord(boolean wholeWord) {
    this.wholeWord = wholeWord;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean accept(JobInfoRow row) {

    if (columnsToFilter.isEmpty()) {
      return true;
    }

    for (String columnName : columnsToFilter) {
      String value = row.get(columnName);
      if (value != null) {
        Matcher matcher = pattern.matcher(value);

        if (wholeWord) {
          if (matcher.matches()) {
            return true;
          }
        }
        else {
          if (matcher.find()) {
            return true;
          }
        }
      }
    }

    return false;
  }
}