/**
 * $Id$
 */

package csbase.client.applications.projectsmanager.proxy;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectInfoData;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.exception.BugException;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.DiskUsageServiceInterface;
import csbase.remote.ProjectServiceInterface;

/**
 * Tarefa que preenche a tabela com os dados dos projetos existentes no servidor
 * e relevantes ao usurio corrente.
 * 
 * @author jnlopes
 */
public class FillProjectsTableUserTask extends
  ProjectsManagerTask<ProjectInfoData> {

  /**
   * Indica se espao ocupado pelos projetos deve ser carregado ou no.
   */
  private boolean loadOccupiedSpace;

  /**
   * Projetos que no puderam ser abertos pois o arquivo .csbase_project_info
   * no existe ou est corrompido.
   */
  private List<UserProjectInfo> failedProjects =
    new ArrayList<UserProjectInfo>();

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   * @param loadOccupiedSpace boolean que indica se espao ocupado pelos
   *        projetos deve ser carregado ou no.
   */
  public FillProjectsTableUserTask(ProjectsManager projectsManager,
    boolean loadOccupiedSpace) {
    super(projectsManager);
    this.loadOccupiedSpace = loadOccupiedSpace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    List<ProjectsManagerData> projectsData = getProjectsData();
    ProjectInfoData pid = new ProjectInfoData(projectsData, failedProjects);
    setResult(pid);
  }

  /**
   * Mtodo que efetivamente obtm os dados dos projetos.
   * 
   * @return vetor de dados de projeto.
   * @throws Exception em caso de falha.
   */
  private List<ProjectsManagerData> getProjectsData() throws Exception {
    List<ProjectsManagerData> projectList =
      new ArrayList<ProjectsManagerData>();
    getProjectsWhereUserIsOwner(projectList);
    getProjectsWhereUserIsNotOwner(projectList);
    return projectList;
  }

  /**
   * Pega os projetos onde o usurio  o criador do projeto.
   * 
   * @param projectList Lista de projetos a ser atualizada.
   * @throws RemoteException em caso de falha na execuo de mtodo remoto.
   * @throws Exception em caso de falha geral.
   */
  private void getProjectsWhereUserIsOwner(List<ProjectsManagerData> projectList)
    throws RemoteException, Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    DiskUsageServiceInterface dus = ClientRemoteLocator.diskUsageService;
    /*
     * Pegamos os projetos cujo dono  o usurio.
     */
    ProjectsManager projectsManager = getProjectsManager();
    User user = projectsManager.getUser();
    Object ownerId = user.getId();
    List<UserProjectInfo> projects = psi.getProjectsFromUser(ownerId);
    if (projects == null) {
      return;
    }
    for (UserProjectInfo prj : projects) {
      Object projectId = prj.getProjectId();
      ProjectAdminInfo projectAdminInfo = psi.getProjectAdminInfo(projectId);
      if (projectAdminInfo != null
        && !projectAdminInfo.isUnlockedWithAreaAllocated()) {
        continue;
      }
      try {
        CommonClientProject ccp = psi.openProject(projectId, false);
        long allocatedSpace = ccp.getLockingAreaSize();
        /*
         * Se o projeto nao tem alocao, ento ele no foi lido pelo loop
         * anterior.
         */
        ProjectSpaceAllocation psa =
          ProjectSpaceAllocation.getItemFromSpace(allocatedSpace);
        if (psa == ProjectSpaceAllocation.NO_ALLOCATION
          || psa == ProjectSpaceAllocation.ALLOCATED) {
          String userId = (String) ccp.getUserId();
          String name = ccp.getName();
          ProjectsManagerData pmd =
            new ProjectsManagerData(ccp.getId(), ccp.getName(), ccp.getUserId());
          pmd.setOwnerName(User.getName(userId));
          pmd.setProjectSpaceAllocation(psa);
          if (psa == ProjectSpaceAllocation.ALLOCATED) {
            pmd.setAllocatedSpace(ccp.getLockingAreaSize());
          }
          pmd.setProjectDescription(ccp.getDescription());
          pmd.setCreationDate(ccp.getCreationDate());
          pmd.setServerOwnerName(ccp.getOwnerServerName());
          pmd.setLocationInServer(ccp.getLocationInServer());
          pmd.setModificationDate(ccp.getLastModificationDate());
          if (loadOccupiedSpace) {
            pmd.setOccupiedSpace(dus.getUsedSpaceForProjectMb(userId, name));
          }
          else {
            pmd.setOccupiedSpace(-1);
          }
          /*
           * FIXME - try adicionado para que excees lanadas por tentativa de
           * abrir projetos no formato de permisses antigo no impeam a
           * abertura do gerenciador de projetos.
           */
          try {
            pmd.setSharingType(ccp.getSharingType());
          }
          catch (IllegalStateException ise) {
            ccp.close(false);
            continue;
          }
          pmd.setUsers(ccp.getUsersRO(), ccp.getUsersRW());
          ccp.close(false);
          projectList.add(pmd);
        }
      }
      catch (BugException be) {
        /*
         * Esta exceo  lanada quando o arquivo .csbase_project_info do
         * projeto que estamos tentando abrir no existe ou est corrompido. A
         * execuo no deve parar por causa disto, apenas uma mensagem de erro
         * deve ser exibida.
         */
        failedProjects.add(prj);
        continue;
      }
    }
  }

  /**
   * Pega os projetos onde o usurio no  o criador do projeto,  apenas um
   * usurio compartilhado.
   * 
   * @param projectList Lista de projetos a ser atualizada.
   * @throws RemoteException em caso de falha na execuo de mtodo remoto.
   * @throws Exception em caso de falha geral.
   */
  private void getProjectsWhereUserIsNotOwner(
    List<ProjectsManagerData> projectList) throws RemoteException, Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    DiskUsageServiceInterface dus = ClientRemoteLocator.diskUsageService;
    ProjectsManager projectsManager = getProjectsManager();
    User currentUser = projectsManager.getUser();
    /*
     * Agora pegamos os projetos que no tem nenhuma alocao e cujo dono no 
     * o usurio corrente da aplicao.
     * 
     * TODO - Possivel bug (contornado). Projetos vem repetidos.
     */
    List<UserProjectInfo> usrPrjInf =
      psi.getProjectsSharedWithUser(currentUser.getId());
    for (UserProjectInfo upi : usrPrjInf) {
      CommonClientProject ccp;
      try {
        ccp = psi.openProject(upi.getProjectId(), false);
        if (ccp.isPublic()) {
          if (checkProjectAlreadyInserted(projectList, upi.getOwnerId(),
            upi.getProjectName())) {
            continue;
          }
        }
        long allocatedSpace = ccp.getLockingAreaSize();
        /*
         * Se o projeto nao tem alocao, ento ele no foi lido pelo loop
         * anterior.
         */
        ProjectSpaceAllocation psa =
          ProjectSpaceAllocation.getItemFromSpace(allocatedSpace);
        if (psa == ProjectSpaceAllocation.NO_ALLOCATION
          || psa == ProjectSpaceAllocation.ALLOCATED) {
          ProjectsManagerData pmd =
            new ProjectsManagerData(ccp.getId(), ccp.getName(), ccp.getUserId());
          pmd.setOwnerName(User.getName(ccp.getUserId()));
          pmd.setProjectSpaceAllocation(psa);
          pmd.setProjectDescription(ccp.getDescription());
          pmd.setCreationDate(ccp.getCreationDate());
          pmd.setServerOwnerName(ccp.getOwnerServerName());
          pmd.setLocationInServer(ccp.getLocationInServer());
          pmd.setModificationDate(ccp.getLastModificationDate());
          if (loadOccupiedSpace) {
            pmd.setOccupiedSpace(dus.getUsedSpaceForProjectMb(ccp.getUserId(),
              ccp.getName()));
          }
          else {
            pmd.setOccupiedSpace(-1);
          }
          pmd.setSharingType(ccp.getSharingType());
          pmd.setUsers(ccp.getUsersRO(), ccp.getUsersRW());
          ccp.close(false);
          projectList.add(pmd);
        }
      }
      catch (BugException be) {
        /*
         * Esta exceo  lanada quando o arquivo .csbase_project_info do
         * projeto que estamos tentando abrir no existe ou est corrompido. A
         * execuo no deve parar por causa disto, apenas uma mensagem de erro
         * deve ser exibida.
         */
        failedProjects.add(upi);
        continue;
      }
    }
  }

  /**
   * Verifica se determinado projeto ja foi adicionado.
   * <p>
   * TODO - Este mtodo s  necessrio caso haja mesmo um bug na task
   * mencionada acima.
   * 
   * @param projectList vetor de projetos.
   * @param userId Id do usurio
   * @param projectName Nome do projeto
   * @return boolean indicando se o projeto j foi inserido.
   */
  private static boolean checkProjectAlreadyInserted(
    List<ProjectsManagerData> projectList, Object userId, String projectName) {
    for (ProjectsManagerData pmd : projectList) {
      if (pmd.getProjectName().equals(projectName)
        && pmd.getOwnerId().equals(userId)) {
        return true;
      }
    }
    return false;
  }
}