/*
 * $Id: ReloginDialog.java 150777 2014-03-19 14:16:56Z oikawa $
 */

package csbase.client.login;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.URL;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;
import javax.swing.border.EmptyBorder;

import tecgraf.javautils.core.lng.LNG;
import csbase.logic.LoginInfo;

/**
 * Cria uma janela modal para obter informaes do login do usurio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ReloginDialog {
  /**
   * Dilogo
   */
  private JDialog loginDialog;

  /**
   * Texto para login
   */
  private JTextField loginTextField;

  /**
   * Texto de senha
   */
  private JTextField passwordTextField;

  /**
   * Informativo de login
   */
  private LoginInfo info;

  /**
   * Path de imagem
   */
  private final String imagePath;

  /**
   * Adiciona dois componentes para exibio em um painel.
   * 
   * @param panel painel em que os componenetes sero adicionados.
   * @param label string para um JLabel que ser adicionado.
   * @param component componente que ser adicionado.
   */
  private void addComponents(final JPanel panel, final String label,
    final JComponent component) {
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.NORTHWEST;
    constraints.insets = new Insets(4, 4, 4, 0);

    panel.add(new JLabel(label), constraints);

    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    panel.add(component, constraints);
  }

  /**
   * Encerra a janela de relogin.
   */
  private void cancelButtonPressed() {
    info = null;
    exit();
  }

  /**
   * Criao de um panel com os botes para login e cancelamento.
   * 
   * @return Painel.
   */
  protected JPanel createButtonPanel() {
    final JButton loginButton = new JButton(LNG.get("relogin.button.login"));
    loginButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        loginButtonPressed();
      }
    });
    loginDialog.getRootPane().setDefaultButton(loginButton);
    final JButton cancelButton = new JButton(LNG.get("relogin.button.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        cancelButtonPressed();
      }
    });

    final Dimension buttonDimension =
      new Dimension(Math.max(loginButton.getPreferredSize().width,
        cancelButton.getPreferredSize().width), Math.max(
        loginButton.getPreferredSize().height,
        cancelButton.getPreferredSize().height));
    loginButton.setPreferredSize(buttonDimension);
    cancelButton.setPreferredSize(buttonDimension);
    final JPanel buttonP = new JPanel();
    buttonP.setBorder(new EmptyBorder(5, 0, 5, 0));
    buttonP.add(loginButton);
    buttonP.add(cancelButton);
    return buttonP;
  }

  /**
   * Cria a interface da janela.
   */
  protected void createGUI() {
    loginDialog = createLoginDialog();
    final JPanel loginPanel = new JPanel(new BorderLayout());
    loginPanel.add(createMainPanel(), BorderLayout.CENTER);
    final JLabel statusBar = createStatusBarLabel();
    loginPanel.add(statusBar, BorderLayout.SOUTH);
    loginDialog.getContentPane().add(loginPanel);
    loginDialog.pack();
  }

  /**
   * Cria e configura um Dialog para janela login.
   * 
   * @return Dilogo onde os componentes da interface devem ser adicionados.
   */
  protected JDialog createLoginDialog() {
    final JDialog dialog =
      new JDialog((JFrame) null, LNG.get("relogin.title"), true);
    dialog.setResizable(false);
    dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(final WindowEvent e) {
        exit();
      }
    });
    return dialog;
  }

  /**
   * Cria um label com imagem que representa o logo do sistema para compor a
   * interface.
   * 
   * @return label com a imagem, null caso no tenha definido nenhuma imagem
   */
  protected JLabel createLoginImageLabel() {
    final Image loginImage = getLoginImage();
    if (loginImage == null) {
      return null;
    }
    return new JLabel(new ImageIcon(loginImage));
  }

  /**
   * Cria o painel principal da janela de login que  composto de:
   * 
   * <ul>
   * <li>
   * Um painel com os campos de entrada de dados pelo usurio criado por
   * {@link #createTextPanel()} em <code>BorderLayout.CENTER</code>.</li>
   * <li>
   * Um painel com butes OK/CANCEL criado por {@link #createButtonPanel()} em
   * <code>BorderLayout.SOUTH</code>.</li>
   * </ul>
   * 
   * 
   * @return Painel principal.
   */
  protected JPanel createMainPanel() {
    final JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.setBorder(new EmptyBorder(5, 5, 5, 5));
    final JLabel loginImageLabel = createLoginImageLabel();
    if (loginImageLabel != null) {
      mainPanel.add(createLoginImageLabel(), BorderLayout.WEST);
    }
    mainPanel.add(createTextPanel(), BorderLayout.CENTER);
    mainPanel.add(createButtonPanel(), BorderLayout.SOUTH);
    return mainPanel;
  }

  /**
   * Cria uma barra de status para a janela de login.
   * 
   * @return um Label que representa uma barra de status.
   */
  protected JLabel createStatusBarLabel() {
    final JLabel label = new JLabel(LNG.get("relogin.message.relogin"));
    label.setBorder(BorderFactory.createLoweredBevelBorder());
    return label;
  }

  /**
   * Criao do painel com os campos de entrada do usurio.
   * 
   * @return Painel.
   */
  protected JPanel createTextPanel() {
    final JPanel textPanel = new JPanel(new GridBagLayout());
    loginTextField = new JTextField(12);
    passwordTextField = new JPasswordField(12);

    final String loginText = LNG.get("relogin.label.login");
    final String passwdText = LNG.get("relogin.label.password");

    this.addComponents(textPanel, loginText + ":", loginTextField);
    this.addComponents(textPanel, passwdText + ":", passwordTextField);
    return textPanel;
  }

  /**
   * Fecha e destri a janela de relogin.
   */
  protected void exit() {
    loginDialog.setVisible(false);
    loginDialog.dispose();
  }

  /**
   * Obtm a imagem armazenada em um arquivo.
   * 
   * @param path path para o arquivo que contm a imagem
   * 
   * @return a imagem
   */
  private Image getImage(final String path) {
    if (path != null && !path.equals("")) {
      try {
        final URL res = getClass().getResource(path);
        final ImageIcon icon = new ImageIcon(res);
        return icon.getImage();
      }
      catch (final Exception e) {
        System.out.println("Falha na carga de imagem: " + path);
        return null;
      }
    }
    return null;
  }

  /**
   * Consulta da imagem de login do cliente.
   * 
   * @return a imagem, <code>null</code> se no houver nenhuma disponvel.
   */
  protected Image getLoginImage() {
    return getImage(imagePath);
  }

  /**
   * Retorna as informaes de login que usurio entrou atravs do dilogo.
   * 
   * @return Informaes de login.
   */
  public LoginInfo getLoginInfo() {
    return info;
  }

  /**
   * Valida e guarda os campos de entrada do usurio. Chamado sempre que o boto
   * de login  pressionado.
   */
  private void loginButtonPressed() {
    final String loginText = loginTextField.getText();
    final String passwordText = passwordTextField.getText();
    final Locale locale = LNG.getLocale();
    info = new LoginInfo(loginText, passwordText, locale);
    exit();
  }

  /**
   * Mostra ou esconde a janela de relogin.
   * 
   * @param visible <code>true</code> para exibir, <code>false</code> para
   *        esconder
   */
  public void setVisible(final boolean visible) {
    // Centralizao do dilogo de login
    final Dimension wSize = loginDialog.getSize();

    // Centralizao do dilogo de login
    final Dimension sSize = Toolkit.getDefaultToolkit().getScreenSize();
    final int newX = (sSize.width - wSize.width) / 2;
    final int newY = (sSize.height - wSize.height) / 2;
    loginDialog.setLocation(newX, newY);
    loginDialog.setVisible(visible);
  }

  /**
   * Cria uma janela modal para solicitar login e senha ao usurio.
   * 
   * @param imagePath imagem do dilogo.
   */
  public ReloginDialog(final String imagePath) {
    this.imagePath = imagePath;
    createGUI();
  }
}
