/**
 * $Id: FlowApplication.java 108535 2010-08-01 02:16:57Z isabella $
 */

package csbase.client.applications.flowapplication;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.datatransfer.Clipboard;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DragGestureEvent;
import java.awt.dnd.DragGestureListener;
import java.awt.dnd.DragSource;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.KeyStroke;
import javax.swing.ScrollPaneConstants;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.MenuButton;
import tecgraf.javautils.gui.MenuButton.PopupPosition;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.tree.FilterableTreePanel;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationEditPrefAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.flowapplication.actions.AlgorithmComponentFocusAction;
import csbase.client.applications.flowapplication.actions.AskForParameterValuesAction;
import csbase.client.applications.flowapplication.actions.CloseAction;
import csbase.client.applications.flowapplication.actions.CopyElementsAction;
import csbase.client.applications.flowapplication.actions.ExecuteGraphAction;
import csbase.client.applications.flowapplication.actions.ExportMultipleConfigurationAction;
import csbase.client.applications.flowapplication.actions.GraphLayoutAction;
import csbase.client.applications.flowapplication.actions.LoadAction;
import csbase.client.applications.flowapplication.actions.MultipleExecuteGraphAction;
import csbase.client.applications.flowapplication.actions.NewAction;
import csbase.client.applications.flowapplication.actions.OrthogonalLinkLayoutAction;
import csbase.client.applications.flowapplication.actions.PasteElementsAction;
import csbase.client.applications.flowapplication.actions.RemoveElementsAction;
import csbase.client.applications.flowapplication.actions.SaveAction;
import csbase.client.applications.flowapplication.actions.SaveAsAction;
import csbase.client.applications.flowapplication.actions.ShowParameterValuesAction;
import csbase.client.applications.flowapplication.actions.SingleColumnGraphLayoutAction;
import csbase.client.applications.flowapplication.actions.UniformSizeLayoutAction;
import csbase.client.applications.flowapplication.actions.UpdateAlgorithmTreeAction;
import csbase.client.applications.flowapplication.actions.UpdateAllVersionsAction;
import csbase.client.applications.flowapplication.commandviewer.CommandPropertiesFilter;
import csbase.client.applications.flowapplication.commandviewer.CommandViewerPanel;
import csbase.client.applications.flowapplication.filters.AskForParameterValuesActionFilter;
import csbase.client.applications.flowapplication.filters.CopyAndPasteFilter;
import csbase.client.applications.flowapplication.filters.HighlightElementFilter;
import csbase.client.applications.flowapplication.filters.HintElementFilter;
import csbase.client.applications.flowapplication.filters.PopupFilter;
import csbase.client.applications.flowapplication.filters.ResizeNodeFilter;
import csbase.client.applications.flowapplication.filters.SelectElementFilter;
import csbase.client.applications.flowapplication.filters.UpdateStatusBarFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.FileTypeColorManager;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphFileDescriptor;
import csbase.client.applications.flowapplication.graph.GraphLink;
import csbase.client.applications.flowapplication.graph.GraphListener;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.applications.flowapplication.graph.actions.BreakLinkAction;
import csbase.client.applications.flowapplication.graph.actions.BypassAction;
import csbase.client.applications.flowapplication.graph.actions.CopyParameterValuesAction;
import csbase.client.applications.flowapplication.graph.actions.PasteParameterValuesAction;
import csbase.client.applications.flowapplication.graph.actions.RemoveSelectedElementsAction;
import csbase.client.applications.flowapplication.graph.actions.UpdateVersionAction;
import csbase.client.applications.flowapplication.messages.PickGraphMessage;
import csbase.client.applications.flowapplication.messages.RemoveElementsMessage;
import csbase.client.applications.flowapplication.messages.ResetMessage;
import csbase.client.applications.flowapplication.messages.ShowParameterValuesMessage;
import csbase.client.applications.flowapplication.util.FlowApplicationRemoteTask;
import csbase.client.applications.flowapplication.zoom.ZoomControl;
import csbase.client.desktop.DesktopFrame;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.util.PreferenceListener;
import csbase.client.project.ProjectTreePath;
import csbase.client.remote.AlgorithmManagementListener;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy.AlgorithmOperation;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.sga.CommandExecutionDialog;
import csbase.client.util.sga.CommandRequestedListener;
import csbase.client.util.sga.MultipleFlowCommandExecutionDialog;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;
import csbase.logic.algorithms.commands.CommandPersistenceNotification;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowAlgorithmParser;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * Applicao de Construo de Fluxos de Algoritmo.
 */
public final class FlowApplication extends ApplicationProject {

  /**
   * O identificar de mensagens enviadas com configuradores para o aplicativo.
   */
  public static final String CONFIGURATOR_MESSAGE = "CONFIGURATOR_MESSAGE";

  /**
   * Posio relativa do splitter vertical.
   */
  private static final double VERTICAL_SPLITTER_LOCATION = 0.75;

  /**
   * rvore de algoritmos.
   */
  private FilterableTreePanel algorithmVersionTreePanel;

  /**
   * Espao de trabalho, onde o fluxo pode ser editado.
   */
  private final Workspace workspace;

  /**
   * Painel de rolagem que engolba o espao de trabalho.
   */
  private JScrollPane workspaceScrollPane;

  /**
   * Visualizador de comandos do editor.
   */
  private final CommandViewerPanel commandViewerPanel;

  /**
   * Controle de nvel de zoom.
   */
  private ZoomControl zoomControl;

  /**
   * A barra de status da aplicao.
   */
  private StatusBar statusBar;

  /**
   * Painel de split horizontal, que contm a rvore de algoritmos e o fluxo
   */
  private JSplitPane horizontalSplitPane;

  /**
   * Observadores interessados nos comandos solicitados com o dilogo de
   * execuo de comandos.
   */
  private final List<CommandRequestedListener> listeners;

  /**
   * Conjunto de comandos submetidos via essa aplicao, que devem ser
   * monitorados at que sejam terminados.
   */
  private final Set<String> commandIds;

  /**
   * Observador de comandos persistidos.
   */
  private final IMessageListener commandPersistenceListener;

  /**
   * Lista de itens de menu pop-up.
   */
  private List<WorkspaceFilter> popupFilters;

  /**
   * Filtro para menu pop-up
   */
  private PopupFilter basicPopupFilter;

  /**
   * Filtro para redimensionamento de ns.
   */
  private ResizeNodeFilter resizeNodeFilter;

  /**
   * A rea de transferncia compartilhada entre as instncias da aplicao.
   */
  private static Clipboard clipboard = new Clipboard(FlowApplication.class
    .getName());

  /**
   * Filtro para ao de copiar/colar.
   */
  private CopyAndPasteFilter copyAndPasteFilter;

  /**
   * Filtro para abrir a janela de configurao de parmetros.
   */
  private AskForParameterValuesActionFilter askForParameterValuesFilter;

  /**
   * Filtro para destacar elementos do fluxo.
   */
  private HighlightElementFilter hightLightFilter;

  /**
   * Filtro para a exibio de dicas.
   */
  private HintElementFilter hintFilter;

  /**
   * Filtro para a seleo de elementos do fluxo.
   */
  private SelectElementFilter selectElementFilter;

  /**
   * Filtro para atualizao da barra de status
   */
  private UpdateStatusBarFilter updateStatusBarFilter;

  /**
   * Ao de carregar um arquivo de fluxo.
   */
  private LoadAction loadAction;

  /**
   * Ao de executar um fluxo.
   */
  private ExecuteGraphAction executeAction;

  /**
   * Ao de exportar configurao para execuo de mltiplos fluxos.
   */
  private ExportMultipleConfigurationAction exportMultipleConfigurationAction;

  /**
   * Ao de executar mltiplos fluxos.
   */
  private MultipleExecuteGraphAction multipleExecuteAction;

  /**
   * Ao de criao de novo fluxo.
   */
  private NewAction newAction;

  /**
   * Ao de remoo de elemento do fluxo.
   */
  private RemoveElementsAction removeAction;

  /**
   * Ao de fechar fluxo.
   */
  private CloseAction closeAction;

  /**
   * Ao de salvar fluxo.
   */
  private SaveAction saveAction;

  /**
   * Ao de salvar fluxo como um novo arquivo.
   */
  private SaveAsAction saveAsAction;

  /**
   * Ao de atualizar a verso de todos os ns do fluxo.
   */
  private UpdateAllVersionsAction updateVersionsAction;

  /**
   * Ao de atualizar rvore de algoritmos do editor.
   */
  private UpdateAlgorithmTreeAction updateAlgorithmTreeAction;

  /**
   * Ao de mostrar propriedades de um elemento do fluxo.
   */
  private ShowParameterValuesAction showPropertiesAction;

  /**
   * Ao de copiar elementos do fluxo.
   */
  private CopyElementsAction copyAction;

  /**
   * Ao de colar elementos do fluxo.
   */
  private PasteElementsAction pasteAction;

  /**
   * Caminho para o arquivo de fluxo atual (se houver).
   */
  private ProjectTreePath currentFilePath;

  /**
   * Sinal que indica se o fluxo foi alterado depois de salvo.
   */
  private boolean wasChanged;

  /**
   * Campo que armazena a descrio do comando a ser executado. Valor aparecer
   * como default.
   */
  private JTextField descriptionTextField;

  /**
   * Boto com menu popup associado, para atualizao (refresh) da rvore de
   * algoritmos e categorias e detalhamento das mudanas na barra de status
   */
  private MenuButton refreshMenuButton;

  /**
   * Item de detalhes do boto com menu popup associado, para atualizao
   * (refresh) da rvore de algoritmos e categorias.
   */
  private JMenuItem algTreeRefreshDetail;

  /** Lista com as categorias modificadas no cache do cliente */
  private List<Category> modifiedCategories;

  /** Lista com os algoritmos modificados no cache do cliente */
  private List<AlgorithmInfo> modifiedAlgorithms;

  /** Indica que o repositrio de algoritmos foi recarregado no servidor */
  private boolean algorithmRepositoryReloaded;

  /**
   * Gerenciador de foco da aplicao.
   */
  private KeyboardFocusManager keyFocusManager;

  /**
   * Gerenciador de eventos de teclado da aplicao.
   */
  private FlowApplicationKeyDispatcher keyDispatcher;

  /**
   * Cpia local das preferncias do aplicativo.
   */
  private PreferenceCategory preferences;

  /**
   * Constri a aplicao de construo de fluxos de algoritmo.
   *
   * @param id O identificador.
   * @throws ApplicationException caso ocorra erro na inicializao da
   *         aplicao.
   */
  public FlowApplication(final String id) throws ApplicationException {
    super(id);
    algorithmRepositoryReloaded = false;
    PreferenceCategory globalPreferences = getPreferences();
    preferences = globalPreferences.copy();
    boolean showVersionInWorkspace =
      preferences
      .getPreferenceAsBoolean(FlowApplicationPref.SHOW_VERSION_IN_WORKSPACE);
    boolean showVersionInTree =
      preferences
      .getPreferenceAsBoolean(FlowApplicationPref.SHOW_VERSION_IN_TREE);
    boolean autoCreateLinks =
      preferences.getPreferenceAsBoolean(FlowApplicationPref.AUTO_CREATE_LINKS);
    commandIds = new HashSet<String>();
    listeners = new LinkedList<CommandRequestedListener>();

    addPreferenceListeners();
    /*
     * Inicializao de listeners, atributos e painis relacionados a rvore de
     * algoritmos e categorias.
     */
    addAlgorithmProxyListener();
    initAlgorithmsAndCategoriesUpdateFields();
    algorithmVersionTreePanel = createAlgorithmTreePanel(showVersionInTree);
    keyFocusManager = KeyboardFocusManager.getCurrentKeyboardFocusManager();
    keyDispatcher = new FlowApplicationKeyDispatcher();
    keyFocusManager.addKeyEventDispatcher(keyDispatcher);

    commandViewerPanel =
      new CommandViewerPanel(getApplicationFrame(), getApplicationProject()
        .getId(), createCommandPropertiesFilter(),
        CommandViewerPanel.ToolBarLocation.LEFT) {
      @Override
      protected boolean exportCommand(final CommandInfo command) {
        final FlowApplicationRemoteTask<AlgorithmConfigurator> task =
          new FlowApplicationRemoteTask<AlgorithmConfigurator>() {
          @Override
          public void performTask() throws Exception {
            final AlgorithmConfigurator configurator =
              command.getConfigurator();
            setResult(configurator);
          }
        };
        final String msg = getString("loadingConfigurator");
        if (!task.execute(getApplicationFrame(), msg, msg)) {
          return false;
        }
        final AlgorithmConfigurator configurator = task.getResult();
        if (configurator.getConfiguratorType() != ConfiguratorType.FLOW) {
          return false;
        }
        if (!createNewGraph()) {
          return false;
        }
        return editConfigurator((FlowAlgorithmConfigurator) configurator,
          command.getDescription());
      }
    };
    workspace = new Workspace(getApplicationFrame(), true, false);
    workspace.setAutoCreateLinks(autoCreateLinks);
    workspace.setVersionInfoVisible(showVersionInWorkspace);
    workspace.getGraph().setVersionInfoVisible(showVersionInWorkspace);
    workspaceScrollPane =
      new JScrollPane(workspace, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
        ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
    workspace.getGraph().addGraphListener(new ChangedListener());
    final JTree tree = algorithmVersionTreePanel.getTree();
    workspace.setTransferHandler(tree.getTransferHandler());
    createFilters();
    final ListIterator<WorkspaceFilter> popupFilterIterator =
      popupFilters.listIterator(popupFilters.size());
    while (popupFilterIterator.hasPrevious()) {
      final WorkspaceFilter filter = popupFilterIterator.previous();
      filter.attach();
    }
    selectElementFilter.attach();
    resizeNodeFilter.attach();
    hightLightFilter.attach();
    hintFilter.attach();
    updateStatusBarFilter.attach();
    basicPopupFilter.attach();
    askForParameterValuesFilter.attach();
    copyAndPasteFilter.attach();
    createGui();
    createNewGraph();
    try {
      final List<String> fileTypes =
        getStringListSpecificProperty("file.type.name");
      final List<String> rgbs = getStringListSpecificProperty("file.type.rgb");
      FileTypeColorManager.createInstance(fileTypes, rgbs);
    }
    catch (final ParseException e) {
      throw new ApplicationException(e.getLocalizedMessage(), e);
    }
    commandPersistenceListener = new IMessageListener() {
      @Override
      public void onMessagesReceived(Message... messages) throws Exception {
        for (Message message : messages) {
          CommandPersistenceNotification notification =
            (CommandPersistenceNotification) message.getBody();
          switch (notification.getType()) {
            case REMOVED: {
              commandIds.remove(notification.getCommandId());
              break;
            }
            case SAVED: {
              break;
            }
            case UPDATED: {
              break;
            }
            default: {
              String errorMessage =
                String
                .format(
                  "Notificao de tipo desconhecido.\nProjeto: %s.\nComando: %s.\nTipo de notificao: %s.\n",
                  notification.getProjectId(), notification.getCommandId(),
                  notification.getType());
              throw new IllegalArgumentException(errorMessage);
            }
          }
        }
      }
    };
  }

  /**
   * Adiciona observadores de mudanas nas preferncias da aplicao.
   */
  private void addPreferenceListeners() {
    PreferenceValue<Boolean> versionInWorkspacePref =
      preferences.getPVBoolean(FlowApplicationPref.SHOW_VERSION_IN_WORKSPACE);
    versionInWorkspacePref
    .addPreferenceListener(new PreferenceListener<Boolean>() {
      @Override
      public void valueChanged(Boolean oldValue, Boolean newValue) {
        if (oldValue == null || !oldValue.equals(newValue)) {
          workspace.setVersionInfoVisible(newValue);
        }
      }
    });
    PreferenceValue<Boolean> versionInTreePref =
      preferences.getPVBoolean(FlowApplicationPref.SHOW_VERSION_IN_TREE);
    versionInTreePref.addPreferenceListener(new PreferenceListener<Boolean>() {
      @Override
      public void valueChanged(Boolean oldValue, Boolean newValue) {
        if (oldValue == null || !oldValue.equals(newValue)) {
          reloadAlgorithmTree();
        }
      }
    });
    PreferenceValue<Boolean> autoCreateLinksPref =
      preferences.getPVBoolean(FlowApplicationPref.AUTO_CREATE_LINKS);
    autoCreateLinksPref
    .addPreferenceListener(new PreferenceListener<Boolean>() {
      @Override
      public void valueChanged(Boolean oldValue, Boolean newValue) {
        if (oldValue == null || !oldValue.equals(newValue)) {
          workspace.setAutoCreateLinks(newValue);
        }
      }
    });
  }

  /**
   * Inicializa os campos usados para indicar as atualizaes de algoritmos e
   * categorias, a partir de mudanas ocorridas nos respectivos caches no
   * cliente.
   */
  private void initAlgorithmsAndCategoriesUpdateFields() {
    modifiedAlgorithms = new Vector<AlgorithmInfo>();
    modifiedCategories = new Vector<Category>();
    algorithmRepositoryReloaded = false;
  }

  /**
   * Adiciona um listener para o proxy de algoritmos.
   */
  private void addAlgorithmProxyListener() {
    AlgorithmManagementProxy
    .addManagementListener(new AlgorithmManagementListener() {

      @Override
      public void categoryRemoved(Category category) {
        modifiedCategories.add(category);
        enableAlgTreeRefreshDetailButtonItem();
      }

      @Override
      public void categoryCreated(Category category) {
        modifiedCategories.add(category);
        enableAlgTreeRefreshDetailButtonItem();
      }

      @Override
      public void categoryUpdated(CategorySet modifiedCategorySet) {
        if (modifiedCategorySet != null) {
          SortedSet<Category> categories =
            modifiedCategorySet.getCategories();
          for (Category category : categories) {
            modifiedCategories.add(modifiedCategorySet.getCategory(category
              .getId()));
          }
          if (!categories.isEmpty()) {
            enableAlgTreeRefreshDetailButtonItem();
          }
        }
      }

      @Override
      public void algorithmCreated(AlgorithmInfo algoInfo) {
        modifiedAlgorithms.add(algoInfo);
        enableAlgTreeRefreshDetailButtonItem();
      }

      @Override
      public void algorithmRemoved(AlgorithmInfo algoInfo) {
        modifiedAlgorithms.add(algoInfo);
        enableAlgTreeRefreshDetailButtonItem();
      }

      @Override
      public void algorithmUpdated(AlgorithmInfo algoInfo) {
        modifiedAlgorithms.add(algoInfo);
        enableAlgTreeRefreshDetailButtonItem();
      }

      @Override
      public void algorithmsReloaded() {
        algorithmRepositoryReloaded = true;
        enableAlgTreeRefreshDetailButtonItem();
      }

      private void enableAlgTreeRefreshDetailButtonItem() {
        Runnable handler = new Runnable() {
          @Override
          public void run() {
            algTreeRefreshDetail.setEnabled(true);
          }
        };
        // Executa o tratador do evento na EDT.
        if (SwingThreadDispatcher.isEventDispatchThread()) {
          handler.run();
        }
        else {
          SwingThreadDispatcher.invokeLater(handler);
        }
      }

    });
  }

  /**
   * Obter todas as categorias disponveis no servidor, que contm algoritmos
   * que tem permisso para ser executados. As categorias vazias, ou seja, que
   * no tem algoritmos com permisso, no devem ser exibidas.
   *
   * @return as categorias com algoritmos com permisso para serem executados
   */
  public SortedSet<Category> getCategories() {
    SortedSet<Category> categories = null;
    CategorySet categorySet =
      AlgorithmManagementProxy.getAllCategories(getApplicationFrame(),
        AlgorithmOperation.EXECUTE_ALGORITHM);
    categories = categorySet.getCategories();
    filterNotEmptyCategories(categories.iterator());
    return categories;
  }

  /**
   * Filtrar as categorias razes que no estejam vazias, isto , que tenham um
   * ou mais algoritmos com permisso para executar.
   *
   * Se uma subcategoria, em algum nivel da sua hierarquia, contm um algoritmo
   * para execuo, ento toda a hierarquia  exibida, ainda que em outros
   * nveis estejam vazias. Caso contrrio, a categoria raiz  removida do
   * conjunto de categorias a serem exibidas na aplicao.
   *
   * @param categoryIterator iterador de categorias razes da rvore de
   *        algoritmos
   */
  private void filterNotEmptyCategories(
    final Iterator<Category> categoryIterator) {
    while (categoryIterator.hasNext()) {
      final Category category = categoryIterator.next();
      if (!category.getAlgorithms().isEmpty()) {
        continue;
      }

      SortedSet<Category> subCategories = category.getCategories();
      if (subCategories != null) {
        boolean hasSubCatNotEmpty =
          verifyAnySubCategoryContainsAlgorithms(subCategories.iterator());
        // false);
        if (hasSubCatNotEmpty) {
          continue;
        }
      }
      if (category.getAlgorithms().isEmpty()) {
        categoryIterator.remove();
      }
    }
  }

  /**
   * Verifica recursivamente se existe pelo menos uma sub-categoria que contm
   * algoritmos com permisso para execuo.
   *
   * @param categoryIterator iterador das categorias filhas da categoria raiz
   *
   * @return Se houver ao menos uma, esse mtodo retorna true, indicando que
   *         toda a hierarquia da categoria deve ser exibida. Caso contrrio, se
   *         todas as subcategorias da categoria raiz estiverem vazias, ento
   *         esse mtodo retorna false.
   */
  private boolean verifyAnySubCategoryContainsAlgorithms(
    final Iterator<Category> categoryIterator) {
    boolean hasAlgorithms = false;

    while (categoryIterator.hasNext()) {
      final Category category = categoryIterator.next();
      if (!category.getAlgorithms().isEmpty()) {
        hasAlgorithms = true;
        return hasAlgorithms;
      }
      SortedSet<Category> subCategories = category.getCategories();
      Iterator<Category> subCatIterator = subCategories.iterator();
      hasAlgorithms = verifyAnySubCategoryContainsAlgorithms(subCatIterator);
    }
    return hasAlgorithms;
  }

  /**
   * Obter todos os algoritmos disponveis no servidor, cujos algoritmos tem
   * permisso para ser executados.
   *
   * @return os algoritmos com permisso para serem executados
   */
  public SortedSet<AlgorithmInfo> getAlgorithms() {
    AlgorithmInfo[] algorithmInfos =
      AlgorithmManagementProxy.getAllAlgorithmInfos(getApplicationFrame(),
        AlgorithmOperation.EXECUTE_ALGORITHM);

    final SortedSet<AlgorithmInfo> uncategorizedAlgorithms =
      new TreeSet<AlgorithmInfo>();
    uncategorizedAlgorithms.addAll(Arrays.asList(algorithmInfos));

    final SortedSet<Category> categories = getCategories();
    for (final Category category : categories) {
      final Iterator<AlgorithmInfo> algorithmIterator =
        uncategorizedAlgorithms.iterator();
      while (algorithmIterator.hasNext()) {
        if (category.containsAlgorithmInChildren(algorithmIterator.next())) {
          algorithmIterator.remove();
        }
      }
    }
    return uncategorizedAlgorithms;
  }

  /**
   * Constri o painel com a rvore de algoritmos.
   *
   * @param showVersions Indica se devem ser mostradas as verses do algoritmo
   *        na rvore.
   *
   * @return O painel com a rvore de algoritmos.
   */
  private FilterableTreePanel createAlgorithmTreePanel(boolean showVersions) {
    final FilterableTreePanel algoVersionTreePanel =
      new FilterableTreePanel(new AllCategoryTreeNode(getCategories(),
        getAlgorithms(), showVersions));
    final DragSource dragSource = DragSource.getDefaultDragSource();
    final JTree algorithmVersionTree = algoVersionTreePanel.getTree();
    dragSource.createDefaultDragGestureRecognizer(algorithmVersionTree,
      DnDConstants.ACTION_COPY, new DragGestureListener() {

      @Override
      public void dragGestureRecognized(final DragGestureEvent dge) {
        final AlgorithmVersionInfo version = getSelectedAlgorithmVersion();

        if (version != null) {
          dge.startDrag(null, new AlgorithmVersionTransferable(version));
        }
      }

    });
    algorithmVersionTree.getSelectionModel().setSelectionMode(
      TreeSelectionModel.SINGLE_TREE_SELECTION);

    /*
     * Desabilita a expanso de ns da rvore via clique duplo para associar
     * esse evento a adicionar um novo n no fluxo.
     */
    algorithmVersionTree.setToggleClickCount(0);
    algorithmVersionTree.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2 && !e.isConsumed()
          && e.getButton() == MouseEvent.BUTTON1) {
          AlgorithmVersionInfo version = getSelectedAlgorithmVersion();
          if (version != null) {
            workspace.addNewNode(version);
          }
        }
      }
    });
    return algoVersionTreePanel;
  }

  /**
   * Solicita o foco para a caixa de busca da rvore de algoritmos.
   */
  public void requestFocusToAlgorithmComponent() {
    algorithmVersionTreePanel.getTextField().requestFocus();
  }

  /**
   * Atualiza a rvore de algoritmos.
   */
  public void reloadAlgorithmTree() {
    initAlgorithmsAndCategoriesUpdateFields();
    Runnable reloadHandler = new Runnable() {
      @Override
      public void run() {
        clearStatusBarMessage();
        horizontalSplitPane.remove(algorithmVersionTreePanel);
        Boolean showVersionInTree =
          preferences
          .getPreferenceAsBoolean(FlowApplicationPref.SHOW_VERSION_IN_TREE);
        algorithmVersionTreePanel = createAlgorithmTreePanel(showVersionInTree);
        horizontalSplitPane.setLeftComponent(algorithmVersionTreePanel);
        algTreeRefreshDetail.setEnabled(false);
      }
    };
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      reloadHandler.run();
    }
    else {
      SwingThreadDispatcher.invokeLater(reloadHandler);
    }
  }

  /**
   * Chama o configurador do algoritmo para obter os parmetros dos ns.
   */
  public void askForParameterValues() {
    final PickGraphMessage pickGraphMessage = new PickGraphMessage();
    workspace.sendVO(pickGraphMessage);
    final Graph graph = pickGraphMessage.getGraph();
    for (final GraphNode node : graph.getNodeCollection()) {
      if (node.isSelected()) {
        node.askForParameterValues();
      }
    }
  }

  /**
   * Limpa o texto da parte da barra de status especfica para mensagens.
   */
  public void clearStatusBarMessage() {
    statusBar.clearStatus();
  }

  /**
   * Fecha o fluxo atual se ele existir e cria um novo grafo.
   *
   * @return <code>true</code> sucesso ou <code>false</code> se o usurio
   *         cancelou o fechamento do fluxo.
   */
  public boolean closeGraph() {
    return createNewGraph();
  }

  /**
   * Fecha o fluxo atual se ele existir e cria um novo grafo.
   *
   * @return <code>true</code> sucesso ou <code>false</code> se o usurio
   *         cancelou o fechamento do fluxo.
   */
  public boolean createNewGraph() {
    if (!userCanKillApplication()) {
      return false;
    }
    setGraph(new Graph(getApplicationFrame()), null);
    return true;
  }

  /**
   * Edita o configurador do fluxo.
   *
   * @param configurator configurador a ser editado
   * @param cmdDescription descrio do comando.
   * @return retorna true, se a edio foi feita com sucesso, caso contrrio,
   *         retorna false
   */
  public boolean editConfigurator(final FlowAlgorithmConfigurator configurator,
    final String cmdDescription) {
    final FlowApplicationRemoteTask<Void> task =
      new FlowApplicationRemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        final Flow flow = configurator.getFlow();
        final Graph graph = new Graph(getApplicationFrame(), flow);
        String description;
        if (cmdDescription != null) {
          description = cmdDescription;
        }
        else {
          description = flow.getDescription();
        }
        setGraph(graph, description);
      }
    };
    if (task.execute(getApplicationFrame(), "Editando fluxo",
      "Carregando informaes do fluxo.")) {
      updateApplicationTitle();
      return true;
    }
    return false;
  }

  /**
   * Executar o fluxo de algoritmos corrente.
   *
   * @param configurator o configurador do fluxo a ser executado.
   */
  public void executeFlow(final FlowAlgorithmConfigurator configurator) {
    final CommandExecutionDialog commandExecutionDialog =
      new CommandExecutionDialog(getApplicationFrame(), configurator,
        getApplicationProject());
    commandExecutionDialog
    .addCommandRequestedListener(new CommandRequestedListener() {
      @Override
      public void commandsWereRequested(
        final Set<CommandInfo> submittedCommands) {
        for (final CommandInfo command : submittedCommands) {
          commandIds.add(command.getId());
        }
      }
    });
    addExternalListeners(commandExecutionDialog);
    commandExecutionDialog.setVisible(true);
  }

  /**
   * * Configura a janela de submisso para execuo mltipla do fluxo com o
   * mapeamento especificado.
   *
   * @param configurator o configurador do fluxo a ser executado.
   * @param mappings o mapeamento dos parmetros para execuo mltipla.
   */
  public void executeMultipleFlow(FlowAlgorithmConfigurator configurator,
    String[][] mappings) {
    CommandExecutionDialog commandExecutionDialog =
      new MultipleFlowCommandExecutionDialog(getApplicationFrame(),
        configurator, getApplicationProject(), mappings);
    commandExecutionDialog
    .addCommandRequestedListener(new CommandRequestedListener() {
      @Override
      public void commandsWereRequested(Set<CommandInfo> submittedCommands) {
        for (CommandInfo command : submittedCommands) {
          commandIds.add(command.getId());
        }
      }
    });
    addExternalListeners(commandExecutionDialog);
    commandExecutionDialog.setVisible(true);
  }

  /**
   *
   * Executa um determinado grafo que representa graficamente um fluxo de
   * algoritmos.
   *
   * @param graph grafo que representa o fluxo de algoritmos
   * @param cmdDescription descrio default para o comando a ser executado
   */
  public void setGraph(final Graph graph, final String cmdDescription) {
    workspace.setGraph(graph);
    setCommandDescription(cmdDescription);
  }

  /**
   * Estabelece a descrio default do comando a ser gerado pelo algoritmo.
   *
   * @param commandDescription descrio do comando
   */
  public void setCommandDescription(final String commandDescription) {
    this.descriptionTextField.setText(commandDescription);
  }

  /**
   * Adiciona listeners externos para o comando solicitado pela janela de
   * execuo de comandos. Esses listeners so cadastrados por classes que
   * executem essa aplicao.
   *
   * @param commandExecutionDialog janela de execuo de comandos
   */
  private void addExternalListeners(
    final CommandExecutionDialog commandExecutionDialog) {
    for (final CommandRequestedListener listener : listeners) {
      commandExecutionDialog.addCommandRequestedListener(listener);
    }
  }

  /**
   * Adiciona um listener para um comando solicitado com o dilogo de execuo
   * de comandos.
   *
   * @param listener listener de um comando solicitado com o dilogo de execuo
   *        de comandos
   */
  public void addCommandRequestedListener(
    final CommandRequestedListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Obtm a rea de trabalho.
   *
   * @return A rea de trabalho.
   */
  public Workspace getWorkspace() {
    return workspace;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    commandIds.clear();

    MessageProxy.removeListener(commandPersistenceListener);

    commandViewerPanel.stop();
    keyFocusManager.removeKeyEventDispatcher(keyDispatcher);
    clipboard.removeFlavorListener(pasteAction);
  }

  /**
   * Solicita ao usurio um arquivo que contenha informaes de um fluxo de
   * algoritmos e o carrega.
   *
   * @return <code>true</code> em caso de sucesso, ou <code>false</code> caso
   *         contrrio.
   */
  public boolean openGraph() {
    try {
      if (!userCanKillApplication()) {
        return true;
      }
      final ClientProjectFile file = browseFileOpen(getDefaultFileType());
      if (file != null) {
        load(file);
      }
      return true;
    }
    catch (final ApplicationException e) {
      showExceptionStack(e);
    }
    return false;
  }

  /**
   * Remove os ns e conexes que esto selecionados.
   */
  public void removeSelectedElements() {
    workspace.sendVO(new RemoveElementsMessage());
    workspace.sendVO(new ResetMessage());
  }

  /**
   * Solicita ao usurio um arquivo que contenha informaes de um fluxo de
   * algoritmos e o salva o fluxo corrente no arquivo.
   *
   * @return <code>true</code> em caso de sucesso, ou <code>false</code> caso
   *         contrrio.
   */
  public boolean saveAsGraph() {
    try {
      final ClientProjectFile file = browseFileSave(getDefaultFileType());
      if (file != null) {
        if (save(file)) {
          currentFilePath = new ProjectTreePath(file);
          updateApplicationTitle();
          return true;
        }
      }
    }
    catch (final ApplicationException e) {
      showExceptionStack(e);
    }
    return false;
  }

  /**
   * Seno houver um arquivo corrente em uso, ele solicita ao usurio um arquivo
   * que contenha informaes de um fluxo de algoritmos e o torna o arquivo
   * corrente, logo aps ele salva as informaes do algoritmo neste arquivo.
   *
   * @return <code>true</code> em caso de sucesso, ou <code>false</code> caso
   *         contrrio.
   */
  public boolean saveGraph() {
    if (currentFilePath == null) {
      return saveAsGraph();
    }
    return save(currentFilePath.getFile());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (value != null) {
      if (name.equals(PROJECT_FILE_MESSAGE)) {
        final ClientProjectFile file = (ClientProjectFile) value;
        load(file);
        return;
      }
      else if (name.equals(CONFIGURATOR_MESSAGE)) {
        final AlgorithmConfigurator algorithmConfigurator =
          (AlgorithmConfigurator) value;
        workspace.importConfigurator(algorithmConfigurator);
      }
    }
    super.sendMessage(name, value, senderId);
  }

  /**
   * Atribui uma mensagem  barra de status.
   *
   * @param message A mensagem.
   */
  public void setStatusBarMessage(final String message) {
    statusBar.setText(message);
  }

  /**
   * Exibe a parametrizao.
   */
  public void showParameterValues() {
    workspace.sendVO(new ShowParameterValuesMessage());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startApplication() throws ApplicationException {
    super.startApplication();
    updateApplicationTitle();
    commandViewerPanel.start();
    MessageProxy.addListener(commandPersistenceListener,
      CommandPersistenceNotification.class);
    commandIds.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    if (wasChanged) {
      switch (StandardDialogs.showYesNoCancelDialog(getApplicationFrame(),
        getName(), getString("askForSaving"))) {
          case JOptionPane.CANCEL_OPTION:
            return false;
          case JOptionPane.YES_OPTION:
            if (!saveGraph()) {
              return false;
            }
            break;
      }
    }
    currentFilePath = null;
    updateApplicationTitle();
    workspace.sendVO(new ResetMessage());
    return true;
  }

  /**
   * Constri as aes.
   */
  private void createActions() {
    closeAction = new CloseAction(this);
    executeAction = new ExecuteGraphAction(this);
    exportMultipleConfigurationAction =
      new ExportMultipleConfigurationAction(this);
    multipleExecuteAction = new MultipleExecuteGraphAction(this);
    loadAction = new LoadAction(this);
    newAction = new NewAction(this);
    removeAction = new RemoveElementsAction(this);
    saveAction = new SaveAction(this);
    saveAsAction = new SaveAsAction(this);
    showPropertiesAction = new ShowParameterValuesAction(this);
    updateVersionsAction = new UpdateAllVersionsAction(this);
    updateAlgorithmTreeAction = new UpdateAlgorithmTreeAction(this);
    zoomControl = new ZoomControl(workspace.getZoomModel());
    copyAction = new CopyElementsAction(this);
    pasteAction = new PasteElementsAction(this);
  }

  /**
   * Cria o filtro de propriedades do comando.
   *
   * @return filtro .
   */
  private CommandPropertiesFilter createCommandPropertiesFilter() {
    return new CommandPropertiesFilter() {

      @Override
      public boolean accept(final CommandInfo command) {
        return commandIds.contains(command.getId());
      }

    };
  }

  /**
   * Cria o menu Editar.
   *
   * @return O menu Editar.
   */
  private JMenu createEditMenu() {
    final JMenu editMenu = new JMenu(getString("editMenu"));
    editMenu.add(new JMenuItem(askForParameterValuesFilter.getAction()));
    editMenu.addSeparator();
    editMenu.add(new JMenuItem(copyAction));
    editMenu.add(new JMenuItem(pasteAction));
    editMenu.addSeparator();
    editMenu.add(new JMenuItem(removeAction));
    editMenu.addSeparator();
    editMenu.add(new JMenuItem(updateVersionsAction));
    return editMenu;
  }

  /**
   * Cria o menu Executar.
   *
   * @return O menu Executar.
   */
  private JMenu createExecuteMenu() {
    final JMenu executeMenu = new JMenu(getString("executeMenu"));
    executeMenu.add(new JMenuItem(executeAction));
    executeMenu.add(new JMenuItem(multipleExecuteAction));
    executeMenu.add(new JMenuItem(exportMultipleConfigurationAction));
    return executeMenu;
  }

  /**
   * Cria o menu Arquivo.
   *
   * @return O menu Arquivo.
   */
  private JMenu createFileMenu() {
    final JMenu fileMenu = new JMenu(getString("fileMenu"));
    fileMenu.add(new JMenuItem(newAction));
    fileMenu.add(new JMenuItem(loadAction));
    fileMenu.add(new JMenuItem(closeAction));
    fileMenu.add(new JMenuItem(saveAction));
    fileMenu.add(new JMenuItem(saveAsAction));
    fileMenu.add(new JMenuItem(updateAlgorithmTreeAction));
    fileMenu.addSeparator();
    fileMenu.add(new ApplicationExitAction(this));
    return fileMenu;
  }

  /**
   * Constri os filtros que a aplicao utiliza.
   */
  private void createFilters() {
    askForParameterValuesFilter =
      new AskForParameterValuesActionFilter(new AskForParameterValuesAction(
        this), workspace);
    basicPopupFilter = new PopupFilter(workspace);
    copyAndPasteFilter = new CopyAndPasteFilter(clipboard, workspace);
    hightLightFilter = new HighlightElementFilter(workspace);
    hintFilter = new HintElementFilter(workspace);
    resizeNodeFilter = new ResizeNodeFilter(workspace);
    selectElementFilter = new SelectElementFilter(workspace);
    updateStatusBarFilter = new UpdateStatusBarFilter(this);
    createPopupFilters();
  }

  /**
   * Contri a interface grfica.
   */
  private void createGui() {
    createActions();
    clipboard.addFlavorListener(pasteAction);
    ApplicationFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(createMenuBar());
    mainFrame.getContentPane().add(createToolBar(), BorderLayout.NORTH);
    mainFrame.getContentPane().add(createMainComponent(), BorderLayout.CENTER);
    mainFrame.getContentPane().add(createStatusBarPanel(), BorderLayout.SOUTH);
    mainFrame.setSize(DesktopFrame.getInstance().getDesktopFrame().getSize());
    mainFrame.showStatusBar();
    updateApplicationTitle();
    createMnemonics(mainFrame);
  }

  /**
   * Cria os atalhos de acesso rpido do desktop.
   *
   * @param mainFrame Janela principal do desktop.
   */
  private void createMnemonics(ApplicationFrame mainFrame) {
    JRootPane rootPane = mainFrame.getRootPane();
    InputMap inputMap = rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
    ActionMap actionMap = rootPane.getActionMap();

    inputMap.put(KeyStroke.getKeyStroke("control F11"), "executeAlg");
    actionMap.put("executeAlg", executeAction);

    inputMap.put(KeyStroke.getKeyStroke("F5"), "updateTree");
    actionMap.put("updateTree", updateAlgorithmTreeAction);

    inputMap.put(KeyStroke.getKeyStroke("control L"), "setFocusAlgTree");
    actionMap.put("setFocusAlgTree", new AlgorithmComponentFocusAction(this));

  }

  /**
   * Cria o componente visual principal da aplicao.
   *
   * @return o componente principal;
   */
  private JComponent createMainComponent() {
    horizontalSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
    horizontalSplitPane.setLeftComponent(algorithmVersionTreePanel);
    horizontalSplitPane.setRightComponent(workspaceScrollPane);
    horizontalSplitPane.setOneTouchExpandable(true);
    horizontalSplitPane.setResizeWeight(0.0);
    horizontalSplitPane.resetToPreferredSizes();
    final JSplitPane verticalSplitPane =
      new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    verticalSplitPane.setTopComponent(horizontalSplitPane);
    verticalSplitPane.setBottomComponent(commandViewerPanel);
    verticalSplitPane.setOneTouchExpandable(true);
    verticalSplitPane.setResizeWeight(1.0);
    verticalSplitPane.setDividerLocation(VERTICAL_SPLITTER_LOCATION);
    return verticalSplitPane;
  }

  /**
   * Cria o menu de ajuda.
   *
   * @return o menu criado.
   */
  private JMenu createHelpMenu() {
    final JMenu helpMenu = new JMenu(getString("helpMenu"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Cria o menu de preferncias do usurio.
   *
   * @return o menu criado.
   */
  private JMenu createPreferencesMenu() {
    ApplicationEditPrefAction preferencesAction =
      new ApplicationEditPrefAction(this, preferences);
    JMenuItem item = new JMenuItem(preferencesAction);
    final JMenu preferencesMenu = new JMenu(getString("preferencesMenu"));
    preferencesMenu.add(item);
    return preferencesMenu;
  }

  /**
   * Cria a barra de menus.
   *
   * @return A barra de menus.
   */
  private JMenuBar createMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(createFileMenu());
    menuBar.add(createEditMenu());
    menuBar.add(createViewMenu());
    menuBar.add(createExecuteMenu());
    menuBar.add(createPreferencesMenu());
    menuBar.add(createHelpMenu());
    return menuBar;
  }

  /**
   * Constri os menus pop-up.
   */
  private void createPopupFilters() {
    popupFilters =
      Arrays
      .<WorkspaceFilter> asList(
        csbase.client.applications.flowapplication.graph.actions.AskForParameterValuesAction
        .createFilter(workspace),
        csbase.client.applications.flowapplication.graph.actions.ShowParameterValuesAction
        .createFilter(workspace),
        CopyParameterValuesAction.createFilter(workspace),
        PasteParameterValuesAction.createFilter(workspace),
        csbase.client.applications.flowapplication.graph.actions.CopySelectedElementsAction
        .createFilter(workspace),
        csbase.client.applications.flowapplication.graph.actions.PasteSelectedElementsAction
        .createFilter(workspace), BreakLinkAction.createFilter(workspace),
        RemoveSelectedElementsAction.createFilter(workspace),
        UpdateVersionAction.createFilter(workspace), BypassAction
        .createFilter(workspace));
  }

  /**
   * Constri o painel da barra de status da aplicao.
   *
   * @return o painel.
   */
  private JPanel createStatusBarPanel() {
    JPanel statusBarPanel = new JPanel(new BorderLayout());
    statusBar = getApplicationFrame().getStatusBar();
    statusBarPanel.add(statusBar, BorderLayout.CENTER);
    return statusBarPanel;
  }

  /**
   * Constri a barra de ferramentas.
   *
   * @return a barra de ferramentas.
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setBorder(BorderFactory.createEtchedBorder());
    toolBar.addSeparator();
    final JButton executeButton = new JButton(executeAction);
    final JButton loadButton = new JButton(loadAction);
    final JButton newButton = new JButton(newAction);
    final JButton removeButton = new JButton(removeAction);
    final JButton saveButton = new JButton(saveAction);
    refreshMenuButton = makeRefreshMenuButton();
    final JButton updateVersionsButton = new JButton(updateVersionsAction);
    final JButton showParameterButton =
      new JButton(askForParameterValuesFilter.getAction());
    final JButton showPropertiesButton = new JButton(showPropertiesAction);
    final MenuButton layoutButton = createLayoutMenuButton();
    final JPanel descriptionPanel = new JPanel();
    descriptionPanel.setLayout(new GridBagLayout());
    final JLabel descriptionLabel = new JLabel(getString("descriptionLabel"));
    descriptionTextField = new JTextField();
    descriptionTextField.setColumns(50);
    GBC gbc = new GBC(1, 1);
    gbc.insets(2);
    descriptionPanel.add(descriptionLabel, gbc);
    descriptionPanel.add(descriptionTextField, gbc.gridx(2).horizontal());

    executeButton.setText("");
    loadButton.setText("");
    newButton.setText("");
    removeButton.setText("");
    saveButton.setText("");
    refreshMenuButton.setText("");
    updateVersionsButton.setText("");
    showParameterButton.setText("");
    showPropertiesButton.setText("");
    toolBar.add(newButton);
    toolBar.add(loadButton);
    toolBar.add(saveButton);
    toolBar.add(refreshMenuButton);
    toolBar.addSeparator();
    toolBar.add(removeButton);
    toolBar.addSeparator();
    toolBar.add(showPropertiesButton);
    toolBar.add(showParameterButton);
    toolBar.add(updateVersionsButton);
    toolBar.addSeparator();
    toolBar.add(executeButton);
    toolBar.addSeparator();
    toolBar.add(descriptionPanel);
    toolBar.addSeparator();
    toolBar.add(layoutButton);
    toolBar.add(zoomControl);
    return toolBar;
  }

  /**
   * Cria o boto-menu com as opes de organizao automtica do fluxo.
   *
   * @return o boto criado.
   */
  private MenuButton createLayoutMenuButton() {
    MenuButton layoutButton =
      new MenuButton(FlowApplicationUI.GENERIC_LAYOUT_ICON,
        PopupPosition.BOTTOM);
    GraphLayoutAction uniformLayoutAction = new UniformSizeLayoutAction(this);
    GraphLayoutAction singleColumnAction =
      new SingleColumnGraphLayoutAction(this);
    GraphLayoutAction linkLayoutAction = new OrthogonalLinkLayoutAction(this);
    layoutButton.add(new JMenuItem(singleColumnAction));
    layoutButton.add(new JMenuItem(uniformLayoutAction));
    layoutButton.add(new JMenuItem(linkLayoutAction));
    return layoutButton;
  }

  /**
   * Cria o menu Exibir.
   *
   * @return O menu Exibir.
   */
  private JMenu createViewMenu() {
    final JMenu viewMenu = new JMenu(getString("viewMenu"));
    final JMenuItem zoomInMenuItem =
      new JMenuItem(zoomControl.getZoomInAction());
    final JMenuItem zoomOutMenuItem =
      new JMenuItem(zoomControl.getZoomOutAction());
    final JMenuItem zoomFitMenuItem =
      new JMenuItem(zoomControl.getZoomFitAction());

    viewMenu.add(new JMenuItem(showPropertiesAction));
    viewMenu.addSeparator();
    viewMenu.add(zoomOutMenuItem);
    viewMenu.add(zoomInMenuItem);
    viewMenu.add(zoomFitMenuItem);
    return viewMenu;
  }

  /**
   * Retorna o tipo padro de arquivo.
   *
   * @return O tipo padro de arquivo.
   */
  private String getDefaultFileType() {
    final List<String> fileTypeVector = getApplicationRegistry().getFileTypes();
    if (fileTypeVector.isEmpty()) {
      return null;
    }
    return fileTypeVector.get(0);
  }

  /**
   * Retorna as informaes de verso do algoritmo selecionado.
   *
   * @return As informaes de verso do algoritmo.
   */
  private AlgorithmVersionInfo getSelectedAlgorithmVersion() {
    final JTree algorithmVersionTree = algorithmVersionTreePanel.getTree();
    final TreePath treePath = algorithmVersionTree.getSelectionPath();
    if (treePath == null) {
      return null;
    }
    final tecgraf.javautils.gui.tree.Node node =
      (tecgraf.javautils.gui.tree.Node) treePath.getLastPathComponent();
    if (node instanceof AlgorithmVersionTreeNode) {
      final AlgorithmVersionTreeNode algorithmVersionTreeNode =
        (AlgorithmVersionTreeNode) node;
      return algorithmVersionTreeNode.getAlgorithmVersion();
    }
    else if (node instanceof AlgorithmTreeNode) {
      final AlgorithmTreeNode algorithmTreeNode = (AlgorithmTreeNode) node;
      final AlgorithmInfo algorithmInfo = algorithmTreeNode.getAlgorithm();
      return algorithmInfo.getLastVersion();
    }
    return null;
  }

  /**
   * Carrega o arquivo de fluxo especificado.
   *
   * @param file O arquivo de fluxo.
   */
  private void load(final ClientProjectFile file) {
    currentFilePath = null;
    final FlowApplicationRemoteTask<Void> task =
      new FlowApplicationRemoteTask<Void>() {
      @Override
      protected void performTask() throws OperationFailureException,
      IOException, ParseException {
        InputStream inputStream = file.getInputStream();
        final FlowAlgorithmParser parser = new FlowAlgorithmParser();
        final Flow flow = parser.read(inputStream);
        final Graph graph = new Graph(getApplicationFrame(), flow);
        setGraph(graph, flow.getDescription());
      }
    };
    final String message =
      getString("loadingFile", new Object[] { file.getStringPath() });
    if (task.execute(getApplicationFrame(), message, message)) {
      currentFilePath = new ProjectTreePath(file);
      updateApplicationTitle();
    }
  }

  /**
   * Salva o fluxo corrente em um arquivo.
   *
   * @param file O arquivo.
   *
   * @return <code>true</code> sucesso ou <code>false</code> em caso de erro.
   */
  private boolean save(final ClientProjectFile file) {
    final FlowApplicationRemoteTask<Void> task =
      new FlowApplicationRemoteTask<Void>() {

      @Override
      protected void handleError(final Exception error) {
        if (error instanceof FileLockedException) {
          StandardDialogs
          .showErrorDialog(getApplicationFrame(), getName(), String.format(
            getString("msg.error.file.locked"), file.getName()));
        }
        else {
          super.handleError(error);
        }
      }

      @Override
      protected void performTask() throws Exception {
        final Flow flow = getWorkspace().getGraph().toFlow();
        flow.setName(file.getName());
        flow.setDescription(descriptionTextField.getText());
        OutputStream outputStream = null;
        try {
          outputStream = file.getOutputStream();
          final FlowAlgorithmParser parser = new FlowAlgorithmParser();
          parser.write(outputStream, flow);
        }
        finally {
          if (outputStream != null) {
            try {
              outputStream.close();
            }
            catch (final IOException e) {
              e.printStackTrace();
            }
          }
        }
      }
    };
    final String message =
      getString("savingFile", new Object[] { FileUtils.joinPath('/', file
        .getPath()) });
    if (task.execute(getApplicationFrame(), message, message)) {
      wasChanged = false;
      return true;
    }
    return false;
  }

  /**
   * Atualiza o ttulo da aplicao construtora de fluxo.
   */
  private void updateApplicationTitle() {
    String titleDetails;
    if (currentFilePath == null) {
      titleDetails = getString("without_file");
    }
    else {
      final ClientProjectFile currentFile = currentFilePath.getFile();
      final String currentFilePathAsString = currentFile.getStringPath();
      titleDetails = currentFilePathAsString;
    }
    getApplicationFrame().setTitle(getName() + " - " + titleDetails);
  }

  /**
   * Esconde o painel com a rvore de algoritmos.
   */
  public void hideAlgorithmTreePanel() {
    horizontalSplitPane.setDividerLocation(0.0);
  }

  /**
   * Exibe o painel com a rvore de algoritmos.
   */
  public void showAlgorithmTreePanel() {
    horizontalSplitPane.setDividerLocation(horizontalSplitPane
      .getMinimumDividerLocation());
  }

  /**
   * Gerenciador de eventos de teclado da aplicao. Permite o redirecionamento
   * dos eventos de teclado recebidos no espao de trabalho e na rvore de
   * algoritmos, quando necessrio.
   */
  private final class FlowApplicationKeyDispatcher implements
  KeyEventDispatcher {

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean dispatchKeyEvent(KeyEvent event) {
      Object source = event.getSource();
      if (horizontalSplitPane.isAncestorOf((Component) source)) {
        if (!event.isAltDown() && !event.isControlDown()) {
          int keyCode = event.getKeyCode();
          switch (keyCode) {
            case KeyEvent.VK_ENTER:
              /*
               * Teste para evitar que a ao seja feita mais de uma vez em
               * eventos similar (ex.: key_pressed e key_typed).
               */
              if (event.getID() == KeyEvent.KEY_PRESSED) {
                AlgorithmVersionInfo version = getSelectedAlgorithmVersion();
                if (version != null) {
                  workspace.addNewNode(version);
                }
                return true;
              }
              //$FALL-THROUGH$
            case KeyEvent.VK_UP:
            case KeyEvent.VK_DOWN:
              /*
               * Redireciona o evento das setas para a rvore.
               */
              JTextField field = algorithmVersionTreePanel.getTextField();
              field.selectAll();
              JTree tree = algorithmVersionTreePanel.getTree();
              keyFocusManager.redispatchEvent(tree, event);
              return true;
            default:
              if (Character.isLetterOrDigit(event.getKeyChar())) {
                /*
                 * Redireciona o evento de letras e nmeros para a caixa de
                 * texto que filtra a rvore.
                 */
                JTextField textField = algorithmVersionTreePanel.getTextField();
                textField.requestFocusInWindow();
                keyFocusManager.redispatchEvent(textField, event);
                return true;
              }
          }
        }
      }
      return false;
    }
  }

  /**
   * Listener que percebe as modificao no grafo para que se possa perguntar ao
   * usurio se deseja salvar as alteraes feitas.
   */
  private final class ChangedListener implements GraphListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChangedWorkspace(final Graph graph) {
      // ignora este tipo de evento
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementCreated(final Graph graph, final GraphElement element) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDragged(final Graph graph,
      final GraphElement element, final double tx, final double ty) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDragged(final Graph graph,
      final GraphElement element, final Point2D startPoint,
      final Point2D endPoint) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementDropped(final Graph graph,
      final GraphElement element, final Point2D point) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementParametrized(final Graph graph,
      final GraphElement element) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementRemoved(final Graph graph, final GraphElement element) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasElementSelected(final Graph graph, final GraphElement element) {
      if (element != null) {
        Rectangle2D rectangle = element.getBounds2D();
        workspace.scrollRectToVisible(rectangle.getBounds());
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkAnchored(final Graph graph, final GraphLink link,
      final GraphFileDescriptor fileDescriptor) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkIncreased(final Graph graph, final GraphLink link) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasLinkStatusChanged(final Graph graph, final GraphLink link) {
      wasChanged = true;
    }

    /**
     *
     * @see GraphListener#wasLinkUnanchored(Graph, GraphLink,
     *      GraphFileDescriptor)
     */
    @Override
    public void wasLinkUnanchored(final Graph graph, final GraphLink link,
      final GraphFileDescriptor fileDescriptor) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasNodeResized(final Graph graph, final GraphNode node) {
      wasChanged = true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetEnabled(final Graph graph, final GraphNode node,
      final String parameterName, final boolean isEnabled) {
      // ignora este tipo de evento
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasParameterSetVisible(final Graph graph, final GraphNode node,
      final String parameterName, final boolean isVisible) {
      // ignora este tipo de evento
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void wasReseted(final Graph graph) {
      // ignora este tipo de evento
    }
  }

  /**
   * Cira um boto com menu popup associado, com itens para atualizao
   * (refresh) da rvore de algoritmos e categorias e detalhamento das mudanas
   * na barra de status.
   *
   * @return o boto com menu popup
   */
  private MenuButton makeRefreshMenuButton() {
    if (refreshMenuButton == null) {
      refreshMenuButton =
        new MenuButton(ApplicationImages.ICON_REFRESH_THIN_16,
          PopupPosition.BOTTOM, true);
      createRefreshButtonItems();
      String tooltip = getString("button.refresh.tooltip");
      refreshMenuButton.setToolTipText(tooltip);
      algTreeRefreshDetail.setEnabled(false);
    }
    return refreshMenuButton;
  }

  /**
   * Cria os itens de menu do boto popup.
   */
  private void createRefreshButtonItems() {
    if (refreshMenuButton == null) {
      return;
    }

    //Item de menu para atualizar a rvore de algoritmos e categorias
    JMenuItem updateAlgoItem =
      new JMenuItem(getString("popup.menu.item.refresh"));
    updateAlgoItem.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        reloadAlgorithmTree();
      }
    });
    algTreeRefreshDetail = new JMenuItem(getString("popup.menu.item.detail"));
    algTreeRefreshDetail.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        String algoChanges =
          getString("msg.update.algorithms", new Object[] { modifiedAlgorithms
            .size() });
        String categoriesChanges =
          getString("msg.update.categories", new Object[] { modifiedCategories
            .size() });

        String changes = algoChanges + " / " + categoriesChanges;
        if (algorithmRepositoryReloaded) {
          String repositoryReloaded =
            getString("msg.update.repository.reloaded");
          changes += " / " + repositoryReloaded;
        }
        setStatusBarMessage(changes);
      }
    });

    refreshMenuButton.add(updateAlgoItem);
    refreshMenuButton.addSeparator();
    refreshMenuButton.add(algTreeRefreshDetail);
  }

  /**
   * Obtm o nome do fluxo corrente.
   *
   * @return o nome do fluxo.
   */
  public String getFlowName() {
    String flowName;
    if (currentFilePath == null) {
      flowName = getString("anonymous");
    }
    else {
      final ClientProjectFile currentFile = currentFilePath.getFile();
      final String currentFilePathAsString = currentFile.getStringPath();
      if (wasChanged) {
        flowName =
          getString("flowModified", new Object[] { currentFilePathAsString });
      }
      else {
        flowName = currentFilePathAsString;
      }
    }
    return flowName;
  }

  /**
   * Obtm a descrio do fluxo corrente.
   *
   * @return a descrio do fluxo.
   */
  public String getFlowDescription() {
    return descriptionTextField.getText();
  }

}
