package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.Client;
import csbase.client.ClientLocalFile;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.project.ClientProjectFileType;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserSave;
import csbase.client.project.ProjectTreePath;
import csbase.client.project.tasks.CreateFileTask;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.project.FileLockedException;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.UserOutline;

/**
 * Dilogo de exportao de usurios.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ExportUsersDialog {
  /**
   * Caracter utilizado quando o usurio tem mais de um e-mail, para separa-los.
   */
  private static final String EMAILS_SEPARATOR = ";";

  /**
   * Janela owner
   */
  private Window owner;

  /**
   * Label
   */
  private String windowLabel;

  /**
   * Janela
   */
  private DesktopComponentDialog window;

  /**
   * Contedo
   */
  private Container contentPane;

  /**
   * Boto
   */
  private JButton prjActionButton;

  /**
   * Boto
   */
  private JButton localActionButton;

  /**
   * Tabela
   */
  private SortableTable sortableTable;

  /**
   * Tipo de arquivo
   */
  private static final String FILE_TYPE = "CSV";

  /**
   * ndice de coluna
   */
  private static final int LOGIN_COLUMN = 0;

  /**
   * ndice de coluna
   */
  private static final int NAME_COLUMN = 1;

  /**
   * ndice de coluna
   */
  private static final int E_MAIL_COLUMN = 2;

  /**
   * ndice de coluna
   */
  private static final int USERGROUP_COLUMN = 3;

  /**
   * Separador de campo
   */
  private static final char FIELD_SEPARATOR = '\t';

  /**
   * Separador de registro.
   */
  private static final char REGISTER_SEPARATOR = '\n';

  /**
   * Construtor.
   * 
   * @param owner janela me.
   */
  public ExportUsersDialog(Window owner) {
    this.owner = owner;
    this.windowLabel = LNG.get("IAS_EXPORT_USERS_TITLE");
    this.window = new DesktopComponentDialog(owner, this.windowLabel);
    this.window.center(owner);
    this.contentPane = this.window.getContentPane();
  }

  /**
   * Exibio de dilogo.
   */
  public void showDialog() {
    createUsersPanel();
    createButtonPanel();
    this.window.pack();
    this.window.setLocationRelativeTo(owner);
    updateActionButtons();
    this.window.setVisible(true);
  }

  /**
   * Criao de painel de usurios.
   */
  private void createUsersPanel() {
    JPanel usersPanel = new JPanel();
    usersPanel.setLayout(new BorderLayout());
    createTable();
    usersPanel.add(new JScrollPane(this.sortableTable), BorderLayout.CENTER);
    this.contentPane.add(usersPanel, BorderLayout.CENTER);
  }

  /**
   * Criao da tabela
   */
  private void createTable() {
    Vector<UserOutline> allOutlines =
      UserProxy.getAllOutlines(this.owner, this.windowLabel, "MESSAGE");
    ObjectTableProvider provider = new DefaultObjectTableProvider() {

      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        UserOutline line = (UserOutline) item;
        if (line == null) {
          return null;
        }

        String[] emails = line.getEmails();
        StringBuilder sb = new StringBuilder();
        if (emails != null && emails.length > 0) {
          final String firstEmail = line.getEmails()[0];
          sb.append(firstEmail);
          for (int inx = 1; inx < emails.length; inx++) {
            sb.append(EMAILS_SEPARATOR).append(emails[inx]);
          }
        }
        else {
          sb.append("---");
        }
        return new String[] { line.getLogin(), line.getName(), sb.toString(),
            line.getUserGroup() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_USER"), LNG.get("IAS_USER_NAME"),
            LNG.get("IAS_USER_EMAIL"), LNG.get("IAS_USERGROUP") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class[] { String.class, String.class, String.class,
            String.class };
      }
    };
    ObjectTableModel<UserOutline> model =
      new ObjectTableModel<UserOutline>(allOutlines, provider);
    this.sortableTable = new SortableTable(model);
    this.sortableTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        updateActionButtons();
      }
    });
  }

  /**
   * Atualiza estado dos botes
   */
  private void updateActionButtons() {
    if (sortableTable.getSelectedRowCount() > 0) {
      prjActionButton.setEnabled(true);
      localActionButton.setEnabled(true);
    }
    else {
      prjActionButton.setEnabled(false);
      localActionButton.setEnabled(false);
    }

    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame == null || desktopFrame.getProject() == null) {
      prjActionButton.setEnabled(false);
    }
    else {
      prjActionButton.setEnabled(true);
    }
  }

  /**
   * Criao de painel de botes.
   */
  private void createButtonPanel() {
    createProjectActionButton();
    createLocalActionButton();
    JButton closeButton = createCloseButton();
    ClientUtilities.adjustEqualSizes(localActionButton, prjActionButton,
      closeButton);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(localActionButton);
    buttonsPanel.add(prjActionButton);
    buttonsPanel.add(closeButton);
    this.contentPane.add(buttonsPanel, BorderLayout.SOUTH);
  }

  /**
   * Criao da ao.
   */
  private void createProjectActionButton() {
    prjActionButton =
      new JButton(LNG.get("IAS_EXPORT_USERS_PRJ_ACTION_BUTTON"));
    prjActionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        showProjectFileChooser();
      }
    });

  }

  /**
   * Criao da ao.
   */
  private void createLocalActionButton() {
    localActionButton =
      new JButton(LNG.get("IAS_EXPORT_USERS_LOCAL_ACTION_BUTTON"));
    localActionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        showFileChooser();
      }
    });
  }

  /**
   * Exibe o sistema de arquivo local para escolha do diretrio onde o arquivo
   * ser gravado.
   */
  private void showFileChooser() {
    JFileChooser fc = new JFileChooser();
    fc.setAcceptAllFileFilterUsed(true);
    fc.addChoosableFileFilter(new Filter("txt", LNG
      .get("ias.file_type.txt.description")));
    fc.addChoosableFileFilter(new Filter("csv", LNG
      .get("ias.file_type.csv.description")));
    int returnVal = fc.showSaveDialog(window);
    if (returnVal == JFileChooser.APPROVE_OPTION) {
      File file = fc.getSelectedFile();
      ClientLocalFile clientLocalFile = new ClientLocalFile(file);
      StringBuffer buffer = generateData();
      if (writeFile(clientLocalFile, buffer)) {
        StandardDialogs.showInfoDialog(window, windowLabel,
          LNG.get("IAS_EXPORT_USERS_SUCCESS"));
        window.close();
      }
    }
  }

  /**
   * Grava o arquivo no disco local do usurio.
   * 
   * @param file o arquivo que ser criado.
   * @param buffer o contedo que ser gravado no arquivo.
   * 
   * @return true, se a gravao finalizar com sucesso, false caso contrrio.
   */
  private boolean writeFile(final ClientFile file, final StringBuffer buffer) {
    LocalTask<Void> localTask = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        final OutputStream stream = file.getOutputStream();
        final Client client = Client.getInstance();
        final Charset charset = client.getSystemDefaultCharset();
        final OutputStreamWriter writer =
          new OutputStreamWriter(stream, charset);
        try {
          writer.write(buffer.toString());
        }
        finally {
          writer.close();
        }
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof IOException) {
          StandardDialogs.showErrorDialog(
            window,
            windowLabel,
            String.format(LNG.get("ExportUsersDialog.save.file.failed"),
              file.getName()));
        }
        else {
          super.handleError(error);
        }
      }

    };
    return localTask.execute(
      window,
      windowLabel,
      String.format(LNG.get("ExportUsersDialog.msg.writing.file"),
        file.getName()));
  }

  /**
   * Exibe a rvore de projeto para escolha do diretrio onde o arquivo ser
   * gravado.
   */
  private void showProjectFileChooser() {
    ClientProjectFile clientProjectFile = getChosenFile();
    if (clientProjectFile == null) {
      return;
    }
    StringBuffer buffer = generateData();
    if (writeFile(clientProjectFile, buffer)) {
      StandardDialogs.showInfoDialog(window, windowLabel,
        LNG.get("IAS_EXPORT_USERS_SUCCESS"));
      window.close();
    }
  }

  /**
   * Grava o arquivo no servidor, na rea de projeto.
   * 
   * @param clientProjectFile o arquivo que ser criado.
   * @param buffer o contedo que ser gravado no arquivo.
   * 
   * @return true, se a gravao finalizar com sucesso, false caso contrrio.
   */
  private boolean writeFile(final ClientProjectFile clientProjectFile,
    final StringBuffer buffer) {
    RemoteTask<Void> task = new RemoteTask<Void>() {

      @Override
      protected void performTask() throws Exception {
        OutputStreamWriter writer = null;
        try {
          OutputStream stream = clientProjectFile.getOutputStream();
          writer = new OutputStreamWriter(stream);
          writer.write(buffer.toString());
          writer.flush();
          writer.close();
        }
        finally {
          if (writer != null) {
            writer.close();
          }
        }
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof FileLockedException) {
          StandardDialogs.showErrorDialog(window, windowLabel, String.format(
            LNG.get("csbase.file.locked.error"), clientProjectFile.getName()));
        }
        else {
          super.handleError(error);
        }
      }
    };
    return task.execute(window, windowLabel, String.format(
      LNG.get("ExportUsersDialog.msg.writing.file"),
      clientProjectFile.getName()));

  }

  /**
   * Escolha de arquivo.
   * 
   * @return o arquivo.
   */
  private ClientProjectFile getChosenFile() {
    final CommonClientProject currentProject =
      DesktopFrame.getInstance().getProject();
    ProjectFileChooserSave chooser = null;
    // Se no existe arquivo CSV no sistema, usa o TEXT.
    String fileTypeExt = FILE_TYPE;
    ClientProjectFileType projFileType =
      ClientProjectFileType.getFileType(FILE_TYPE);
    if (projFileType.getCode().equals(ProjectFileType.UNKNOWN)) {
      fileTypeExt = "TEXT";
    }
    try {
      chooser =
        new ProjectFileChooserSave(window, currentProject,
          ProjectFileChooser.FILE_ONLY, fileTypeExt);
    }
    catch (ClientException ce) {
      StandardErrorDialogs.showErrorDialog(window, windowLabel,
        LNG.get("ExportUsersDialog.error.creating.chooser"), ce);
      return null;
    }
    final ProjectTreePath treePath = chooser.getSelectedPath();
    if (treePath == null) {
      return null;
    }
    if (treePath.exists()) {
      return treePath.getFile();
    }
    String[] path = treePath.getPath();
    final String name = path[path.length - 1];

    // criamos o arquivo no projeto
    CreateFileTask task =
      new CreateFileTask(currentProject, treePath.getParent(), name,
        chooser.getFileType());
    final boolean succeeded =
      task.execute(window, windowLabel,
        String.format(LNG.get("ExportUsersDialog.msg.creating.file"), name));
    if (succeeded) {
      return task.getResult();
    }
    return null;
  }

  /**
   * Gerao de dados
   * 
   * @return texto completo.
   */
  private StringBuffer generateData() {
    StringBuffer buffer = new StringBuffer();
    int[] selectedRows = sortableTable.getSelectedRows();
    for (int i = 0; i < selectedRows.length; i++) {
      String login =
        (String) sortableTable.getValueAt(selectedRows[i], LOGIN_COLUMN);
      String name =
        (String) sortableTable.getValueAt(selectedRows[i], NAME_COLUMN);
      String eMail =
        (String) sortableTable.getValueAt(selectedRows[i], E_MAIL_COLUMN);
      String userGroup =
        (String) sortableTable.getValueAt(selectedRows[i], USERGROUP_COLUMN);
      buffer.append(login);
      buffer.append(FIELD_SEPARATOR);
      buffer.append(name);
      buffer.append(FIELD_SEPARATOR);
      buffer.append(eMail);
      buffer.append(FIELD_SEPARATOR);
      buffer.append(userGroup);
      buffer.append(REGISTER_SEPARATOR);
    }
    return buffer;
  }

  /**
   * Criao de boto de close.
   * 
   * @return boto
   */
  private JButton createCloseButton() {
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.setEnabled(true);
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        window.close();
      }
    });
    return closeButton;
  }
}
